# Copyright 2008 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import random
from skool import Location

class CommandListTemplate:
    def __init__(self, num=-1):
        self.num = num
        self.commands = []

    def addCommand(self, commandClass, *params):
        self.commands.append((commandClass, params))

    def getCommands(self):
        return [cmdClass(*params) for cmdClass, params in self.commands]

class APTable:
    def __init__(self, template):
        self.template = template
        self.stack = []
        self.restart()

    def command(self, character, controllingCommand):
        # If the controlling command is not already in control of the character,
        # give it a chance
        if controllingCommand and controllingCommand not in self.stack:
            self.stack.append(controllingCommand)
        while True:
            if self.stack:
                command = self.stack[-1]
                subcommand = command.command(character)
                if subcommand is None:
                    break
                elif subcommand is not command:
                    self.stack.append(subcommand)
                else:
                    self.stack.pop()
            else:
                # Remove any controlling command before moving to the next
                # command
                character.setControllingCommand(None)
                self.stack.append(self.commands[self.index])
                self.index += 1

    def restart(self):
        self.index = 0
        self.commands = self.template.getCommands()

class signals:
    ASSEMBLY_FINISHED = 'AssemblyFinished'
    TIME_FOR_ASSEMBLY = 'TimeForAssembly'
    FINISHED_QUESTION = 'FinishedQuestion'
    FINISHED_ANSWER = 'FinishedAnswer'

class ComplexCommand:
    def __init__(self, commands):
        self.index = 0
        self.commands = commands

    def command(self, character):
        val = self.commands[self.index](character)
        if val is not False:
            self.index = (self.index + 1) % len(self.commands)
        return val or None

    def restart(self):
        self.index = -1

    def done(self, character):
        return self

class SitForAssembly(ComplexCommand):
    def __init__(self):
        ComplexCommand.__init__(self, (self.findSpotToSit, self.sitDown, self.getUp, self.done))

    def findSpotToSit(self, character):
        if character.gotSignal(signals.ASSEMBLY_FINISHED):
            return self
        return GoToXY(character.x - random.randrange(1, 5), character.y)

    def sitDown(self, character):
        if character.gotSignal(signals.ASSEMBLY_FINISHED):
            return self
        if character.direction < 0:
            character.turn()
            return False
        character.sit()

    def getUp(self, character):
        if character.gotSignal(signals.ASSEMBLY_FINISHED):
            character.getUp()
        else:
            return False

class MoveDoor(ComplexCommand):
    def __init__(self, doorId, shut):
        self.doorId = doorId
        self.shut = shut
        ComplexCommand.__init__(self, (self.raiseArm, self.moveDoor, self.lowerArm, self.done))

    def raiseArm(self, character):
        character.raiseArm()

    def moveDoor(self, character):
        character.moveDoor(self.doorId, self.shut)

    def lowerArm(self, character):
        character.lowerArm()

class OpenDoor(MoveDoor):
    def __init__(self, doorId):
        MoveDoor.__init__(self, doorId, False)

class ShutDoor(MoveDoor):
    def __init__(self, doorId):
        MoveDoor.__init__(self, doorId, True)

class WipeBoard(ComplexCommand):
    def __init__(self):
        self.column = 7
        ComplexCommand.__init__(self, (self.walk, self.wipe, self.lowerArm, self.walkBack, self.done))

    def walk(self, character):
        character.setWipingBoard(True)
        return GoToXY(character.x - 1, character.y)

    def wipe(self, character):
        character.raiseArm()
        character.wipeBoard(self.column)

    def lowerArm(self, character):
        self.column -= 1
        character.lowerArm()
        if self.column >= 0:
            # Repeat until the board is clean
            self.restart()

    def walkBack(self, character):
        character.setWipingBoard(False)
        return GoToXY(character.x + 3, character.y)

class ConductClassWithEric(ComplexCommand):
    def __init__(self):
        ComplexCommand.__init__(self, (self.ask, self.signalAsked, self.waitForAnswer))

    def ask(self, character):
        return Say(character.prepareQA())

    def signalAsked(self, character):
        character.signal(signals.FINISHED_QUESTION)

    def waitForAnswer(self, character):
        if character.gotSignal(signals.FINISHED_ANSWER):
            character.unsignal(signals.FINISHED_ANSWER)
        else:
            return False

class GoTo:
    def __init__(self, locationId):
        self.locationId = locationId
        self.destination = None

    def command(self, character):
        if self.destination is None:
            self.destination = character.resolveLocationId(self.locationId)
        if (character.x, character.y) == (self.destination.x, self.destination.y):
            return self
        if character.onStairs():
            character.walk(True)
            return
        if character.isSitting():
            character.getUp()
            return
        nextStaircase = character.getNextStaircase(self.destination)
        if nextStaircase:
            if character.y == nextStaircase.bottom.y:
                nextX = nextStaircase.bottom.x
            else:
                nextX = nextStaircase.top.x
        else:
            nextX = self.destination.x
        if character.x < nextX:
            return character.right()
        elif character.x > nextX:
            return character.left()
        else:
            if nextStaircase.contains(character):
                character.walk(True)
            else:
                character.turn()

class GoToRandomLocation(GoTo):
    def __init__(self):
        self.destination = None

    def command(self, character):
        if self.destination is None:
            self.destination = Location(character.getRandomDestination())
        return GoTo.command(self, character)

class GoToXY(GoTo):
    def __init__(self, x, y):
        self.destination = Location((x, y))

class Restart:
    def command(self, character):
        character.restartTable()
        return self

class FindSeat:
    def __init__(self, moveAlong=False):
        self.moveAlong = moveAlong
        self.delay = 5

    def command(self, character):
        if character.isSittingOnFloor():
            self.delay -= 1
            if self.delay < 0:
                character.getUp()
            return
        nextChair = character.getNextChair(self.moveAlong)
        self.moveAlong = False
        if nextChair == character.chair():
            character.sit()
            return self
        if character.x == nextChair.x and character.direction > 0:
            character.turn()
            return
        return GoToXY(nextChair.x, nextChair.y)

class MoveAboutUntil:
    def __init__(self, signal):
        self.signal = signal

    def command(self, character):
        if character.gotSignal(self.signal):
            return self
        return WalkAround(1)

class WalkAround:
    def __init__(self, walkabouts):
        self.count = walkabouts
        self.origin = None

    def command(self, character):
        if self.count == 0:
            return self
        # Cannot move about an x-coordinate on a staircase, so get off the
        # stairs first
        if character.onStairs():
            character.walk(True)
            return
        if self.origin is None:
            self.origin = character.x
        if character.x == self.origin:
            return GoToXY(character.x - random.randrange(0, 8), character.y)
        self.count -= 1
        return GoToXY(self.origin, character.y)

class SitStill:
    def command(self, character):
        if not character.isSittingOnChair():
            return FindSeat(True)

class GrassAndAnswerQuestions:
    def __init__(self):
        self.waiting = True

    def command(self, character):
        if not character.isSittingOnChair():
            return FindSeat()
        if self.waiting:
            if character.gotSignal(signals.FINISHED_QUESTION):
                self.waiting = False
                character.unsignal(signals.FINISHED_QUESTION)
                return Say(character.getAnswer())
        else:
            character.signal(signals.FINISHED_ANSWER)
            self.waiting = True

class StartAssemblyIfReady:
    def command(self, character):
        if character.gotSignal(signals.ASSEMBLY_FINISHED):
            return self
        elif character.gotSignal(signals.TIME_FOR_ASSEMBLY):
            return
        character.restartTable()
        return self

class AwaitAssemblyTime:
    def command(self, character):
        if character.isTimeToStartLesson():
            character.signal(signals.TIME_FOR_ASSEMBLY)
        else:
            character.restartTable()
        return self

class ConductAssembly:
    def __init__(self):
        self.spoken = False

    def command(self, character):
        if self.spoken:
            character.signal(signals.ASSEMBLY_FINISHED)
            return self
        self.spoken = True
        return Say(character.getAssemblyMessage())

class StartLessonIfReady:
    def __init__(self, signal):
        self.signal = signal
        self.ready = False

    def command(self, character):
        if not self.ready:
            if character.isTimeToStartLesson():
                self.ready = True
                character.signal(self.signal)
                return TellKidsToSitDown()
            else:
                character.restartTable()
        return self

class TellKidsToSitDown:
    def __init__(self):
        self.spoken = False

    def command(self, character):
        if self.spoken:
            return self
        self.spoken = True
        return Say(character.sitDownMessage)

class ConductClass:
    def __init__(self):
        self.wipedBoard = False
        self.wroteOnBoard = False

    def command(self, character):
        if character.room.blackboard:
            if not self.wipedBoard:
                self.wipedBoard = True
                return WipeBoard()
            if not self.wroteOnBoard:
                self.wroteOnBoard = True
                shouldWrite = 184 if character.isTeachingEric() else 160
                if random.randrange(0, 256) >= shouldWrite:
                    return WriteOnBoard(character.getBlackboardMessage())
        if character.isTeachingEric():
            return ConductClassWithEric()
        return ConductClassWithoutEric()

class WriteOnBoard:
    def __init__(self, message):
        self.message = message
        self.index = 1
        self.armDown = True

    def command(self, character):
        if character.room.blackboard:
            if character.writeOnBoard(self.message, self.index):
                character.lowerArm()
                return self
            if self.armDown:
                character.raiseArm()
            else:
                character.lowerArm()
            self.armDown = not self.armDown
        self.index += 1

class ConductClassWithoutEric:
    def __init__(self):
        self.spoken = False

    def command(self, character):
        if self.spoken:
            return GoToXY(character.x - 3 * character.direction, character.y)
        self.spoken = True
        return Say('Write an essay with this title')

class Say:
    def __init__(self, words):
        self.words = words
        self.shift = -4

    def command(self, character):
        if character.say(self.words, self.shift):
            character.removeBubble()
            return self
        self.shift += 1

class Signal:
    def __init__(self, signal):
        self.signal = signal

    def command(self, character):
        character.signal(self.signal)
        return self

class Unsignal:
    def __init__(self, signal):
        self.signal = signal

    def command(self, character):
        character.unsignal(self.signal)
        return self

class SetControllingCommand:
    def __init__(self, commandName):
        self.commandName = commandName

    def command(self, character):
        commandClass = eval(self.commandName)
        character.setControllingCommand(commandClass())
        return self

class HitOrFireNowAndThen:
    def ready(self, character):
        # TODO Decide properly when to hit or fire catapult; character must be:
        #   - not on stairs
        #   - not within sight of a teacher
        # and then only hit/fire half the time
        return random.randrange(100) > 80

    def command(self, character):
        if not self.ready(character):
            return self
        return self.getCommand()

class FireNowAndThen(HitOrFireNowAndThen):
    def getCommand(self):
        return FireCatapult()

class FireCatapult(ComplexCommand):
    def __init__(self):
        ComplexCommand.__init__(self, (self.aim, self.fire, self.lower, self.done))

    def aim(self, character):
        if character.animatoryState == 10:
            character.animatoryState = 11
        else:
            character.animatoryState = 10
            return False

    def fire(self, character):
        # TODO Let loose a catapult pellet
        return

    def lower(self, character):
        if character.animatoryState == 10:
            character.animatoryState = 0
        else:
            character.animatoryState = 10
            return False

class HitNowAndThen(HitOrFireNowAndThen):
    def ready(self, character):
        return HitOrFireNowAndThen.ready(self, character) and self.inFrontOfVictim(character)

    def inFrontOfVictim(self, character):
        # TODO Return true only if character is in front of a victim
        return True

    def getCommand(self):
        return Hit()

class Hit(ComplexCommand):
    def __init__(self):
        ComplexCommand.__init__(self, (self.aim, self.hit, self.lower, self.done))

    def aim(self, character):
        if character.animatoryState == 8:
            character.animatoryState = 9
        else:
            character.animatoryState = 8
            return False

    def hit(self, character):
        # TODO Hit anyone nearby
        return

    def lower(self, character):
        if character.animatoryState == 8:
            character.animatoryState = 0
        else:
            character.animatoryState = 8
            return False

#
# Unimplemented commands
#
class UnimplementedCommand:
    def __init__(self, *params):
        self.terminal = False

    def command(self, character):
        return None if self.terminal else self

class WriteOnBoardUnless(UnimplementedCommand):
    # Parameter will be a signal or 'Dirty'
    pass

class FindEric(UnimplementedCommand):
    pass

class FindEricIfMissing(UnimplementedCommand):
    pass

class GiveEric2000Lines(UnimplementedCommand):
    pass

class TellEricToGoHome(UnimplementedCommand):
    pass

class TellEricAboutSwot(UnimplementedCommand):
    pass

class TellEricAboutBully(UnimplementedCommand):
    pass

class TellEricAboutTearaway(UnimplementedCommand):
    pass

class SetRestartPoint(UnimplementedCommand):
    pass

class AwaitGateClosure(UnimplementedCommand):
    pass

class JumpIfShut(UnimplementedCommand):
    pass

class WaitForKids(UnimplementedCommand):
    pass

class FindAndHitHayley(UnimplementedCommand):
    pass

class TripPeopleUp(UnimplementedCommand):
    pass

class FollowLittleBoy(UnimplementedCommand):
    pass

class WalkFast(UnimplementedCommand):
    pass

class CheckIfTouchingEric(UnimplementedCommand):
    pass

class WatchForEric(UnimplementedCommand):
    pass
