# Copyright 2008 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import random
import pygame
from pygame.locals import *
from skool import Location
from ai import OpenDoor, FindSeat, CommandListTemplate, APTable

class Character:
    def __init__(self, characterId, name, adult=False):
        self.characterId = characterId
        self.name = name
        self.x = 0
        self.y = 0
        self.adult = adult
        self.direction = 1 # 1 = facing right, -1 = facing left
        self.verticalDirection = 0 # 1 = down, 0 = neither up nor down
        self.speed = 1 # 1 = walk, 2 = run
        self.speedChangeDelay = 1
        self.walkDelay = random.randrange(3, 6)
        self.speechDelay = 1
        self.actionDelay = 1
        self.staircase = None
        self.barrier = None
        self.room = None
        self.commandLists = {}
        self.sitDownMessage = None
        self.bubble = None
        self.assemblyMessageGenerator = None
        self.blackboardMessages = []
        self.qaGenerator = None
        self.wipingBoard = False
        self.controllingCommand = None
        self.controllingCommandTimer = 0

    def setGame(self, game):
        self.game = game

    def setSkool(self, skool):
        self.skool = skool

    def setTimetable(self, timetable):
        self.timetable = timetable

    def setSitDownMessage(self, message):
        self.sitDownMessage = message

    def setAnimatoryStates(self, asDictL, asDictR):
        self.animatoryState = 0
        self.asDictL = asDictL
        self.asDictR = asDictR

    def setControllingCommand(self, command):
        if command:
            self.controllingCommand = command
            self.controllingCommandTimer = 1
        else:
            self.controllingCommandTimer -= 1
            if self.controllingCommandTimer < 0:
                # We're on the second command after a SetControllingCommand
                # command, so remove the controlling command
                self.controllingCommand = None

    def getImage(self):
        asDict = self.asDictR if self.direction > 0 else self.asDictL
        return asDict[self.animatoryState]

    def draw(self, surface, column):
        surface.blit(self.getImage(), self.game.scaleCoords((self.x - column, self.y)))
        if self.bubble:
            bubblePos = (8 * ((self.x + 1) / 8) - column, self.y - 3)
            surface.blit(self.bubble, self.game.scaleCoords(bubblePos))

    def setRandomLocations(self, locations):
        self.randomLocations = locations

    def getRandomDestination(self):
        return self.randomLocations[random.randrange(len(self.randomLocations))]

    def addCommandList(self, lessonId, tapId):
        self.commandLists[lessonId] = tapId

    def getTapId(self, lessonId):
        return self.commandLists[lessonId]

    def setAPTable(self, apTable):
        self.apTable = apTable

    def restartTable(self):
        self.apTable.restart()

    def isTimeToMove(self):
        if self.bubble:
            self.speechDelay = (self.speechDelay + 1) % 3
            return self.speechDelay == 0
        if self.animatoryState > 7:
            # Perform hitting, firing etc. quickly
            self.actionDelay = (self.actionDelay + 1) % 2
            return self.actionDelay == 0
        self.walkDelay -= 1
        if self.walkDelay > 0:
            return False
        self.speedChangeDelay -= 1
        if self.speedChangeDelay == 0:
            self.speedChangeDelay = random.randrange(16, 32)
            if not self.adult:
                self.run(self.speedChangeDelay & 1)
        self.walkDelay = 7 - 2 * self.speed
        return True

    def move(self, moveNow):
        if not (self.isTimeToMove() or moveNow):
            return
        if self.midstride():
            return self.walk()
        self.staircase = self.skool.staircase(self)
        self.barrier = self.skool.barrier(self)
        self.room = self.skool.room(self)
        self.apTable.command(self, self.controllingCommand)

    def left(self):
        if self.direction > 0:
            self.turn()
        else:
            if self.barrier and self.barrier.x <= self.x:
                if self.adult and self.barrier.isOpenable():
                    return self.openDoor()
                return
            onStairs = False
            if self.staircase:
                if self.staircase.direction < 0:
                    onStairs = self.x != self.staircase.bottom.x or self.staircase.force
                else:
                    onStairs = self.x != self.staircase.top.x or self.staircase.force
            self.walk(onStairs)

    def right(self):
        if self.direction > 0:
            if self.barrier and self.x < self.barrier.x:
                if self.adult and self.barrier.isOpenable():
                    return self.openDoor()
                return
            onStairs = False
            if self.staircase:
                if self.staircase.direction > 0:
                    onStairs = self.x != self.staircase.bottom.x or self.staircase.force
                else:
                    onStairs = self.x != self.staircase.top.x or self.staircase.force
            self.walk(onStairs)
        else:
            self.turn()

    def up(self):
        if self.staircase:
            if self.direction == self.staircase.direction or (self.x == self.staircase.top.x and self.staircase.force):
                return self.walk(True)
            elif self.x != self.staircase.top.x:
                return self.turn()
        if self.direction > 0:
            self.right()
        else:
            self.left()

    def down(self):
        if self.staircase:
            if self.direction != self.staircase.direction or (self.x == self.staircase.bottom.x and self.staircase.force):
                return self.walk(True)
            elif self.x != self.staircase.bottom.x:
                return self.turn()
        if self.direction > 0:
            self.right()
        else:
            self.left()

    def turn(self):
        self.direction *= -1

    def walk(self, onStairs=False):
        if onStairs:
            if self.direction == self.staircase.direction:
                if not self.midstride():
                    self.y -= 1
            else:
                self.verticalDirection = 1
        self.animatoryState += 1
        self.animatoryState &= 3
        if self.animatoryState %2 == 0:
            self.x += self.direction
            self.y += self.verticalDirection
            if self.wipingBoard:
                # Animatory state sequence is 0, 1, 0, 1... when wiping board
                self.animatoryState = 0
            self.verticalDirection = 0

    def chair(self):
        return self.skool.chair(self)

    def sit(self):
        if self.isSitting():
            self.getUp()
            return
        self.previousAS = self.animatoryState
        chair = self.chair()
        if chair:
            self.animatoryState = 4
            occupant = chair.occupant
            chair.seat(self)
            if occupant:
                occupant.sitOnFloor()
        else:
            self.sitOnFloor()

    def sitOnFloor(self):
        self.animatoryState = 5

    def isSitting(self):
        return self.isSittingOnFloor() or self.isSittingOnChair()

    def isSittingOnFloor(self):
        return self.animatoryState == 5

    def isSittingOnChair(self):
        return self.animatoryState == 4

    def getUp(self):
        if self.isSittingOnChair():
            self.chair().vacate()
        self.animatoryState = self.previousAS

    def midstride(self):
        return self.animatoryState & 13 == 1

    def getFloor(self, object=None):
        return self.skool.floor(object or self)

    def onStairs(self):
        if self.staircase:
            return self.x not in (self.staircase.bottom.x, self.staircase.top.x)
        return False

    def getNextStaircase(self, destination):
        return self.skool.nextStaircase(self.getFloor(), self.getFloor(destination))

    def openDoor(self):
        return OpenDoor(self.barrier.doorId)

    def moveDoor(self, doorId, shut):
        self.skool.moveDoor(doorId, shut)

    def raiseArm(self):
        self.animatoryState = 7

    def lowerArm(self):
        self.animatoryState = 0

    def getNextChair(self, moveAlong):
        return self.room.getNextChair(self, moveAlong)

    def signal(self, signal):
        self.game.signal(signal)

    def unsignal(self, signal):
        self.game.unsignal(signal)

    def gotSignal(self, signal):
        return self.game.gotSignal(signal)

    def isTimeToStartLesson(self):
        return self.timetable.isTimeToStartLesson()

    def say(self, words, shift):
        lipPos = (self.x + 1) % 8
        self.bubble, done = self.game.getBubble(words, lipPos, shift)
        return done

    def removeBubble(self):
        self.bubble = None

    def getAssemblyMessage(self):
        return self.assemblyMessageGenerator.generateMessage()

    def setWipingBoard(self, wiping):
        self.wipingBoard = wiping

    def wipeBoard(self, column):
        self.game.wipeBoard(self.room.blackboard, column)

    def writeOnBoard(self, message, index):
        return self.game.writeOnBoard(self.room.blackboard, message, index)

    def resolveLocationId(self, locationId):
        return self.game.resolveLocationId(locationId)

    def isTeachingEric(self):
        return self.timetable.isTeachingEric(self)

    def getAssemblyMessageGenerator(self):
        if self.assemblyMessageGenerator is None:
            self.assemblyMessageGenerator = AssemblyMessageGenerator()
        return self.assemblyMessageGenerator

    def prepareQA(self):
        question, answer = self.qaGenerator.prepareQA()
        self.game.setAnswer(answer)
        return question

    def getAnswer(self):
        return self.game.getAnswer()

    def getQAGenerator(self):
        if self.qaGenerator is None:
            self.qaGenerator = QAGenerator()
        return self.qaGenerator

    def addBlackboardMessage(self, message):
        self.blackboardMessages.append(message)

    def getBlackboardMessage(self):
        return self.blackboardMessages[random.randrange(len(self.blackboardMessages))]

    def run(self, run):
        self.speed = 2 if run else 1

class Eric(Character):
    def move(self, pressed_keys, moveNow):
        self.walkDelay -= 1
        if self.walkDelay > 0 and not moveNow:
            return 0
        self.walkDelay = 2
        if self.midstride():
            return self.walk()
        xd = 0
        yd = 0
        if pressed_keys[K_LEFT]:
            xd -= 1
        if pressed_keys[K_RIGHT]:
            xd += 1
        if pressed_keys[K_UP]:
            yd -= 1
        if pressed_keys[K_DOWN]:
            yd += 1
        if pressed_keys[K_s]:
            self.sit()
            return 0
        if self.isSitting():
            return 0
        self.staircase = self.skool.staircase(self)
        if not self.staircase and not self.skool.onFloor(self):
            self.y += 1
            return 0
        self.barrier = self.skool.barrier(self)
        if xd > 0:
            self.right()
        elif xd < 0:
            self.left()
        elif yd < 0:
            self.up()
        elif yd > 0:
            self.down()
        return 0

    def walk(self, onStairs=False):
        Character.walk(self, onStairs)
        if not self.midstride():
            offset = self.x - self.game.column
            if offset > 21:
                return 1
            elif offset < 10:
                return -1
        return 0

    def dethrone(self):
        self.animatoryState = 5

class QAGenerator:
    def __init__(self):
        self.questions = []
        self.answers = {}
        self.qaPairs = {}

    def addQuestion(self, questionId, qaGroup, text):
        self.questions.append((questionId, qaGroup, text))

    def addAnswer(self, questionId, text):
        self.answers[questionId] = text

    def addQAPair(self, qaGroup, word1, word2):
        if not self.qaPairs.has_key(qaGroup):
            self.qaPairs[qaGroup] = []
        self.qaPairs[qaGroup].append((word1, word2))

    def expand(self, template, word1, word2):
        return template.replace('$1', word1).replace('$2', word2)

    def prepareQA(self):
        questionId, qaGroup, question = self.questions[random.randrange(len(self.questions))]
        answer = self.answers[questionId]
        word1, word2 = self.qaPairs[qaGroup][random.randrange(len(self.qaPairs[qaGroup]))]
        return self.expand(question, word1, word2), self.expand(answer, word1, word2)

class AssemblyMessageGenerator:
    def __init__(self):
        self.text = None
        self.verbs = []
        self.nouns = []

    def setText(self, text):
        self.text = text

    def addVerb(self, verb):
        self.verbs.append(verb)

    def addNoun(self, noun):
        self.nouns.append(noun)

    def generateMessage(self):
        verb = self.verbs[random.randrange(len(self.verbs))]
        noun = self.nouns[random.randrange(len(self.nouns))]
        message = self.text.replace('$VERB', verb)
        return message.replace('$NOUN', noun)

class Cast:
    def __init__(self, surface):
        self.sprites = surface
        self.eric = None
        self.characters = {}
        self.taps = {}

    def addEric(self, characterId, name, states):
        self.eric = Eric(characterId, name)
        self.eric.setAnimatoryStates(*self.getAnimatoryStates(states))

    def addCharacter(self, characterId, name, states, adult):
        character = Character(characterId, name, adult)
        character.setAnimatoryStates(*self.getAnimatoryStates(states))
        self.characters[characterId] = character

    def getAnimatoryState(self, state):
        width = self.sprites.get_width() / 16
        height = self.sprites.get_height() / 8
        surface = self.sprites.subsurface((width * (state % 16), height * (state / 16)), (width, height))
        surface.set_colorkey((0, 255, 0))
        return surface

    def getAnimatoryStates(self, states):
        baseAS = states[0]
        asDictL = {}
        asDictR = {}
        for index in states[1:]:
            state = baseAS + index
            spriteL = self.getAnimatoryState(state)
            spriteR = pygame.transform.flip(spriteL, True, False)
            asDictL[index] = spriteL
            asDictR[index] = spriteR
        return asDictL, asDictR

    def initialise(self, game):
        self.eric.setGame(game)
        self.eric.setSkool(game.skool)
        self.eric.setTimetable(game.timetable)
        for character in self.characters.values():
            character.setGame(game)
            character.setSkool(game.skool)
            character.setTimetable(game.timetable)

    def addCommand(self, tapId, commandClass, *params):
        if not self.taps.has_key(tapId):
            self.taps[tapId] = CommandListTemplate(tapId)
        self.taps[tapId].addCommand(commandClass, *params)

    def setRandomLocations(self, characterId, locations):
        self.characters[characterId].setRandomLocations(locations)

    def addCommandList(self, characterId, lessonId, tapId):
        self.characters[characterId].addCommandList(lessonId, tapId)

    def setLocation(self, characterId, x, y):
        character = self.get(characterId)
        character.x, character.y = x, y

    def setSitDownMessage(self, characterId, message):
        self.characters[characterId].setSitDownMessage(message)

    def getEric(self):
        return self.eric

    def get(self, characterId):
        if characterId == self.eric.characterId:
            return self.eric
        return self.characters.get(characterId, None)

    def addBlackboardMessage(self, characterId, message):
        self.characters[characterId].addBlackboardMessage(message)

    def move(self, pressedKeys, moveNow=False):
        for character in self.characters.values():
            character.move(moveNow)
        return self.eric.move(pressedKeys, moveNow)

    def draw(self, surface, column):
        for character in self.characters.values():
            character.draw(surface, column)
        self.eric.draw(surface, column)

    def setLesson(self, lessonId):
        for character in self.characters.values():
            tapId = character.getTapId(lessonId)
            character.setAPTable(APTable(self.taps[tapId]))
            character.removeBubble()
