# Copyright 2008 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import sys
import os
import pygame
from pygame.locals import *
from character import *
from skool import *
from ai import *
from skoolbuilder import SkoolBuilder

class Game:
    def __init__(self):
        builder = SkoolBuilder()
        self.initialiseColours()
        self.config = builder.getGameConfig(self.iniFile)
        self.scale = self.config.get('scale', 2)
        self.imageDir = self.config['imagedir']
        self.imageSet = self.config.get('imageset', 'original')
        lessonLength = self.config.get('lessonlength', 1200)
        lessonStartTime = self.config.get('lessonstarttime', 600)
        self.timetable = Timetable(lessonLength, lessonStartTime)
        self.screen = pygame.display.set_mode((256 * self.scale, 192 * self.scale))
        self.screen.fill(self.colours[self.background])

        self.lessonBoxAttr = int(self.config['lessonboxattribute'])
        lessonBoxCoords = [int(c) for c in self.config['lessonboxpos'].split(',')]
        self.lessonBoxPos = self.scaleCoords(lessonBoxCoords)
        logoCoords = [int(c) for c in self.config['logopos'].split(',')]
        self.screen.blit(self.getImage("logo.png"), self.scaleCoords(logoCoords))
        self.scoreBox = self.getImage("scorebox.png")
        self.scoreBoxCoords = tuple(int(c) for c in self.config['scoreboxpos'].split(','))
        self.scoreBoxXOffset = int(self.config['scoreboxxoffset'])
        self.screen.blit(self.scoreBox, self.scaleCoords(self.scoreBoxCoords))

        self.cast = Cast(self.getImage("sprites.png"))
        skool = skool_ink = skool_paper = None
        mode = int(self.config.get('graphicsmode', 1))
        if mode == 0:
            skool = self.getImage("skool.png")
        else:
            skool_ink = self.getImage("skool_ink.png")
            skool_paper = self.getImage("skool_paper.png")
        self.skool = Skool(self.scale, skool, skool_ink, skool_paper)
        self.bubble = self.getImage("bubble.png")
        self.loadMutableImages(mode)
        self.loadInventoryImage()

        builder.readIniFile(self.iniFile, self.skool, self.timetable, self.cast)
        self.font = Font(self.getImage("font.png"), builder.characterOffsets)
        self.initialiseDoorImages(builder.doorImages)

        self.maxColumn = self.skool.getWidth() - 32

        self.cast.initialise(self)
        self.score = 0
        self.lines = 0
        self.hiscore = 0
        self.column = min(self.maxColumn, max((self.cast.getEric().x - 16) / 8 * 8, 0))

        self.clock = pygame.time.Clock()
        self.lastKeys = pygame.key.get_pressed()
        self.screenshot = 0

    def getImage(self, fname):
        scaleUp = True
        imageSetDir = os.path.join(self.imageDir, '%sx%i' % (self.imageSet, self.scale))
        if os.path.isdir(imageSetDir):
            scaleUp = False
        else:
            imageSetDir = os.path.join(self.imageDir, '%sx1' % self.imageSet)
        img = pygame.image.load(os.path.join(imageSetDir, fname)).convert()
        if scaleUp:
            img = pygame.transform.scale(img, (self.scale * img.get_width(), self.scale * img.get_height()))
        return img

    def loadMutableImages(self, mode):
        return

    def loadInventoryImage(self):
        return

    def initialiseDoorImages(self, doorImages):
        for doorId, tuples in doorImages.iteritems():
            self.setDoorImages(doorId, *tuples)

    def scaleCoords(self, coords):
        return (8 * self.scale * coords[0], 8 * self.scale * coords[1])

    def play(self):
        while True:
            for event in pygame.event.get():
                if event.type == QUIT:
                    return

            if self.timetable.tick():
                self.changeLesson()

            pressed_keys = pygame.key.get_pressed()

            if pressed_keys[K_q]:
                return

            if pressed_keys[K_l] and not self.lastKeys[K_l]:
                self.changeLesson()
            if pressed_keys[K_z]:
                self.addToScore(10 * random.randrange(1, 9))
            if pressed_keys[K_x]:
                self.addLines(10 * random.randrange(1, 9))
            if pressed_keys[K_c]:
                self.setHiScore()
            if pressed_keys[K_k] and not self.lastKeys[K_k]:
                self.takeScreenshot()

            # Hold down 'a' for turbo mode
            turboMode = pressed_keys[K_a]
            scroll = self.cast.move(pressed_keys, turboMode)
            self.skool.autoShutDoors()

            if turboMode:
                self.timetable.tick()
            else:
                self.clock.tick(20)

            self.lastKeys = pressed_keys

            self.draw()
            if self.column < self.maxColumn and scroll > 0:
                self.scroll(1)
            elif self.column > 0 and scroll < 0:
                self.scroll(-1)

    def scroll(self, inc):
        for i in range(0, 8):
            self.column += inc
            self.draw()
            self.clock.tick(12)

    def draw(self):
        self.skool.draw(self.screen, self.column, self.cast)
        pygame.display.update()

    def takeScreenshot(self):
        fname = 'pyskool%s.bmp' % str(self.screenshot).zfill(3)
        pygame.image.save(self.screen, fname)
        self.screenshot += 1

    def getLessonDesc(self):
        teacherId, roomId = self.timetable.getLessonDetails()
        room = self.skool.getRoom(roomId)
        teacher = self.cast.get(teacherId)
        return teacher.name if teacher else teacherId, room.name if room else roomId

    def printLesson(self):
        teacher, room = self.getLessonDesc()
        if not teacher:
            elements = room.split(' ')
            if len(elements) > 1:
                teacher = elements[0]
                room = elements[1]
        lessonBox = pygame.Surface((self.scale * 64, self.scale * 24))
        paper = self.colours[(self.lessonBoxAttr & 120) / 8]
        lessonBox.fill(paper)
        ink = self.colours[(self.lessonBoxAttr & 64) / 8 + (self.lessonBoxAttr & 7)]
        teacherText = self.getText(teacher, ink, paper)
        roomText = self.getText(room, ink, paper)
        fontHeight = self.scale * 8
        teacherX = (lessonBox.get_width() - teacherText.get_width()) / 2
        teacherY = (lessonBox.get_height() - 2 * fontHeight) / 2
        lessonBox.blit(teacherText, (teacherX, teacherY))
        roomX = (lessonBox.get_width() - roomText.get_width()) / 2
        roomY = teacherY + fontHeight
        lessonBox.blit(roomText, (roomX, roomY))
        self.screen.blit(lessonBox, self.lessonBoxPos)

    def changeLesson(self):
        self.timetable.nextLesson()
        self.printLesson()
        lessonId = self.timetable.getLessonId()
        self.cast.setLesson(lessonId)
        self.signals = {}

    def signal(self, signal):
        self.signals[signal] = True

    def unsignal(self, signal):
        self.signals[signal] = False

    def gotSignal(self, signal):
        return self.signals.get(signal, False)

    def getText(self, words, ink, paper):
        return self.font.render(words, ink, paper)

    def getBubble(self, words, lipPos, shift):
        ink = self.colours[0]
        paper = self.colours[7]
        bubble = pygame.Surface(self.scaleCoords((8, 3)))
        colorkey = (0, 255, 0)
        bubble.fill(colorkey)
        bubble.set_colorkey(colorkey)
        bubble.blit(self.bubble, (0, 0))
        lip = self.bubble.subsurface(self.scaleCoords((8, 0)), self.scaleCoords((1, 1)))
        lipCoords = self.scaleCoords((lipPos, 2))
        bubble.blit(lip, lipCoords)
        openLipByte = 66
        for bit in range(0, 8 * self.scale, self.scale):
            colour = ink if openLipByte & 128 else paper
            for r in range(0, self.scale + 1):
                for c in range(0, self.scale + 1):
                    bubble.set_at((lipCoords[0] + bit + c, lipCoords[1] - 1 - r), colour)
            openLipByte *= 2
        text = self.getText(words, ink, paper)
        insetX = (4 - 8 * min(shift, 0)) * self.scale
        insetY = 4 * self.scale
        textX = max(shift, 0) * 8 * self.scale
        width = min(56 * self.scale, text.get_width() - textX)
        if insetX + width > 56 * self.scale:
            width = 56 * self.scale - insetX
        textWindow = text.subsurface((textX, 0), (width, 8 * self.scale))
        bubble.blit(textWindow, (insetX, insetY))
        return (bubble, textX > text.get_width())

    def printNumber(self, number, yOffset):
        if number <= 0:
            return

        # Collect the ink and paper colour by inspecting pixels in the score
        # box; maybe these should be game config parameters?
        ink = self.scoreBox.get_at(((self.scoreBoxXOffset + 3) * self.scale, self.scale))
        paper = self.scoreBox.get_at((0, 0))

        numberText = self.getText(str(number), ink, paper)
        width = 24 * self.scale
        background = pygame.Surface((width, 7 * self.scale))
        background.fill(paper)
        background.blit(numberText, (width - numberText.get_width(), 0))
        coords = self.scaleCoords(self.scoreBoxCoords)
        topLeftX = coords[0] + self.scoreBoxXOffset * self.scale - width
        topLeftY = coords[1] + yOffset * self.scale
        self.screen.blit(background, (topLeftX, topLeftY))

    def printScore(self):
        self.printNumber(self.score, 1)

    def printLines(self):
        self.printNumber(self.lines, 9)

    def printHiScore(self):
        self.printNumber(self.hiscore, 17)

    def addToScore(self, addend):
        self.score += addend
        self.printScore()

    def addLines(self, addend):
        self.lines += addend
        self.printLines()

    def setHiScore(self):
        self.hiscore = self.score
        self.printHiScore()

    def wipeBoard(self, blackboard, column):
        if blackboard is None:
            sys.stderr.write('Cannot wipe non-existent blackboard')
            return
        if column == 0:
            blackboard.clear()
            return
        wipedBit = pygame.Surface((8 * self.scale, 8 * self.scale))
        for line in blackboard.text:
            line.blit(wipedBit, (8 * column * self.scale, 0))

    def writeOnBoard(self, blackboard, message, index):
        if blackboard is None:
            sys.stderr.write('Cannot write on non-existent blackboard')
            return
        lines = [self.getText(line, (255, 255, 255), self.colours[0]) for line in message[:index].split('^')]
        blackboard.write(lines)
        return index == len(message)

    def resolveLocationId(self, locationId):
        destMarker = 'Destination:'
        if locationId.startswith(destMarker):
            character = self.cast.get(locationId[len(destMarker):])
            # TODO This should be wherever the character is GOING, not where he
            # IS
            return Location((character.x, 3 + 7 * (character.y / 7)))
        return self.skool.resolveLocationId(locationId)

    def getAnswer(self):
        return self.answer

    def setAnswer(self, answer):
        self.answer = answer

    def initialiseColours(self):
        self.colours = []
        self.colours.append((0, 0, 0))       #  0 - black
        self.colours.append((0, 0, 197))     #  1 - blue
        self.colours.append((197, 0, 0))     #  2 - red
        self.colours.append((197, 0, 197))   #  3 - magenta
        self.colours.append((0, 198, 0))     #  4 - green
        self.colours.append((0, 198, 197))   #  5 - cyan
        self.colours.append((197, 198, 0))   #  6 - yellow
        self.colours.append((205, 198, 205)) #  7 - white
        self.colours.append((0, 0, 0))       #  8 - bright black
        self.colours.append((0, 0, 255))     #  9 - bright blue
        self.colours.append((255, 0, 0))     # 10 - bright red
        self.colours.append((255, 0, 255))   # 11 - bright magenta
        self.colours.append((0, 255, 0))     # 12 - bright green
        self.colours.append((0, 255, 255))   # 13 - bright cyan
        self.colours.append((255, 255, 0))   # 14 - bright yellow
        self.colours.append((255, 255, 255)) # 15 - bright white
