# Copyright 2008 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

from skool import *
from ai import *
from character import *

class SkoolBuilder:
    def getGameConfig(self, iniFile):
        config = {}
        f = open(iniFile, 'r')
        for line in f:
            if line[0] == '[' and line[1:].strip() != 'Game]':
                break
            if line.isspace():
                continue
            elif line.startswith(';'):
                continue
            elif line.startswith('[Game]'):
                continue
            else:
                elements = [e.strip() for e in line.strip().lower().split(':')]
                param = elements[0]
                value = elements[1]
                if value.isdigit():
                    value = int(value)
                config[param] = value
        f.close()
        return config

    def readIniFile(self, iniFile, skool, timetable, cast):
        self.cast = cast
        self.timetable = timetable
        self.skool = skool
        self.animatoryStates = {}
        self.doorImages = {}
        self.characterOffsets = []
        f = open(iniFile, 'r')
        for line in f:
            if line.isspace():
                continue
            elif line.startswith(';'):
                continue
            elif line.startswith('[Game]'):
                addConfig = self.doNothing
                params = ()
            elif line.startswith('[AnimatoryStates]'):
                addConfig = self.addAnimatoryStates
                params = ()
            elif line.startswith('[Eric]'):
                addConfig = self.addEric
                params = ()
            elif line.startswith('[Characters]'):
                addConfig = self.addCharacter
                params = ()
            elif line.startswith('[Timetable]'):
                addConfig = self.addLesson
                params = ()
            elif line.startswith('[Lesson '):
                addConfig = self.addCommandList
                details = line[8:line.index(']')]
                index = details.index(' ')
                lessonId = details[:index]
                lessonDetails = [e.strip() for e in details[index + 1:].split(',')]
                roomId = lessonDetails[-1]
                teacherId = lessonDetails[0] if len(lessonDetails) > 1 else ''
                self.timetable.addLessonDetails(lessonId, teacherId, roomId)
                params = (lessonId,)
            elif line.startswith('[InitialLocations]'):
                addConfig = self.addInitialLocation
                params = ()
            elif line.startswith('[RandomLocations]'):
                addConfig = self.addRandomLocations
                params = ()
            elif line.startswith('[SkoolLocations]'):
                addConfig = self.addSkoolLocation
                params = ()
            elif line.startswith('[CommandList '):
                addConfig = self.addCommand
                params = (line[13:line.index(']')],)
            elif line.startswith('[Rooms]'):
                addConfig = self.addRoom
                params = ()
            elif line.startswith('[Chairs]'):
                addConfig = self.addChairs
                params = ()
            elif line.startswith('[Doors]'):
                addConfig = self.addDoor
                params = ()
            elif line.startswith('[DoorImages]'):
                addConfig = self.addDoorImages
                params = ()
            elif line.startswith('[Walls]'):
                addConfig = self.addWall
                params = ()
            elif line.startswith('[Staircases]'):
                addConfig = self.addStaircase
                params = ()
            elif line.startswith('[Floors]'):
                addConfig = self.addFloor
                params = ()
            elif line.startswith('[Routes]'):
                addConfig = self.addRoute
                params = ()
            elif line.startswith('[SitDownMessages]'):
                addConfig = self.addSitDownMessage
                params = ()
            elif line.startswith('[CharacterWidths]'):
                addConfig = self.addCharacterOffset
                params = ()
            elif line.startswith('[AssemblyMessages]'):
                addConfig = self.addAssemblyMessage
                params = ()
            elif line.startswith('[Blackboards]'):
                addConfig = self.addBlackboard
                params = ()
            elif line.startswith('[BlackboardMessages]'):
                addConfig = self.addBlackboardMessage
                params = ()
            elif line.startswith('[QuestionsAndAnswers '):
                addConfig = self.addQA
                teacherId = line[21:line.index(']')].strip()
                params = (teacherId,)
            else:
                addConfig(line.rstrip(), *params)
        f.close()

    def doNothing(self, details):
        return

    def addAnimatoryStates(self, details):
        elements = [int(e) for e in details.split(',')]
        self.animatoryStates[elements[0]] = elements

    def addEric(self, details):
        elements = [e.strip() for e in details.split(',')]
        characterId = elements[0]
        name = elements[1]
        baseAS = int(elements[2])
        self.cast.addEric(characterId, name, self.animatoryStates[baseAS])
        self.cast.setLocation(characterId, int(elements[3][1:]), int(elements[4][:-1]))

    def addCharacter(self, details):
        elements = [e.strip() for e in details.split(',')]
        characterId = elements[0]
        name = elements[1]
        baseAS = int(elements[2])
        adult = 'A' in elements[3]
        self.cast.addCharacter(characterId, name, self.animatoryStates[baseAS], adult)

    def addLesson(self, lessonType):
        self.timetable.addLesson(lessonType)

    def addCommandList(self, details, lessonId):
        elements = [e.strip() for e in details.split(',')]
        characterId = elements[0]
        tapId = elements[1]
        self.cast.addCommandList(characterId, lessonId, tapId)

    def addInitialLocation(self, details):
        elements = [e.strip() for e in details.split(',')]
        self.cast.setLocation(elements[0], int(elements[1]), int(elements[2]))

    def addRandomLocations(self, details):
        elements = self.getElements(details)
        characterId = elements[0]
        locations = elements[1:]
        self.cast.setRandomLocations(characterId, locations)

    def addSkoolLocation(self, details):
        elements = [e.strip() for e in details.split(',')]
        locationId = elements[0]
        x, y = int(elements[1]), int(elements[2])
        self.skool.addLocation(locationId, (x, y))

    def addCommand(self, command, tapId):
        elements = self.getElements(command)
        commandClass = eval(elements[0])
        params = [int(e) if e.isdigit() else e for e in elements[1:]]
        self.cast.addCommand(tapId, commandClass, *params)

    def addRoom(self, details):
        elements = [e.strip() for e in details.split(',')]
        roomId = elements[0]
        name = elements[1]
        y = int(elements[2])
        minX = int(elements[3])
        maxX = int(elements[4])
        self.skool.addRoom(roomId, name, y, minX, maxX)

    def addChairs(self, details):
        elements = [e.strip() for e in details.split(',')]
        roomId = elements[0]
        for x in elements[1:]:
            self.skool.addChair(roomId, int(x))

    def addDoor(self, details):
        elements = [e.strip() for e in details.split(',')]
        doorId = elements[0]
        x = int(elements[1])
        bottomY = int(elements[2])
        topY = int(elements[3])
        shut = elements[4][0].lower() == 's'
        autoShuts = len(elements) > 5
        self.skool.addDoor(doorId, x, bottomY, topY, shut, autoShuts)

    def addDoorImages(self, details):
        elements = self.getElements(details)
        doorId = elements[0]
        tuples = elements[1:]
        self.doorImages[doorId] = tuples

    def addWall(self, details):
        params = [int(e) for e in details.split(',')]
        x = params[0]
        bottomY = params[1]
        topY = params[2]
        self.skool.addWall(x, bottomY, topY)

    def addStaircase(self, details):
        elements = [e.strip() for e in details.split(',')]
        staircaseIds = elements[0].partition(':')
        bottom = (int(elements[1][1:]), int(elements[2][:-1]))
        top = (int(elements[3][1:]), int(elements[4][:-1]))
        force = len(elements) == 6 and elements[5][0].lower() == 't'
        self.skool.addStaircase(staircaseIds[0], bottom, top, force, staircaseIds[2])

    def addFloor(self, details):
        elements = [e.strip() for e in details.split(',')]
        floorId = elements[0]
        minX = int(elements[1])
        maxX = int(elements[2])
        y = int(elements[3])
        self.skool.addFloor(floorId, minX, maxX, y)

    def addRoute(self, details):
        elements = [e.strip() for e in details.split(',')]
        homeFloorId = elements[0]
        staircaseId = elements[-1]
        destFloorIds = elements[1:-1]
        self.skool.addRoutes(homeFloorId, destFloorIds, staircaseId)

    def addSitDownMessage(self, details):
        elements = [e.strip() for e in details.split(',')]
        characterId = elements[0]
        message = elements[1]
        self.cast.setSitDownMessage(characterId, message)

    def addCharacterOffset(self, details):
        width = 1 + int(details[details.index(',', 1) + 1:])
        if self.characterOffsets:
            lastOffset, lastWidth = self.characterOffsets[-1]
            self.characterOffsets.append((lastOffset + lastWidth, width))
        else:
            self.characterOffsets.append((0, width))

    def addAssemblyMessage(self, details):
        elements = [e.strip() for e in details.split(',')]
        generator = self.cast.get('WACKER').getAssemblyMessageGenerator()
        if elements[0] == 'MESSAGE':
            generator.setText(elements[1])
        elif elements[0] == 'VERB':
            generator.addVerb(elements[1])
        elif elements[0] == 'NOUN':
            generator.addNoun(elements[1])

    def addBlackboard(self, details):
        elements = [e.strip() for e in details.split(',')]
        roomId = elements[0]
        x, y = int(elements[1]), int(elements[2])
        self.skool.addBlackboard(roomId, x, y)

    def addBlackboardMessage(self, details):
        elements = details.partition(',')
        characterId = elements[0]
        message = elements[2].strip()
        self.cast.addBlackboardMessage(characterId, message)

    def addQA(self, details, teacherId):
        qaGenerator = self.cast.get(teacherId).getQAGenerator()
        elements = self.getElements(details)
        entryType = elements[0][:2].lower()
        if entryType == 'qu':
            questionId = elements[1]
            qaGroup = elements[2]
            text = elements[3]
            qaGenerator.addQuestion(questionId, qaGroup, text)
        elif entryType == 'an':
            questionId = elements[1]
            text = elements[2]
            qaGenerator.addAnswer(questionId, text)
        elif entryType == 'qa':
            qaGroup = elements[1]
            qaGenerator.addQAPair(qaGroup, elements[2], elements[3])

    def getElements(self, details):
        elements = []
        index = 0
        while index < len(details):
            if details[index].isspace():
                index += 1
            elif details[index] in '"\'':
                quote = details[index]
                end = details.index(quote, index + 1)
                elements.append(details[index + 1:end])
                index = details.find(',', end + 1)
                if index < 0:
                    index = len(details)
                index += 1
            elif details[index] == '(':
                end = details.index(')', index)
                coords = details[index + 1:end].split(',')
                elements.append((int(coords[0]), int(coords[1])))
                index = details.find(',', end + 1)
                if index < 0:
                    index = len(details)
                index += 1
            else:
                end = details.find(',', index)
                if end < 0:
                    end = len(details)
                elements.append(details[index:end].strip())
                index = end + 1
        return elements
