# Copyright 2008 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import random
import pygame
from pygame.locals import *
from skool import Location
from ai import *

class Character:
    def __init__(self, characterId, name, flags=''):
        self.characterId = characterId
        self.name = name
        self.flags = flags
        self.commandList = CommandList()
        self.x = 0
        self.y = 0
        self.direction = 1 # 1 = facing right, -1 = facing left
        self.verticalDirection = 0 # 1 = down, 0 = neither up nor down
        self.speed = 1 # 1 = walk, 2 = run
        self.speedChangeDelay = 1
        self.walkDelay = random.randrange(3, 6)
        self.speechDelay = 1
        self.actionDelay = 1
        self.staircase = None
        self.barrier = None
        self.room = None
        self.commandLists = {}
        self.sitDownMessage = None
        self.bubble = None
        self.blackboardMessages = []
        self.qaGenerator = None
        self.wipingBoard = False
        self.pellet = None

    def setGame(self, game):
        self.game = game

    def setCast(self, cast):
        self.cast = cast

    def setSkool(self, skool):
        self.skool = skool

    def setTimetable(self, timetable):
        self.timetable = timetable

    def setSitDownMessage(self, message):
        self.sitDownMessage = message

    def setAnimatoryStates(self, asDictL, asDictR):
        self.animatoryState = 0
        self.asDictL = asDictL
        self.asDictR = asDictR

    def initialiseAnimatoryState(self, initialAS):
        self.direction = -1 if initialAS[0].upper() == 'L' else 1
        self.animatoryState = int(initialAS[1:])

    def setControllingCommand(self, command):
        self.commandList.setControllingCommand(command)

    def isAdult(self):
        return 'A' in self.flags

    def isConkerable(self):
        return 'C' in self.flags

    def canOpenDoors(self):
        return 'D' in self.flags

    def isPunchable(self):
        return 'F' in self.flags

    def canGiveLines(self):
        return 'L' in self.flags

    def isScaredOfMice(self):
        return 'M' in self.flags

    def isPelletable(self):
        return 'P' in self.flags

    def sometimesRuns(self):
        return 'W' not in self.flags

    def isHome(self, x):
        if 'G' in self.flags:
            return self.x > x
        elif 'B' in self.flags:
            return self.x < x - 1
        return True

    def getImage(self):
        asDict = self.asDictR if self.direction > 0 else self.asDictL
        return asDict[self.animatoryState]

    def draw(self, surface, column):
        surface.blit(self.getImage(), self.game.scaleCoords((self.x - column, self.y)))
        if self.bubble:
            bubbleX = 8 * ((self.x + 1) / 8) - column
            bubbleY = self.y - (3 if self.isAdult() else 2)
            surface.blit(self.bubble, self.game.scaleCoords((bubbleX, bubbleY)))

    def setRandomLocations(self, locations):
        self.randomLocations = locations

    def getRandomDestination(self):
        return self.randomLocations[random.randrange(len(self.randomLocations))]

    def addCommandList(self, lessonId, tapId):
        self.commandLists[lessonId] = tapId

    def getTapId(self, lessonId):
        return self.commandLists[lessonId]

    def setCommandListTemplate(self, template):
        self.commandList.setTemplate(template)

    def restartTable(self):
        self.commandList.restart()

    def isTimeToMove(self):
        if self.bubble:
            self.speechDelay = (self.speechDelay + 1) % 3
            return self.speechDelay == 0
        if self.animatoryState > 7:
            # Perform hitting, firing etc. quickly
            self.actionDelay = (self.actionDelay + 1) % 2
            return self.actionDelay == 0
        self.walkDelay -= 1
        if self.walkDelay > 0:
            return False
        self.speedChangeDelay -= 1
        if self.speedChangeDelay == 0:
            self.speedChangeDelay = random.randrange(16, 32)
            if self.sometimesRuns():
                self.run(self.speedChangeDelay & 1)
        self.walkDelay = 7 - 2 * self.speed
        return True

    def move(self, moveNow):
        if not (self.isTimeToMove() or moveNow):
            return
        if self.midstride():
            return self.walk()
        self.staircase = self.skool.staircase(self)
        self.barrier = self.skool.barrier(self)
        self.room = self.skool.room(self)
        self.commandList.command(self)

    def left(self):
        if self.direction > 0:
            self.turn()
        else:
            if self.barrier and self.barrier.x <= self.x:
                if self.canOpenDoors() and self.barrier.isOpenable():
                    return self.openDoor()
                return
            onStairs = False
            if self.staircase:
                if self.staircase.direction < 0:
                    onStairs = self.x != self.staircase.bottom.x or self.staircase.force
                else:
                    onStairs = self.x != self.staircase.top.x or self.staircase.force
            self.walk(onStairs)

    def right(self):
        if self.direction > 0:
            if self.barrier and self.x < self.barrier.x:
                if self.canOpenDoors() and self.barrier.isOpenable():
                    return self.openDoor()
                return
            onStairs = False
            if self.staircase:
                if self.staircase.direction > 0:
                    onStairs = self.x != self.staircase.bottom.x or self.staircase.force
                else:
                    onStairs = self.x != self.staircase.top.x or self.staircase.force
            self.walk(onStairs)
        else:
            self.turn()

    def up(self):
        if self.staircase:
            if self.direction == self.staircase.direction or (self.x == self.staircase.top.x and self.staircase.force):
                return self.walk(True)
            elif self.x != self.staircase.top.x:
                return self.turn()
        if self.direction > 0:
            self.right()
        else:
            self.left()

    def down(self):
        if self.staircase:
            if self.direction != self.staircase.direction or (self.x == self.staircase.bottom.x and self.staircase.force):
                return self.walk(True)
            elif self.x != self.staircase.bottom.x:
                return self.turn()
        if self.direction > 0:
            self.right()
        else:
            self.left()

    def turn(self):
        self.direction *= -1

    def walk(self, onStairs=False):
        if onStairs:
            if self.direction == self.staircase.direction:
                if not self.midstride():
                    self.y -= 1
            else:
                self.verticalDirection = 1
        self.animatoryState += 1
        self.animatoryState &= 3
        if self.animatoryState %2 == 0:
            self.x += self.direction
            self.y += self.verticalDirection
            if self.wipingBoard:
                # Animatory state sequence is 0, 1, 0, 1... when wiping board
                self.animatoryState = 0
            self.verticalDirection = 0

    def chair(self):
        return self.skool.chair(self)

    def sit(self):
        if self.isSitting():
            self.getUp()
            return
        self.previousAS = self.animatoryState
        chair = self.chair()
        if chair:
            self.animatoryState = 4
            occupant = chair.occupant
            chair.seat(self)
            if occupant:
                occupant.dethrone()
        else:
            self.sitOnFloor()

    def dethrone(self):
        self.sitOnFloor()
        self.addCommand(Dethroned())

    def sitOnFloor(self):
        self.animatoryState = 5

    def raiseFist(self):
        self.animatoryState = 8

    def punch(self):
        self.animatoryState = 9

    def lowerFist(self):
        self.animatoryState = 8

    def raiseCatapult(self):
        self.animatoryState = 10

    def fireCatapult(self):
        self.animatoryState = 11

    def lowerCatapult(self):
        self.animatoryState = 10

    def completeAction(self):
        self.animatoryState = 0

    def isStanding(self):
        return self.animatoryState < 4

    def isSitting(self):
        return self.isSittingOnFloor() or self.isSittingOnChair()

    def isSittingOnFloor(self):
        return self.animatoryState == 5

    def isSittingOnChair(self):
        return self.animatoryState == 4

    def isKnockedOver(self):
        return self.animatoryState == 6

    def isRaisingCatapult(self):
        return self.animatoryState == 10

    def isFiringCatapult(self):
        return self.animatoryState == 11

    def isLoweringCatapult(self):
        return self.animatoryState == 10

    def isRaisingFist(self):
        return self.animatoryState == 8

    def isPunching(self):
        return self.animatoryState == 9

    def isLoweringFist(self):
        return self.animatoryState == 8

    def isHittable(self):
        """Returns whether this character is in an animatory state that makes
        him punchable or vulnerable to a catapult pellet."""
        if self.isAdult():
            return self.isStanding() or self.animatoryState == 7
        return self.isStanding() or self.isSittingOnChair()

    def getUp(self):
        if self.isSittingOnChair():
            self.chair().vacate()
        self.animatoryState = self.previousAS

    def midstride(self):
        return self.animatoryState & 13 == 1

    def getFloor(self, object=None):
        return self.skool.floor(object or self)

    def onStairs(self):
        if self.staircase:
            return self.x not in (self.staircase.bottom.x, self.staircase.top.x)
        return False

    def getNextStaircase(self, destination):
        return self.skool.nextStaircase(self.getFloor(), self.getFloor(destination))

    def openDoor(self):
        return OpenDoor(self.barrier.doorId)

    def moveDoor(self, doorId, shut):
        self.skool.moveDoor(doorId, shut)

    def raiseArm(self):
        self.animatoryState = 7

    def lowerArm(self):
        self.animatoryState = 0

    def getNextChair(self, moveAlong):
        return self.room.getNextChair(self, moveAlong)

    def signal(self, signal):
        self.game.signal(signal)

    def unsignal(self, signal):
        self.game.unsignal(signal)

    def gotSignal(self, signal):
        return self.game.gotSignal(signal)

    def isTimeToStartLesson(self):
        return self.timetable.isTimeToStartLesson()

    def say(self, words, shift):
        lipPos = (self.x + 1) % 8
        self.bubble, done = self.game.getBubble(words, lipPos, shift)
        return done

    def removeBubble(self):
        self.bubble = None

    def getAssemblyMessage(self):
        return self.skool.getAssemblyMessage()

    def setWipingBoard(self, wiping):
        self.wipingBoard = wiping

    def wipeBoard(self, column):
        self.game.wipeBoard(self.room.blackboard, column)

    def writeOnBoard(self, message, index):
        return self.game.writeOnBoard(self.room.blackboard, message, index)

    def resolveLocationId(self, locationId):
        return self.game.resolveLocationId(locationId)

    def isTeachingEric(self):
        return self.timetable.isTeachingEric(self)

    def prepareQA(self):
        question, answer = self.qaGenerator.prepareQA()
        self.game.setAnswer(answer)
        return question

    def getAnswer(self):
        return self.game.getAnswer()

    def getQAGenerator(self):
        if self.qaGenerator is None:
            self.qaGenerator = QAGenerator()
        return self.qaGenerator

    def addBlackboardMessage(self, message):
        self.blackboardMessages.append(message)

    def getBlackboardMessage(self):
        message = self.blackboardMessages[random.randrange(len(self.blackboardMessages))]
        return self.cast.expandNames(message)

    def run(self, run):
        self.speed = 2 if run else 1

    def getFacingCharacters(self, offset):
        """Returns a list of any characters facing this character from a
        distance equal to offset."""
        return self.cast.getFacingCharacters(self, offset)

    def addCommand(self, command):
        self.commandList.addCommand(command)

    def knockOver(self):
        self.previousAS = self.animatoryState
        self.animatoryState = 6

    def deck(self):
        self.knockOver()
        if self.isAdult():
            self.addCommand(KnockedOver())
        else:
            if self.previousAS == 4:
                self.chair().vacate()
                self.addCommand(FindSeat())
            self.addCommand(KnockedOut())

    def standUp(self):
        self.animatoryState = 0

    def setPellet(self, pellet):
        self.pellet = pellet

    def fire(self):
        self.pellet.x = self.x + self.direction
        self.pellet.y = self.y
        self.pellet.direction = self.direction
        self.pellet.actionDelay = 1
        self.pellet.addCommand(Fly())

    def getNearbyAdults(self):
        """Returns a list of adults who are close enough to this character
        (and are facing the right way) to be able to see him."""
        return self.cast.getNearbyAdults(self)

    def setRestartPoint(self):
        """Discard the current and all previous commands in the command list (so
        any command that restarts the command list will see the next command as
        the first)."""
        self.commandList.setRestartPoint()

    def jumpIfShut(self, doorId, offset):
        """Jump forwards (or backwards) in the command list if the given door is
        shut."""
        if self.skool.isDoorShut(doorId):
            self.commandList.jump(offset)

    def jumpIfOpen(self, doorId, offset):
        """Jump forwards (or backwards) in the command list if the given door is
        open."""
        if not self.skool.isDoorShut(doorId):
            self.commandList.jump(offset)

    def checkDoorStatus(self, doorId, shut):
        return self.skool.isDoorShut(doorId) == shut

    def stalk(self, characterId):
        """Sets this character's destination equal to that of another
        character's destination (if they are both GoTo-ing)."""
        target = self.cast.get(characterId)
        if target:
            self.commandList.setGoToDestination(target.getGoToDestination())

    def getGoToDestination(self):
        """Returns the destination of this character, or None if they are not
        under the control of the GoTo command."""
        return self.commandList.getGoToDestination()

    def waitAtDoor(self, doorId):
        """Returns whether the characters are on the correct side of the given
        door."""
        return self.cast.isHome(self.skool.getDoor(doorId).x)

class Eric(Character):
    def __init__(self, characterId, name, flags):
        Character.__init__(self, characterId, name, flags)
        self.controller = None

    def move(self, pressed_keys, moveNow):
        self.walkDelay -= 1
        if self.walkDelay > 0 and not moveNow:
            return 0
        self.walkDelay = 2
        if self.controller:
            if self.controller.command(self) is self.controller:
                self.controller = None
            return 0
        if self.midstride():
            return self.walk()
        xd = 0
        yd = 0
        if pressed_keys[K_LEFT]:
            xd -= 1
        if pressed_keys[K_RIGHT]:
            xd += 1
        if pressed_keys[K_UP]:
            yd -= 1
        if pressed_keys[K_DOWN]:
            yd += 1
        if pressed_keys[K_s]:
            if self.isKnockedOver():
                self.standUp()
            else:
                self.sit()
            return 0
        if pressed_keys[K_f] and not self.pellet.isAirborne():
            self.controller = FireCatapult()
            return 0
        if pressed_keys[K_h]:
            self.controller = Hit()
            return 0
        if self.isSitting():
            return 0
        self.staircase = self.skool.staircase(self)
        if not self.staircase and not self.skool.onFloor(self):
            self.y += 1
            return 0
        self.barrier = self.skool.barrier(self)
        if xd > 0:
            self.right()
        elif xd < 0:
            self.left()
        elif yd < 0:
            self.up()
        elif yd > 0:
            self.down()
        return 0

    def walk(self, onStairs=False):
        Character.walk(self, onStairs)
        if not self.midstride():
            offset = self.x - self.game.column
            if offset > 21:
                return 1
            elif offset < 10:
                return -1
        return 0

    def dethrone(self):
        self.animatoryState = 5

    def deck(self):
        self.knockOver()

class Pellet(Character):
    def __init__(self, pelletId, tapId):
        Character.__init__(self, pelletId, pelletId)
        self.tapId = tapId

    def isTimeToMove(self):
        self.actionDelay = (self.actionDelay + 1) % 3
        return self.actionDelay == 0

    def getVictim(self):
        """Returns whoever has been hit by this pellet, if anyone."""
        return self.cast.getVictimAt(self.x, self.y)

    def blocked(self):
        """Returns whether this pellet is blocked by a wall."""
        if self.barrier:
            if self.x > self.barrier.x:
                return self.x == self.barrier.x
            return self.x == self.barrier.x - 1
        return False

    def isAirborne(self):
        """Returns whether this pellet is airborne."""
        return self.x >= 0

    def getTapId(self, lessonId):
        return self.tapId

class QAGenerator:
    def __init__(self):
        self.questions = []
        self.answers = {}
        self.qaPairs = {}

    def addQuestion(self, questionId, qaGroup, text):
        self.questions.append((questionId, qaGroup, text))

    def addAnswer(self, questionId, text):
        self.answers[questionId] = text

    def addQAPair(self, qaGroup, word1, word2):
        if not self.qaPairs.has_key(qaGroup):
            self.qaPairs[qaGroup] = []
        self.qaPairs[qaGroup].append((word1, word2))

    def expand(self, template, word1, word2):
        return template.replace('$1', word1).replace('$2', word2)

    def prepareQA(self):
        questionId, qaGroup, question = self.questions[random.randrange(len(self.questions))]
        answer = self.answers[questionId]
        word1, word2 = self.qaPairs[qaGroup][random.randrange(len(self.qaPairs[qaGroup]))]
        return self.expand(question, word1, word2), self.expand(answer, word1, word2)

class Cast:
    def __init__(self, surface):
        self.sprites = surface
        self.eric = None
        self.all = [] # Everything that must be drawn
        self.characters = {} # Humans
        self.characterList = [] # Ordered list of humans
        self.movables = [] # All computer-controlled things
        self.taps = {}

    def addEric(self, characterId, name, states, initialAS, flags):
        self.eric = Eric(characterId, name, flags)
        self.eric.setAnimatoryStates(*self.getAnimatoryStates(states))
        self.eric.initialiseAnimatoryState(initialAS)
        self.all.append(self.eric)
        self.characters[characterId] = self.eric
        self.characterList.append(self.eric)

    def addCharacter(self, characterId, name, states, initialAS, flags):
        character = Character(characterId, name, flags)
        character.setAnimatoryStates(*self.getAnimatoryStates(states))
        character.initialiseAnimatoryState(initialAS)
        self.all.append(character)
        self.characters[characterId] = character
        self.characterList.append(character)
        self.movables.append(character)

    def addPellet(self, characterId, states, tapId):
        pelletId = characterId + '-PELLET'
        pellet = Pellet(pelletId, tapId)
        pellet.setAnimatoryStates(*self.getAnimatoryStates(states))
        pellet.x = -3
        self.all.append(pellet)
        self.movables.append(pellet)
        self.get(characterId).setPellet(pellet)

    def getAnimatoryState(self, state):
        width = self.sprites.get_width() / 16
        height = self.sprites.get_height() / 8
        surface = self.sprites.subsurface((width * (state % 16), height * (state / 16)), (width, height))
        surface.set_colorkey((0, 255, 0))
        return surface

    def getAnimatoryStates(self, states):
        baseAS = states[0]
        asDictL = {}
        asDictR = {}
        for index in states[1:]:
            state = baseAS + index
            spriteL = self.getAnimatoryState(state)
            spriteR = pygame.transform.flip(spriteL, True, False)
            asDictL[index] = spriteL
            asDictR[index] = spriteR
        return asDictL, asDictR

    def initialise(self, game):
        for character in self.all:
            character.setGame(game)
            character.setCast(self)
            character.setSkool(game.skool)
            character.setTimetable(game.timetable)

    def addCommand(self, tapId, commandClass, *params):
        if not self.taps.has_key(tapId):
            self.taps[tapId] = CommandListTemplate(tapId)
        self.taps[tapId].addCommand(commandClass, *params)

    def setRandomLocations(self, characterId, locations):
        self.characters[characterId].setRandomLocations(locations)

    def addCommandList(self, characterId, lessonId, tapId):
        self.characters[characterId].addCommandList(lessonId, tapId)

    def setLocation(self, characterId, x, y):
        character = self.get(characterId)
        character.x, character.y = x, y

    def setSitDownMessage(self, characterId, message):
        self.characters[characterId].setSitDownMessage(message)

    def getEric(self):
        return self.eric

    def get(self, characterId):
        return self.characters.get(characterId, None)

    def addBlackboardMessage(self, characterId, message):
        self.characters[characterId].addBlackboardMessage(message)

    def move(self, pressedKeys, moveNow=False):
        for movable in self.movables:
            movable.move(moveNow)
        return self.eric.move(pressedKeys, moveNow)

    def draw(self, surface, column):
        for thing in self.all:
            thing.draw(surface, column)

    def setLesson(self, lessonId):
        for movable in self.movables:
            tapId = movable.getTapId(lessonId)
            movable.setCommandListTemplate(self.taps[tapId])
            movable.removeBubble()

    def getFacingCharacters(self, character, offset):
        """Returns a list of any characters facing the given character from a
        distance equal to offset."""
        targetX = character.x + offset * character.direction
        targetY = character.y
        targetDirection = -1 * character.direction
        facingCharacters = []
        for c in self.getPunchables():
            if (c.x, c.y, c.direction) == (targetX, targetY, targetDirection) and c.isHittable():
                facingCharacters.append(c)
        return facingCharacters

    def getPunchables(self):
        """Returns a list of the punchable characters."""
        return [c for c in self.characterList if c.isPunchable()]

    def getPelletables(self):
        """Returns a list of the pelletable characters."""
        return [c for c in self.characterList if c.isPelletable()]

    def getVictimAt(self, x, y):
        """Returns the most suitable character to hit (if any) with a catapult
        pellet at the given coordinates."""
        pelletables = self.getPelletables()
        for adult in [c for c in pelletables if c.isAdult()]:
            if (adult.x, adult.y) == (x, y):
                return adult
        for child in [c for c in pelletables if not c.isAdult()]:
            if (child.x, child.y) == (x, y):
                return child

    def getAdults(self):
        return [c for c in self.characterList if c.isAdult()]

    def getNearbyAdults(self, character):
        """Returns a list of adults who are close enough to the given character
        (and are facing the right way) to be able to see him."""
        x0 = character.x - 10
        x1 = character.x + 10
        y0 = character.y - 3
        y1 = character.y + 3
        nearbyAdults = []
        for c in self.getAdults():
            if x0 <= c.x <= x1 and y0 <= c.y <= y1 and c.direction * (character.x - c.x) >= 0:
                nearbyAdults.append(c)
        return nearbyAdults

    def expandNames(self, message):
        """Replace occurrences of $BLAH with the name of the character whose
        unique ID is 'BLAH'."""
        index = 0
        marker = '$'
        while message.find(marker, index) >=0:
            start = message.index(marker, index)
            end = start + 1
            while end < len(message) and message[end].isalnum():
                end += 1
            character = self.get(message[start + 1:end])
            if character:
                message = message.replace(message[start:end], character.name, 1)
            index = end
        return message

    def isHome(self, x):
        """Returns whether every character is on the 'home' side of the given
        x-coordinate."""
        for c in self.characterList:
            if not c.isHome(x):
                return False
        return True
