# Copyright 2008 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import random
import pygame

class Skool:
    def __init__(self, scale, skool, ink, paper):
        self.scale = scale
        self.skool = skool
        self.ink = ink
        self.paper = paper
        self.locations = {}
        self.rooms = {}
        self.doors = {}
        self.walls = []
        self.staircases = {}
        self.floors = {}
        self.routes = {}
        self.assemblyMessageGenerator = AssemblyMessageGenerator()

    def addLocation(self, locationId, coords):
        self.locations[locationId] = coords

    def addRoom(self, roomId, name, y, minX, maxX):
        self.rooms[roomId] = Room(roomId, name, y, minX, maxX)

    def addChair(self, roomId, x):
        self.rooms[roomId].addChair(x)

    def addDoor(self, doorId, x, bottomY, topY, shut, autoShuts):
        self.doors[doorId] = Door(doorId, x, bottomY, topY, shut, autoShuts)

    def setDoorImages(self, doorId, openImages, shutImages, coords):
        self.doors[doorId].setImages(openImages, shutImages, coords)

    def addWall(self, x, bottomY, topY):
        self.walls.append(Wall(x, bottomY, topY))

    def addStaircase(self, staircaseId, bottom, top, force, alias):
        staircase = Staircase(bottom, top, force)
        self.staircases[staircaseId] = staircase
        if alias:
            self.staircases[alias] = staircase

    def addFloor(self, floorId, minX, maxX, y):
        self.floors[floorId] = Floor(floorId, minX, maxX, y)

    def addRoutes(self, homeFloorId, destFloorIds, staircaseId):
        if not self.routes.has_key(homeFloorId):
            self.routes[homeFloorId] = {}
        for destFloorId in destFloorIds:
            self.routes[homeFloorId][destFloorId] = staircaseId

    def addBlackboard(self, roomId, x, y):
        self.rooms[roomId].addBlackboard(x, y)

    def getRoom(self, roomId):
        return self.rooms.get(roomId, None)

    def getDoor(self, doorId):
        return self.doors.get(doorId, None)

    def drawBlackboards(self, surface, column):
        for room in self.rooms.values():
            room.drawBlackboard(surface, column, self.scale)

    def room(self, character):
        for room in self.rooms.values():
            if room.contains(character):
                return room

    def staircase(self, character):
        for staircase in self.staircases.values():
            if staircase.contains(character):
                return staircase

    def nextStaircase(self, homeFloor, destFloor):
        if homeFloor is destFloor:
            return None
        routes = self.routes[homeFloor.floorId]
        staircaseId = routes.get(destFloor.floorId, routes['*'])
        return self.staircases[staircaseId]

    def floor(self, location):
        for floor in self.floors.values():
            if floor.supports(location):
                return floor

    def onFloor(self, character):
        for floor in self.floors.values():
            if floor.supports(character):
                return True
        return False

    def barrier(self, character):
        for barrier in self.walls + self.doors.values():
            if barrier.impedes(character):
                return barrier

    def chair(self, character):
        room = self.room(character)
        if room:
            return room.chair(character)

    def scaleCoords(self, coords):
        return (8 * self.scale * coords[0], 8 * self.scale * coords[1])

    def isDoorShut(self, doorId):
        return self.doors[doorId].isShut()

    def moveDoor(self, doorId, shut):
        images, coords = self.doors[doorId].move(shut)
        scaledCoords = self.scaleCoords(coords)
        if self.skool:
            self.skool.blit(images[0], scaledCoords)
        else:
            self.ink.blit(images[0], scaledCoords)
            self.paper.blit(images[1], scaledCoords)

    def autoShutDoors(self):
        for door in self.doors.values():
            if door.autoShut():
                self.moveDoor(door.doorId, True)

    def resolveLocationId(self, locationId):
        return Location(self.locations[locationId])

    def getWidth(self):
        surface = self.skool or self.ink
        return surface.get_width() / 8 / self.scale

    def draw(self, surface, column, cast):
        topLeft = (-8 * self.scale * column, 0)
        if self.skool:
            surface.blit(self.skool, topLeft)
            self.drawBlackboards(surface, column)
            cast.draw(surface, column)
        else:
            ink = pygame.Surface(self.scaleCoords((32, 21)))
            ink.blit(self.ink, topLeft)
            self.drawBlackboards(ink, column)
            cast.draw(ink, column)
            ink.set_colorkey((255, 255, 255))
            surface.blit(self.paper, topLeft)
            surface.blit(ink, (0, 0))

    def getAssemblyMessage(self):
        return self.assemblyMessageGenerator.generateMessage()

class Location:
    def __init__(self, coords):
        self.x = coords[0]
        self.y = coords[1]

    def __str__(self):
        return '(%s, %s)' % (self.x, self.y)

class Staircase:
    def __init__(self, bottom, top, force=False):
        self.bottom = Location(bottom)
        self.top = Location(top)
        # Whether character *must* go up or down staircase when at bottom or
        # top (instead of continuing left or right)
        self.force = force
        self.direction = 1 if self.bottom.x < self.top.x else -1

    def contains(self, character):
        """
        Returns whether the character is one of the following:
            i) on this staircase
           ii) at the bottom of this staircase facing the top
          iii) at the top of this staircase facing the bottom
        """
        if self.containsLocation(character.x, character.y):
            if character.x == self.bottom.x:
                return character.direction == self.direction
            if character.x == self.top.x:
                return character.direction != self.direction
            return True
        return False

    def containsLocation(self, x, y):
        """
        Returns whether the given location (x, y) is at the bottom, at the top,
        or on this staircase.
        """
        nx = self.direction * x
        if self.direction * self.bottom.x <= nx <= self.direction * self.top.x:
            return y == self.bottom.y - abs(x - self.bottom.x)
        return False

class Floor:
    def __init__(self, floorId, leftX, rightX, y):
        self.floorId = floorId
        self.leftX = leftX
        self.rightX = rightX
        self.y = y

    def supports(self, character):
        return self.y == character.y and self.leftX <= character.x <= self.rightX

class Barrier:
    def __init__(self, x, bottomY, topY):
        self.x = x
        self.bottomY = bottomY
        self.topY = topY

    def impedes(self, character):
        if not self.isShut():
            return False
        if self.bottomY >= character.y >= self.topY:
            return self.x - 2 <= character.x <= self.x
        return False

    def isOpenable(self):
        return False

    def isShut(self):
        return True

class Wall(Barrier):
    pass

class Door(Barrier):
    def __init__(self, doorId, x, bottomY, topY, shut, autoShuts=True):
        Barrier.__init__(self, x, bottomY, topY)
        self.doorId = doorId
        self.shut = shut
        self.autoShuts = autoShuts
        self.openImages = None
        self.shutImages = None
        self.topLeft = None
        self.autoShutTimer = 0
        self.autoShutDelay = 40

    def impedes(self, character):
        return self.shut and Barrier.impedes(self, character)

    def isShut(self):
        return self.shut

    def isOpenable(self):
        return True

    def move(self, shut):
        self.shut = shut
        if not shut:
            self.autoShutTimer = self.autoShutDelay
        images = self.shutImages if shut else self.openImages
        return (images, self.topLeft)

    def setImages(self, openImages, shutImages, topLeft):
        self.openImages = openImages
        self.shutImages = shutImages
        self.topLeft = topLeft

    def autoShut(self):
        if not self.shut and self.autoShuts:
            self.autoShutTimer -= 1
        return self.autoShutTimer < 0

class Room:
    def __init__(self, roomId, name, y, minX, maxX):
        self.roomId = roomId
        self.name = name
        self.y = y
        self.minX = minX
        self.maxX = maxX
        self.chairs = []
        self.blackboard = None

    def addBlackboard(self, x, y):
        self.blackboard = Blackboard(x, y)

    def drawBlackboard(self, surface, column, scale):
        if self.blackboard:
            self.blackboard.draw(surface, column, scale)

    def addChair(self, x):
        self.chairs.append(Chair(self, x))

    def contains(self, character):
        return character.y == self.y and self.minX <= character.x <= self.maxX

    def chair(self, character):
        for chair in self.chairs:
            if character.direction < 0 and (character.x, character.y) == (chair.x, chair.y):
                return chair
        return None

    def getNextChair(self, character, moveAlong):
        """Returns the chair nearest to the character, or the next chair he
        should sit in if he was dethroned (moveAlong=True)."""
        if moveAlong and self.chairs[0].x == character.x:
            return self.chairs[-1]
        minDistance = 100
        nextChair = None
        for chair in self.chairs:
            distance = character.x - chair.x
            if distance == 0 and moveAlong:
                continue
            if 0 <= distance < minDistance:
                minDistance = distance
                nextChair = chair
        return nextChair

class Chair:
    def __init__(self, room, x):
        self.x = x
        self.y = room.y
        self.room = room
        self.occupant = None

    def seat(self, character):
        self.occupant = character

    def vacate(self):
        self.occupant = None

class Blackboard:
    def __init__(self, x, y):
        self.x = x
        self.y = y
        self.clear()

    def write(self, images):
        self.text = images

    def draw(self, surface, column, scale):
        for lineNo in range(0, len(self.text)):
            coords = (scale * 8 * (self.x - column), scale * (8 * (self.y + lineNo) + 1))
            surface.blit(self.text[lineNo], coords)

    def clear(self):
        self.text = []

class Font:
    def __init__(self, image, offsets):
        self.image = image
        self.characterOffsets = offsets

    def render(self, words, ink, paper):
        characterImages = []
        totalWidth = 0
        height = self.image.get_height()
        scale = height / 8
        for c in words:
            offset, width = self.characterOffsets[c]
            image = self.image.subsurface((scale * offset, 0), (scale * width, height))
            characterImages.append(image)
            totalWidth += width
        text = pygame.Surface((scale * totalWidth, height))
        offset = 0
        for image in characterImages:
            text.blit(image, (offset, 0))
            offset += image.get_width()
        textInk = (0, 1, 2)
        textPaper = (255, 254, 253)
        paperSurface = pygame.Surface((text.get_width(), text.get_height()))
        paperSurface.fill(paper)
        inkSurface = pygame.Surface((text.get_width(), text.get_height()))
        inkSurface.fill(ink)
        text.set_colorkey(textInk)
        inkSurface.blit(text, (0, 0))
        inkSurface.set_colorkey(textPaper)
        paperSurface.blit(inkSurface, (0, 0))
        return paperSurface

class Timetable:
    def __init__(self, lessonLength, lessonStartTime):
        self.index = -1
        self.lessons = []
        self.lessonDetails = {}
        self.counter = 0
        self.lessonLength = lessonLength
        self.lessonStartTime = lessonStartTime

    def addLesson(self, lessonId):
        self.lessons.append(lessonId)

    def addLessonDetails(self, lessonId, teacherId, roomId):
        self.lessonDetails[lessonId] = (teacherId, roomId)

    def nextLesson(self):
        self.counter = self.lessonLength
        self.index = (self.index + 1) % len(self.lessons)

    def getLessonId(self):
        return self.lessons[self.index]

    def getLessonDetails(self):
        return self.lessonDetails[self.getLessonId()]

    def tick(self):
        self.counter -= 1
        return self.counter < 0

    def isTime(self, time):
        return self.counter + time < self.lessonLength

    def isTimeToStartLesson(self):
        return self.isTime(self.lessonStartTime)

    def isTeachingEric(self, character):
        lessonDetails = self.getLessonDetails()
        return len(lessonDetails) == 2 and character.characterId == lessonDetails[0]

class AssemblyMessageGenerator:
    def __init__(self):
        self.text = None
        self.verbs = []
        self.nouns = []

    def setText(self, text):
        self.text = text

    def addVerb(self, verb):
        self.verbs.append(verb)

    def addNoun(self, noun):
        self.nouns.append(noun)

    def generateMessage(self):
        verb = self.verbs[random.randrange(len(self.verbs))]
        noun = self.nouns[random.randrange(len(self.nouns))]
        message = self.text.replace('$VERB', verb)
        return message.replace('$NOUN', noun)
