# Copyright 2008 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import random
from skool import Location

class CommandListTemplate:
    def __init__(self, tapId):
        self.tapId = tapId
        self.commands = []

    def addCommand(self, commandClass, *params):
        self.commands.append((commandClass, params))

    def getCommands(self, start):
        return [cmdClass(*params) for cmdClass, params in self.commands][start:]

class CommandList:
    def __init__(self):
        self.stack = []
        self.index = None
        self.restartIndex = None
        self.template = None
        self.controllingCommand = None
        self.controllingCommandTimer = None

    def command(self, character):
        # If the controlling command is not already in control of the character,
        # give it a chance
        if self.controllingCommand and self.controllingCommand not in self.stack:
            self.stack.append(self.controllingCommand)
        while True:
            if self.stack:
                command = self.stack[-1]
                subcommand = command.command(character)
                if subcommand is None:
                    break
                elif subcommand is not command:
                    self.stack.append(subcommand)
                else:
                    self.stack.pop()
            else:
                # Remove any controlling command before moving to the next
                # command
                if self.controllingCommand:
                    self.controllingCommandTimer -= 1
                    if self.controllingCommandTimer < 0:
                        self.controllingCommand = None
                self.stack.append(self.commands[self.index])
                self.index += 1

    def restart(self, index=None):
        self.index = index or 0
        self.commands = self.template.getCommands(self.restartIndex)

    def addCommand(self, command):
        self.stack.append(command)

    def setTemplate(self, template):
        self.template = template
        while self.stack and self.stack[0].isInterruptible():
            self.stack.pop(0)
        self.restartIndex = 0
        self.restart()

    def setControllingCommand(self, command):
        self.controllingCommand = command
        self.controllingCommandTimer = 1

    def setRestartPoint(self):
        """Discard the current and all previous commands."""
        self.restartIndex = self.index
        self.restart()

    def jump(self, offset):
        """Jump forwards (or backwards) in the list of commands."""
        self.restart(self.index + offset)

    def isGoToing(self):
        """Returns whether the character is under the control of the GoTo
        command."""
        return self.stack and self.stack[0].__class__.__name__ == 'GoTo'

    def getGoToDestination(self):
        """Returns the destination of the character, or None if he is not under
        the control of the GoTo command."""
        if self.isGoToing():
            return self.stack[0].getDestination()

    def setGoToDestination(self, destination):
        """Sets the destination of the character if he is under the control of
        the GoTo command."""
        if destination and self.isGoToing():
            self.stack[0].setDestination(destination)

class signals:
    ASSEMBLY_FINISHED = 'AssemblyFinished'
    TIME_FOR_ASSEMBLY = 'TimeForAssembly'
    SWOT_READY = 'SwotReady'
    FINISHED_QUESTION = 'FinishedQuestion'
    FINISHED_ANSWER = 'FinishedAnswer'

class Command:
    def isInterruptible(self):
        return True

class ComplexCommand(Command):
    def __init__(self):
        self.index = 0
        self.commands = self.getCommands()

    def command(self, character):
        val = self.commands[self.index](character)
        if val is not False:
            self.index = (self.index + 1) % len(self.commands)
        return val or None

    def restart(self):
        self.index = -1

    def done(self, character):
        return self

class SitForAssembly(ComplexCommand):
    def getCommands(self):
        return (self.findSpotToSit, self.sitDown, self.getUp, self.done)

    def findSpotToSit(self, character):
        if character.gotSignal(signals.ASSEMBLY_FINISHED):
            return self
        return GoToXY(character.x - random.randrange(1, 5), character.y)

    def sitDown(self, character):
        if character.gotSignal(signals.ASSEMBLY_FINISHED):
            return self
        if character.direction < 0:
            character.turn()
            return False
        character.sit()

    def getUp(self, character):
        if character.gotSignal(signals.ASSEMBLY_FINISHED):
            character.getUp()
        else:
            return False

class MoveDoor(ComplexCommand):
    def __init__(self, doorId, shut):
        ComplexCommand.__init__(self)
        self.doorId = doorId
        self.shut = shut

    def getCommands(self):
        return (self.raiseArm, self.moveDoor, self.lowerArm, self.done)

    def raiseArm(self, character):
        if character.checkDoorStatus(self.doorId, self.shut):
            """Move on now if the door is already in the desired state."""
            return self
        character.raiseArm()

    def moveDoor(self, character):
        character.moveDoor(self.doorId, self.shut)

    def lowerArm(self, character):
        character.lowerArm()

class OpenDoor(MoveDoor):
    def __init__(self, doorId):
        MoveDoor.__init__(self, doorId, False)

class ShutDoor(MoveDoor):
    def __init__(self, doorId):
        MoveDoor.__init__(self, doorId, True)

class WipeBoard(ComplexCommand):
    def __init__(self):
        ComplexCommand.__init__(self)
        self.column = 7
        self.interrupted = False

    def getCommands(self):
        return (self.walk, self.wipe, self.lowerArm, self.walkBack, self.done)

    def walk(self, character):
        character.setWipingBoard(True)
        return GoToXY(character.x - 1, character.y)

    def wipe(self, character):
        character.raiseArm()
        character.wipeBoard(self.column)

    def lowerArm(self, character):
        self.column -= 1
        character.lowerArm()
        if self.column >= 0:
            # Repeat until the board is clean
            self.restart()

    def walkBack(self, character):
        if self.interrupted:
            # No need to walk to the middle of the board if we were interrupted
            return self
        character.setWipingBoard(False)
        return GoToXY(character.x + 3, character.y)

    def isInterruptible(self):
        self.interrupted = True
        return False

class ConductQASession(ComplexCommand):
    def getCommands(self):
        return (self.ask, self.signalAsked, self.waitForAnswer)

    def ask(self, character):
        return Say(character.prepareQA())

    def signalAsked(self, character):
        character.signal(signals.FINISHED_QUESTION)

    def waitForAnswer(self, character):
        if character.gotSignal(signals.FINISHED_ANSWER):
            character.unsignal(signals.FINISHED_ANSWER)
        else:
            return False

class GoTo(Command):
    def __init__(self, locationId):
        self.locationId = locationId
        self.destination = None

    def command(self, character):
        if self.destination is None:
            self.destination = character.resolveLocationId(self.locationId)
        if (character.x, character.y) == (self.destination.x, self.destination.y):
            return self
        if character.onStairs():
            character.walk(True)
            return
        if character.isSitting():
            character.getUp()
            return
        nextStaircase = character.getNextStaircase(self.destination)
        if nextStaircase:
            if character.y == nextStaircase.bottom.y:
                nextX = nextStaircase.bottom.x
            else:
                nextX = nextStaircase.top.x
        else:
            nextX = self.destination.x
        if character.x < nextX:
            return character.right()
        elif character.x > nextX:
            return character.left()
        else:
            if nextStaircase.contains(character):
                character.walk(True)
            else:
                character.turn()

    def getDestination(self):
        return self.destination

    def setDestination(self, destination):
        self.destination = destination

class GoToRandomLocation(GoTo):
    def __init__(self):
        self.destination = None

    def command(self, character):
        if self.destination is None:
            self.destination = Location(character.getRandomDestination())
        return GoTo.command(self, character)

class GoToXY(GoTo):
    def __init__(self, x, y):
        self.destination = Location((x, y))

class Restart(Command):
    def command(self, character):
        character.restartTable()
        return self

class FindSeat(Command):
    def __init__(self, moveAlong=False):
        self.moveAlong = moveAlong

    def command(self, character):
        nextChair = character.getNextChair(self.moveAlong)
        self.moveAlong = False
        if nextChair == character.chair():
            character.sit()
            return self
        if character.x == nextChair.x and character.direction > 0:
            character.turn()
            return
        return GoToXY(nextChair.x, nextChair.y)

class MoveAboutUntil(Command):
    def __init__(self, signal):
        self.signal = signal

    def command(self, character):
        if character.gotSignal(self.signal):
            return self
        return WalkAround(1)

class WalkAround(Command):
    def __init__(self, walkabouts):
        self.count = walkabouts
        self.origin = None

    def command(self, character):
        if self.count == 0:
            return self
        # Cannot move about an x-coordinate on a staircase, so get off the
        # stairs first
        if character.onStairs():
            character.walk(True)
            return
        if self.origin is None:
            self.origin = character.x
        if character.x == self.origin:
            return GoToXY(character.x - random.randrange(0, 8), character.y)
        self.count -= 1
        return GoToXY(self.origin, character.y)

class SitStill(Command):
    def command(self, character):
        if not character.isSittingOnChair():
            return FindSeat(True)

class GrassAndAnswerQuestions(Command):
    def __init__(self):
        self.waiting = True

    def command(self, character):
        if not character.isSittingOnChair():
            return FindSeat()
        character.signal(signals.SWOT_READY)
        if self.waiting:
            if character.gotSignal(signals.FINISHED_QUESTION):
                self.waiting = False
                character.unsignal(signals.FINISHED_QUESTION)
                return Say(character.getAnswer())
        else:
            character.signal(signals.FINISHED_ANSWER)
            self.waiting = True

class StartAssemblyIfReady(Command):
    def command(self, character):
        if character.gotSignal(signals.ASSEMBLY_FINISHED):
            return self
        elif character.gotSignal(signals.TIME_FOR_ASSEMBLY):
            return
        character.restartTable()
        return self

class AwaitAssemblyTime(Command):
    def command(self, character):
        if character.isTimeToStartLesson():
            character.signal(signals.TIME_FOR_ASSEMBLY)
        else:
            character.restartTable()
        return self

class ConductAssembly(Command):
    def __init__(self):
        self.spoken = False

    def command(self, character):
        if self.spoken:
            character.signal(signals.ASSEMBLY_FINISHED)
            return self
        self.spoken = True
        return Say(character.getAssemblyMessage())

class StartLessonIfReady(Command):
    def __init__(self, signal):
        self.signal = signal
        self.ready = False

    def command(self, character):
        if not self.ready:
            if character.isTimeToStartLesson():
                self.ready = True
                character.signal(self.signal)
                return TellKidsToSitDown()
            else:
                character.restartTable()
        return self

class TellKidsToSitDown(Command):
    def __init__(self):
        self.spoken = False

    def command(self, character):
        if self.spoken:
            return self
        self.spoken = True
        return Say(character.sitDownMessage)

class ConductClass(Command):
    def __init__(self):
        self.wipedBoard = False
        self.wroteOnBoard = False

    def command(self, character):
        if character.isTeachingEric() and not character.gotSignal(signals.SWOT_READY):
            return
        if character.room.blackboard:
            if not self.wipedBoard:
                self.wipedBoard = True
                return WipeBoard()
            if not self.wroteOnBoard:
                self.wroteOnBoard = True
                shouldWrite = 184 if character.isTeachingEric() else 160
                if random.randrange(0, 256) >= shouldWrite:
                    return WriteOnBoard(character.getBlackboardMessage())
        if character.isTeachingEric():
            return ConductClassWithEric()
        return TellClassWhatToDo()

class WriteOnBoard(Command):
    def __init__(self, message):
        self.message = message
        self.index = 1
        self.armDown = True

    def command(self, character):
        if character.room.blackboard:
            if character.writeOnBoard(self.message, self.index):
                character.lowerArm()
                return self
            if self.armDown:
                character.raiseArm()
            else:
                character.lowerArm()
            self.armDown = not self.armDown
        self.index += 1

class ConductClassWithEric(Command):
    def command(self, character):
        if random.randrange(256) < 232:
            return ConductQASession()
        return TellClassWhatToDo()

class TellClassWhatToDo(Command):
    def __init__(self):
        self.spoken = False

    def command(self, character):
        if self.spoken:
            return GoToXY(character.x - 3 * character.direction, character.y)
        self.spoken = True
        #if character.room.blackboard and character.room.blackboard.text:
        #    return Say('Write an essay with this title')
        return Say(character.getLessonMessage())

class Say(Command):
    def __init__(self, words):
        self.words = words
        self.shift = -4

    def command(self, character):
        if character.say(self.words, self.shift):
            character.removeBubble()
            return self
        self.shift += 1

class Signal(Command):
    def __init__(self, signal):
        self.signal = signal

    def command(self, character):
        character.signal(self.signal)
        return self

class Unsignal(Command):
    def __init__(self, signal):
        self.signal = signal

    def command(self, character):
        character.unsignal(self.signal)
        return self

class SetControllingCommand(Command):
    def __init__(self, commandName, *params):
        self.commandName = commandName
        self.params = params

    def command(self, character):
        commandClass = eval(self.commandName)
        character.setControllingCommand(commandClass(*self.params))
        return self

class HitOrFireNowAndThen(Command):
    def ready(self, character):
        if not character.isStanding():
            return False
        if character.onStairs():
            return False
        if character.getNearbyAdults():
            return False
        return random.randrange(100) > 50

    def command(self, character):
        if not self.ready(character):
            return self
        return self.getCommand()

class FireNowAndThen(HitOrFireNowAndThen):
    def ready(self, character):
        if not HitOrFireNowAndThen.ready(self, character):
            return False
        if character.x % 4 != 0:
            return False
        return not character.pellet.isAirborne()

    def getCommand(self):
        return FireCatapult()

class FireCatapult(ComplexCommand):
    def getCommands(self):
        return (self.aim, self.fire, self.lower, self.done)

    def aim(self, character):
        if character.isRaisingCatapult():
            character.fireCatapult()
        else:
            character.raiseCatapult()
            return False

    def fire(self, character):
        character.fire()

    def lower(self, character):
        if character.isLoweringCatapult():
            character.completeAction()
        else:
            character.lowerCatapult()
            return False

    def isInterruptible(self):
        return False

class Hide(Command):
    def command(self, pellet):
        pellet.x = -3
        return

class Fly(Command):
    def __init__(self, pelletRange, hitZone):
        self.count = pelletRange
        self.hitZone = hitZone
        self.xInc = None
        self.yInc = 0

    def command(self, pellet):
        if self.xInc is None:
            self.xInc = pellet.direction
        if pellet.blocked():
            return self
        if pellet.checkShields():
            return self
        if self.count < self.hitZone:
            victim = pellet.getVictim()
            if victim:
                if victim.isHittable():
                    victim.deck()
                    return self
                if victim.isAdult() and victim.isKnockedOver():
                    self.xInc = 0
                    self.yInc = -1
                    self.count = 4
        self.count -= 1
        if self.count < 0:
            return self
        pellet.x += self.xInc
        pellet.y += self.yInc

    def isInterruptible(self):
        return False

class HitNowAndThen(HitOrFireNowAndThen):
    def ready(self, character):
        return HitOrFireNowAndThen.ready(self, character) and self.inFrontOfVictim(character)

    def inFrontOfVictim(self, character):
        return len(character.getFacingCharacters(3)) > 0

    def getCommand(self):
        return Hit()

class Hit(ComplexCommand):
    def getCommands(self):
        return (self.aim, self.hit, self.lower, self.done)

    def aim(self, character):
        if character.isRaisingFist():
            character.punch()
        else:
            character.raiseFist()
            return False

    def hit(self, character):
        facingCharacters = character.getFacingCharacters(2)
        if facingCharacters:
            facingCharacters[0].deck()

    def lower(self, character):
        if character.isLoweringFist():
            character.completeAction()
        else:
            character.lowerFist()
            return False

    def isInterruptible(self):
        return False

class Floored(Command):
    def __init__(self, count):
        self.count = count

    def command(self, character):
        if character.isStanding():
            return self
        self.count -= 1
        if self.count < 0:
            character.standUp()

    def isInterruptible(self):
        return False

class KnockedOut(Floored):
    def __init__(self):
        Floored.__init__(self, 10)

class Dethroned(Floored):
    def __init__(self):
        Floored.__init__(self, 8)

class KnockedOver(Floored):
    def __init__(self):
        Floored.__init__(self, 15)

    def command(self, character):
        # TODO Make character reveal combination letter/number
        self.count -= 1
        if self.count < 0:
            character.standUp()
            return self
        if self.count == 7:
            character.reprimand()

class WriteOnBoardUnless(Command):
    def __init__(self, signal):
        self.signal = signal
        self.wroteOnBoard = False

    def command(self, character):
        if character.gotSignal(self.signal) or character.room.blackboard.text:
            return self
        if self.wroteOnBoard:
            return self
        self.wroteOnBoard = True
        return WriteOnBoard(character.getBlackboardMessage())

class SetRestartPoint(Command):
    def command(self, character):
        character.setRestartPoint()
        return self

class JumpIfShut(Command):
    def __init__(self, doorId, offset):
        self.doorId = doorId
        self.offset = offset

    def command(self, character):
        character.jumpIfShut(self.doorId, self.offset)
        return self

class JumpIfOpen(Command):
    def __init__(self, doorId, offset):
        self.doorId = doorId
        self.offset = offset

    def command(self, character):
        character.jumpIfOpen(self.doorId, self.offset)
        return self

class WalkFast(Command):
    def command(self, character):
        character.run(True)
        return self

class StalkAndHit(HitNowAndThen):
    def __init__(self, characterId):
        self.characterId = characterId

    def command(self, character):
        character.stalk(self.characterId)
        return HitNowAndThen.command(self, character)

class WaitAtDoor(Command):
    def __init__(self, doorId):
        self.doorId = doorId

    def command(self, character):
        if character.waitAtDoor(self.doorId):
            return self

class GiveLines(Command):
    def __init__(self, recipientId, messageId):
        self.recipientId = recipientId
        self.messageId = messageId

    def command(self, character):
        character.giveLines(self.recipientId, self.messageId)
        return self

class Jump(ComplexCommand):
    def getCommands(self):
        return (self.up, self.down, self.done)

    def up(self, character):
        character.jump()

    def down(self, character):
        character.descend()

#
# Unimplemented commands
#
class UnimplementedCommand(Command):
    def __init__(self, *params):
        self.terminal = False

    def command(self, character):
        return None if self.terminal else self

class FindEric(UnimplementedCommand):
    pass

class FindEricIfMissing(UnimplementedCommand):
    pass

class GiveEric2000Lines(UnimplementedCommand):
    pass

class TellEricToGoHome(UnimplementedCommand):
    pass

class TellEricAboutSwot(UnimplementedCommand):
    pass

class TellEricAboutBully(UnimplementedCommand):
    pass

class TellEricAboutTearaway(UnimplementedCommand):
    pass

class TripPeopleUp(UnimplementedCommand):
    pass

class FollowLittleBoy(UnimplementedCommand):
    pass

class CheckIfTouchingEric(UnimplementedCommand):
    pass

class WatchForEric(UnimplementedCommand):
    pass
