# Copyright 2008 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import random
import pygame
from pygame.locals import *
from skool import Location
from ai import *

class Character:
    def __init__(self, characterId, name, flags=''):
        self.characterId = characterId
        self.name = name
        self.flags = flags
        self.commandList = CommandList()
        self.x = 0
        self.y = 0
        self.direction = 1 # 1 = facing right, -1 = facing left
        self.verticalDirection = 0 # 1 = down, 0 = neither up nor down
        self.action = 0 # 0 = walking, 1 = still, 2 = writing, 3 = other
        self.walkStates = ['WALK0', 'WALK1', 'WALK2', 'WALK3']
        self.speed = 1 # 1 = walk, 2 = run
        self.speedChangeDelay = 1
        self.walkDelay = random.randrange(3, 6)
        self.actionDelay = 1
        self.staircase = None
        self.barrier = None
        self.room = None
        self.commandLists = {}
        self.sitDownMessage = None
        self.bubble = None
        self.blackboardMessages = []
        self.qaGenerator = None
        self.wipingBoard = False
        self.pellet = None
        self.linesMessages = {}
        self.lessonMessages = []

    def setGame(self, game):
        self.game = game

    def setCast(self, cast):
        self.cast = cast

    def setSkool(self, skool):
        self.skool = skool

    def setScreen(self, screen):
        self.screen = screen

    def setBeeper(self, beeper):
        self.beeper = beeper

    def setSitDownMessage(self, message):
        self.sitDownMessage = message

    def setAnimatoryStates(self, asDictL, asDictR):
        self.asDictL = asDictL
        self.asDictR = asDictR

    def initialiseAnimatoryState(self, initialAS):
        self.direction = -1 if initialAS[0].upper() == 'L' else 1
        self.animatoryState = self.walkStates[int(initialAS[1:])]

    def setControllingCommand(self, command):
        self.commandList.setControllingCommand(command)

    def isAdult(self):
        return 'A' in self.flags

    def isConkerable(self):
        return 'C' in self.flags

    def canOpenDoors(self):
        return 'D' in self.flags

    def isPunchable(self):
        return 'F' in self.flags

    def canGiveLines(self):
        return 'L' in self.flags

    def isScaredOfMice(self):
        return 'M' in self.flags

    def isPelletable(self):
        return 'P' in self.flags

    def canReceiveLines(self):
        return 'R' in self.flags

    def isATeacher(self):
        return 'T' in self.flags

    def sometimesRuns(self):
        return 'W' not in self.flags

    def isHome(self, x):
        if 'G' in self.flags:
            return self.x > x
        elif 'B' in self.flags:
            return self.x < x - 1
        return True

    def getImage(self):
        asDict = self.asDictR if self.direction > 0 else self.asDictL
        return (self.x, self.y, asDict[self.animatoryState])

    def getSpeechBubble(self):
        if self.bubble:
            bubbleX = 8 * ((self.x + 1) // 8)
            bubbleY = self.y - (3 if self.isAdult() else 2)
            return (bubbleX, bubbleY, self.bubble)
        return (0, 0, None)

    def setRandomLocations(self, locations):
        self.randomLocations = locations

    def getRandomDestination(self):
        return self.randomLocations[random.randrange(len(self.randomLocations))]

    def addCommandList(self, lessonId, tapId):
        self.commandLists[lessonId] = tapId

    def getTapId(self, lessonId):
        return self.commandLists[lessonId]

    def setCommandListTemplate(self, template):
        self.commandList.setTemplate(template)

    def restartTable(self):
        self.commandList.restart()

    def isTimeToMove(self):
        if self.bubble:
            self.actionDelay = (self.actionDelay + 1) % 3
            return self.actionDelay == 0
        if self.action in (1, 2):
            # Writing or not moving
            self.actionDelay = (self.actionDelay + 1) % 5
            return self.actionDelay == 0
        if self.action == 3:
            # Perform hitting, firing etc. quickly
            self.actionDelay = (self.actionDelay + 1) % 2
            return self.actionDelay == 0
        self.walkDelay -= 1
        if self.walkDelay > 0:
            return False
        self.speedChangeDelay -= 1
        if self.speedChangeDelay == 0:
            self.speedChangeDelay = random.randrange(16, 32)
            if self.sometimesRuns():
                self.run(self.speedChangeDelay & 1)
        self.walkDelay = 7 - 2 * self.speed
        return True

    def move(self, moveNow):
        if not (self.isTimeToMove() or moveNow):
            return
        if self.midstride():
            return self.walk()
        self.staircase = self.skool.staircase(self)
        self.barrier = self.skool.barrier(self)
        self.room = self.skool.room(self)
        self.commandList.command(self)

    def left(self):
        if self.direction > 0:
            self.turn()
        else:
            if self.barrier and self.barrier.x <= self.x:
                if self.canOpenDoors() and self.barrier.isOpenable():
                    return self.openDoor()
                return
            onStairs = False
            if self.staircase:
                if self.staircase.direction < 0:
                    onStairs = self.x != self.staircase.bottom.x or self.staircase.force
                else:
                    onStairs = self.x != self.staircase.top.x or self.staircase.force
            self.walk(onStairs)

    def right(self):
        if self.direction > 0:
            if self.barrier and self.x < self.barrier.x:
                if self.canOpenDoors() and self.barrier.isOpenable():
                    return self.openDoor()
                return
            onStairs = False
            if self.staircase:
                if self.staircase.direction > 0:
                    onStairs = self.x != self.staircase.bottom.x or self.staircase.force
                else:
                    onStairs = self.x != self.staircase.top.x or self.staircase.force
            self.walk(onStairs)
        else:
            self.turn()

    def up(self):
        if self.staircase:
            if self.direction == self.staircase.direction or (self.x == self.staircase.top.x and self.staircase.force):
                return self.walk(True)
            elif self.x != self.staircase.top.x:
                return self.turn()
        if self.direction > 0:
            self.right()
        else:
            self.left()

    def down(self):
        if self.staircase:
            if self.direction != self.staircase.direction or (self.x == self.staircase.bottom.x and self.staircase.force):
                return self.walk(True)
            elif self.x != self.staircase.bottom.x:
                return self.turn()
        if self.direction > 0:
            self.right()
        else:
            self.left()

    def turn(self):
        self.direction *= -1

    def getWalkStateIndex(self):
        if self.animatoryState in self.walkStates:
            return self.walkStates.index(self.animatoryState)
        return -1

    def walk(self, onStairs=False):
        if onStairs:
            if self.direction == self.staircase.direction:
                if not self.midstride():
                    self.y -= 1
            else:
                self.verticalDirection = 1
        walkState = (self.getWalkStateIndex() + 1) % 4
        if walkState % 2 == 0:
            self.x += self.direction
            self.y += self.verticalDirection
            if self.wipingBoard:
                # Animatory state sequence is 0, 1, 0, 1... when wiping board
                walkState = 0
            self.verticalDirection = 0
        self.animatoryState = self.walkStates[walkState]
        self.action = 0

    def chair(self):
        return self.skool.chair(self)

    def sit(self):
        if self.isSitting():
            self.getUp()
            return
        self.previousAS = self.animatoryState
        chair = self.chair()
        if chair:
            self.animatoryState = 'SITTING_ON_CHAIR'
            occupant = chair.occupant
            chair.seat(self)
            if occupant:
                occupant.dethrone()
        else:
            self.sitOnFloor()
        self.action = 1

    def standUp(self):
        self.animatoryState = 'WALK0'

    def dethrone(self):
        self.sitOnFloor()
        self.addCommand(Dethroned())

    def sitOnFloor(self):
        self.animatoryState = 'SITTING_ON_FLOOR'

    def raiseArm(self):
        self.animatoryState = 'ARM_UP'

    def lowerArm(self):
        self.animatoryState = 'WALK0'

    def knockOver(self):
        self.previousAS = self.animatoryState
        if self.isAdult():
            self.animatoryState = 'KNOCKED_OVER'
        else:
            self.animatoryState = 'KNOCKED_OUT'
        self.action = 1

    def raiseFist(self):
        self.animatoryState = 'HITTING0'
        self.action = 3

    def punch(self):
        self.animatoryState = 'HITTING1'

    def lowerFist(self):
        self.animatoryState = 'HITTING0'

    def raiseCatapult(self):
        self.animatoryState = 'CATAPULT0'
        self.action = 3

    def fireCatapult(self):
        self.animatoryState = 'CATAPULT1'

    def lowerCatapult(self):
        self.animatoryState = 'CATAPULT0'

    def completeAction(self):
        self.animatoryState = 'WALK0'
        self.action = 0

    def isStanding(self):
        return self.animatoryState in ('WALK0', 'WALK1', 'WALK2', 'WALK3')

    def isSitting(self):
        return self.isSittingOnFloor() or self.isSittingOnChair()

    def isSittingOnFloor(self):
        return self.animatoryState == 'SITTING_ON_FLOOR'

    def isSittingOnChair(self):
        return self.animatoryState == 'SITTING_ON_CHAIR'

    def isKnockedOver(self):
        return self.animatoryState == 'KNOCKED_OVER'

    def isKnockedOut(self):
        return self.animatoryState == 'KNOCKED_OUT'

    def isRaisingCatapult(self):
        return self.animatoryState == 'CATAPULT0'

    def isFiringCatapult(self):
        return self.animatoryState == 'CATAPULT1'

    def isLoweringCatapult(self):
        return self.animatoryState == 'CATAPULT0'

    def isRaisingFist(self):
        return self.animatoryState == 'HITTING0'

    def isPunching(self):
        return self.animatoryState == 'HITTING1'

    def isLoweringFist(self):
        return self.animatoryState == 'HITTING0'

    def isHittable(self):
        """Returns whether this character is in an animatory state that makes
        him punchable or vulnerable to a catapult pellet."""
        if self.isAdult():
            return self.isStanding() or self.animatoryState == 'ARM_UP'
        return self.isStanding() or self.isSittingOnChair()

    def getUp(self):
        if self.isSittingOnChair():
            self.chair().vacate()
        self.animatoryState = self.previousAS

    def midstride(self):
        return self.animatoryState in ('WALK1', 'WALK3')

    def getFloor(self, object=None):
        return self.skool.floor(object or self)

    def onStairs(self):
        if self.staircase:
            return self.x not in (self.staircase.bottom.x, self.staircase.top.x)
        return False

    def getNextStaircase(self, destination):
        return self.skool.nextStaircase(self.getFloor(), self.getFloor(destination))

    def openDoor(self):
        return OpenDoor(self.barrier.doorId)

    def moveDoor(self, doorId, shut):
        self.skool.moveDoor(doorId, shut)

    def getNextChair(self, moveAlong):
        return self.room.getNextChair(self, moveAlong)

    def signal(self, signal):
        self.game.signal(signal)

    def unsignal(self, signal):
        self.game.unsignal(signal)

    def gotSignal(self, signal):
        return self.game.gotSignal(signal)

    def isTimeToStartLesson(self):
        return self.skool.isTimeToStartLesson()

    def say(self, words, shift):
        lipPos = (self.x + 1) % 8
        self.bubble, done = self.screen.getBubble(words, lipPos, shift)
        return done

    def removeBubble(self):
        self.bubble = None

    def getAssemblyMessage(self):
        return self.skool.getAssemblyMessage()

    def setWipingBoard(self, wiping):
        self.wipingBoard = wiping

    def wipeBoard(self, column):
        self.skool.wipeBoard(self.room.blackboard, column)

    def writeOnBoard(self, message, index):
        self.action = 2
        return self.skool.writeOnBoard(self.room.blackboard, message, index)

    def resolveLocationId(self, locationId):
        return self.skool.resolveLocationId(locationId)

    def isTeachingEric(self):
        return self.skool.isTeachingEric(self)

    def prepareQA(self):
        question, answer = self.qaGenerator.prepareQA()
        self.game.setAnswer(answer)
        return question

    def getAnswer(self):
        return self.game.getAnswer()

    def getQAGenerator(self):
        if self.qaGenerator is None:
            self.qaGenerator = QAGenerator()
        return self.qaGenerator

    def addBlackboardMessage(self, message):
        self.blackboardMessages.append(message)

    def getBlackboardMessage(self):
        message = self.blackboardMessages[random.randrange(len(self.blackboardMessages))]
        return self.cast.expandNames(message)

    def run(self, run):
        self.speed = 2 if run else 1

    def getFacingCharacters(self, offset):
        """Returns a list of any characters facing this character from a
        distance equal to offset."""
        return self.cast.getFacingCharacters(self, offset)

    def addCommand(self, command):
        self.commandList.addCommand(command)

    def deck(self):
        self.knockOver()
        if self.isAdult():
            self.addCommand(KnockedOver())
        else:
            if self.previousAS == 'SITTING_ON_CHAIR':
                self.chair().vacate()
                self.addCommand(FindSeat())
            self.addCommand(KnockedOut())

    def setPellet(self, pellet):
        self.pellet = pellet

    def fire(self):
        """Launches a catapult pellet."""
        self.pellet.launch(self.x + self.direction, self.y, self.direction)

    def getNearbyAdults(self):
        """Returns a list of adults who are close enough to this character
        (and are facing the right way) to be able to see him."""
        return self.cast.getNearbyAdults(self)

    def setRestartPoint(self):
        """Discard the current and all previous commands in the command list (so
        any command that restarts the command list will see the next command as
        the first)."""
        self.commandList.setRestartPoint()

    def jumpIfShut(self, doorId, offset):
        """Jump forwards (or backwards) in the command list if the given door is
        shut."""
        if self.skool.isDoorShut(doorId):
            self.commandList.jump(offset)

    def jumpIfOpen(self, doorId, offset):
        """Jump forwards (or backwards) in the command list if the given door is
        open."""
        if not self.skool.isDoorShut(doorId):
            self.commandList.jump(offset)

    def checkDoorStatus(self, doorId, shut):
        return self.skool.isDoorShut(doorId) == shut

    def stalk(self, characterId):
        """Sets this character's destination equal to that of another
        character's destination (if they are both GoTo-ing)."""
        target = self.cast.get(characterId)
        if target:
            self.commandList.setGoToDestination(target.getGoToDestination())

    def getGoToDestination(self):
        """Returns the destination of this character, or None if they are not
        under the control of the GoTo command."""
        return self.commandList.getGoToDestination()

    def waitAtDoor(self, doorId):
        """Returns whether the characters are on the correct side of the given
        door."""
        return self.cast.isHome(self.skool.getDoor(doorId).x)

    def giveLines(self, recipientId, messageId):
        """Makes this character give lines."""
        recipient = self.cast.get(recipientId)
        admonition = self.linesMessages.get(messageId, None)
        if not (recipient and admonition):
            return
        bubbleY = self.y - 2 if self.isKnockedOver() else self.y - 3
        numLines = 10 * random.randrange(1, 9)
        message = ('%i0 LINES' % numLines, recipient.name)
        if recipient.isEric():
            paper = 2
            self.game.addLines(numLines)
        else:
            paper = 4
            self.game.addToScore(numLines)
        self.screen.printLinesBubble(self.x, bubbleY, message, paper)
        self.beeper.makeLinesSound(1)
        self.screen.printLinesBubble(self.x, bubbleY, admonition, paper)
        self.beeper.makeLinesSound(2)

    def isEric(self):
        return False

    def addLinesMessage(self, messageId, messageLines):
        if not self.linesMessages.has_key(messageId):
            self.linesMessages[messageId] = messageLines

    def addLessonMessage(self, message, condition):
        self.lessonMessages.append((message, condition))

    def lessonMessageCondition(self, condition):
        if condition == 'BoardDirty':
            return self.room and self.room.blackboard and self.room.blackboard.text
        return True

    def getLessonMessage(self):
        while True:
            message, condition = self.lessonMessages[random.randrange(len(self.lessonMessages))]
            if self.lessonMessageCondition(condition):
                break
        randNum = random.randrange(99, 1000)
        return message.replace('$n', str(randNum))

    def reprimand(self):
        """Make character give lines to the nearest lines recipient for knocking
        him over."""
        recipient = self.cast.getNearestLinesRecipient(self)
        if recipient:
            self.addCommand(GiveLines(recipient.characterId, 'NEVER_AGAIN'))

    def checkShieldsAt(self, x, y):
        if self.skool.hitShield(x, y):
            self.beeper.makeShieldSound()
            return True

    def hasLineOfSightTo(self, character):
        """Returns whether this character has a line of sight to the given
        character, i.e. there are no walls between them."""
        return self.skool.lineOfSightBetween(self, character)

class Eric(Character):
    def __init__(self, characterId, name, flags):
        Character.__init__(self, characterId, name, flags)
        self.controller = None

    def move(self, pressed_keys, moveNow):
        self.walkDelay -= 1
        if self.walkDelay > 0 and not moveNow:
            return 0
        self.walkDelay = 2
        if self.controller:
            if self.controller.command(self) is self.controller:
                self.controller = None
            return 0
        if self.midstride():
            self.makeWalkingSound(self.getWalkStateIndex())
            return self.walk()
        if self.isKnockedOut() or self.isSitting():
            if pressed_keys[K_s]:
                self.standUp()
                self.beeper.makeSittingSound()
            return 0
        if pressed_keys[K_s]:
            self.sit()
            self.beeper.makeSittingSound()
            return 0
        if self.isSitting():
            return 0
        if pressed_keys[K_f] and not self.pellet.isAirborne():
            self.controller = FireCatapult()
            return 0
        if pressed_keys[K_h]:
            self.controller = Hit()
            return 0
        if pressed_keys[K_j]:
            self.controller = Jump()
            return 0
        self.staircase = self.skool.staircase(self)
        if not (self.staircase or self.cast.isStandingOnKid(self) or self.skool.onFloor(self)):
            self.y += 1
            return 0
        self.barrier = self.skool.barrier(self)
        oldWS, oldAS, oldDirection = self.getWalkStateIndex(), self.animatoryState, self.direction
        if pressed_keys[K_LEFT]:
            self.left()
        elif pressed_keys[K_RIGHT]:
            self.right()
        elif pressed_keys[K_UP]:
            self.up()
        elif pressed_keys[K_DOWN]:
            self.down()
        if (self.animatoryState, self.direction) != (oldAS, oldDirection):
            self.makeWalkingSound(oldWS)
        return 0

    def makeWalkingSound(self, index):
        self.beeper.makeWalkingSound(index)

    def walk(self, onStairs=False):
        Character.walk(self, onStairs)
        if not self.midstride():
            return self.screen.getScrollIncrement(self.x)
        return 0

    def dethrone(self):
        self.sitOnFloor()
        self.beeper.makeKnockedOutSound()

    def deck(self):
        self.knockOver()
        self.beeper.makeKnockedOutSound()

    def alertLinesGivers(self, messageId):
        nearbyLinesGivers = self.cast.getNearbyLinesGivers(self)
        if nearbyLinesGivers:
            nearbyLinesGivers[0].addCommand(GiveLines(self.characterId, messageId))

    def fireCatapult(self):
        Character.fireCatapult(self)
        self.alertLinesGivers('NO_CATAPULTS')
        self.beeper.makeCatapultSound()

    def punch(self):
        Character.punch(self)
        self.alertLinesGivers('NO_HITTING')

    def jump(self):
        self.previousAS = self.animatoryState
        self.y -= 1
        self.animatoryState = 'ARM_UP'
        self.alertLinesGivers('NO_JUMPING')

    def descend(self):
        """Makes Eric finish his jump. There is no need to increment his
        y-coordinate here, since move() will do that (if need be) next time
        round."""
        self.checkShields()
        self.animatoryState = self.previousAS

    def checkShields(self):
        self.checkShieldsAt(self.x + self.direction + 1, self.y)

    def isEric(self):
        return True

class Pellet(Character):
    def __init__(self, pelletId, tapId, pelletRange, hitZone):
        Character.__init__(self, pelletId, pelletId)
        self.tapId = tapId
        self.pelletRange = pelletRange
        self.hitZone = hitZone
        self.animatoryState = 'FLY'

    def isTimeToMove(self):
        self.actionDelay = (self.actionDelay + 1) % 3
        return self.actionDelay == 0

    def getVictim(self):
        """Returns whoever has been hit by this pellet, if anyone."""
        return self.cast.getVictimAt(self.x, self.y)

    def blocked(self):
        """Returns whether this pellet is blocked by a wall."""
        if self.barrier:
            if self.x > self.barrier.x:
                return self.x == self.barrier.x
            return self.x == self.barrier.x - 1
        return False

    def isAirborne(self):
        """Returns whether this pellet is airborne."""
        return self.x >= 0

    def getTapId(self, lessonId):
        return self.tapId

    def checkShields(self):
        return self.checkShieldsAt(self.x + 1, self.y + 1)

    def launch(self, x, y, direction):
        self.x = x
        self.y = y
        self.direction = direction
        self.actionDelay = 1
        self.addCommand(Fly(self.pelletRange, self.hitZone))

class QAGenerator:
    def __init__(self):
        self.questions = []
        self.answers = {}
        self.qaPairs = {}

    def addQuestion(self, questionId, qaGroup, text):
        self.questions.append((questionId, qaGroup, text))

    def addAnswer(self, questionId, text):
        self.answers[questionId] = text

    def addQAPair(self, qaGroup, word1, word2):
        if not self.qaPairs.has_key(qaGroup):
            self.qaPairs[qaGroup] = []
        self.qaPairs[qaGroup].append((word1, word2))

    def expand(self, template, word1, word2):
        return template.replace('$1', word1).replace('$2', word2)

    def prepareQA(self):
        questionId, qaGroup, question = self.questions[random.randrange(len(self.questions))]
        answer = self.answers[questionId]
        word1, word2 = self.qaPairs[qaGroup][random.randrange(len(self.qaPairs[qaGroup]))]
        return self.expand(question, word1, word2), self.expand(answer, word1, word2)

class Cast:
    def __init__(self, surface):
        self.sprites = surface
        self.spriteGroups = {}
        self.eric = None
        self.all = []           # Everything that must be drawn
        self.characters = {}    # Humans
        self.characterList = [] # Ordered list of humans
        self.movables = []      # All computer-controlled things
        self.taps = {}

    def addSprite(self, groupId, spriteId, spriteIndex):
        if not self.spriteGroups.has_key(groupId):
            self.spriteGroups[groupId] = [{}, {}]
        sprite = self.getAnimatoryState(spriteIndex)
        self.spriteGroups[groupId][0][spriteId] = sprite
        self.spriteGroups[groupId][1][spriteId] = pygame.transform.flip(sprite, True, False)

    def addEric(self, characterId, name, spriteGroupId, initialAS, flags):
        self.eric = Eric(characterId, name, flags)
        self.eric.setAnimatoryStates(*self.spriteGroups[spriteGroupId])
        self.eric.initialiseAnimatoryState(initialAS)
        self.all.append(self.eric)
        self.characters[characterId] = self.eric
        self.characterList.append(self.eric)

    def addCharacter(self, characterId, name, spriteGroupId, initialAS, flags):
        character = Character(characterId, name, flags)
        character.setAnimatoryStates(*self.spriteGroups[spriteGroupId])
        character.initialiseAnimatoryState(initialAS)
        self.all.append(character)
        self.characters[characterId] = character
        self.characterList.append(character)
        self.movables.append(character)

    def addPellet(self, characterId, spriteGroupId, tapId, pelletRange, hitZone):
        pelletId = characterId + '-PELLET'
        pellet = Pellet(pelletId, tapId, pelletRange, hitZone)
        pellet.setAnimatoryStates(*self.spriteGroups[spriteGroupId])
        pellet.x = -3
        self.all.append(pellet)
        self.movables.append(pellet)
        self.get(characterId).setPellet(pellet)

    def getAnimatoryState(self, state):
        width = self.sprites.get_width() / 16
        height = self.sprites.get_height() / 8
        surface = self.sprites.subsurface((width * (state % 16), height * (state / 16)), (width, height))
        surface.set_colorkey((0, 255, 0))
        return surface

    def initialise(self, game):
        for character in self.all:
            character.setGame(game)
            character.setCast(self)
            character.setSkool(game.skool)
            character.setScreen(game.screen)
            character.setBeeper(game.skool.beeper)

    def addCommand(self, tapId, commandClass, *params):
        if not self.taps.has_key(tapId):
            self.taps[tapId] = CommandListTemplate(tapId)
        self.taps[tapId].addCommand(commandClass, *params)

    def setRandomLocations(self, characterId, locations):
        self.characters[characterId].setRandomLocations(locations)

    def addCommandList(self, characterId, lessonId, tapId):
        self.characters[characterId].addCommandList(lessonId, tapId)

    def setLocation(self, characterId, x, y):
        character = self.get(characterId)
        character.x, character.y = x, y

    def setSitDownMessage(self, characterId, message):
        self.characters[characterId].setSitDownMessage(message)

    def getEric(self):
        return self.eric

    def get(self, characterId):
        return self.characters.get(characterId, None)

    def addBlackboardMessage(self, characterId, message):
        self.characters[characterId].addBlackboardMessage(message)

    def addLinesMessage(self, characterId, messageId, message):
        messageLines = message.split('^')
        if characterId == '*':
            for c in self.getLinesGivers():
                c.addLinesMessage(messageId, messageLines)
        else:
            self.get(characterId).addLinesMessage(messageId, messageLines)

    def addLessonMessage(self, characterId, message, condition):
        if characterId == '*':
            for c in self.characterList:
                if c.isATeacher():
                    c.addLessonMessage(message, condition)
        else:
            self.get(characterId).addLessonMessage(message, condition)

    def move(self, pressedKeys, moveNow=False):
        for movable in self.movables:
            movable.move(moveNow)
        return self.eric.move(pressedKeys, moveNow)

    def getImages(self):
        return [thing.getImage() for thing in self.all]

    def getSpeechBubbles(self):
        return [character.getSpeechBubble() for character in self.characterList]

    def setLesson(self, lessonId):
        for movable in self.movables:
            tapId = movable.getTapId(lessonId)
            movable.setCommandListTemplate(self.taps[tapId])
            movable.removeBubble()

    def getFacingCharacters(self, character, offset):
        """Returns a list of any characters facing the given character from a
        distance equal to offset."""
        targetX = character.x + offset * character.direction
        targetY = character.y
        targetDirection = -1 * character.direction
        facingCharacters = []
        for c in self.getPunchables():
            if (c.x, c.y, c.direction) == (targetX, targetY, targetDirection) and c.isHittable():
                facingCharacters.append(c)
        return facingCharacters

    def getPunchables(self):
        """Returns a list of the punchable characters."""
        return [c for c in self.characterList if c.isPunchable()]

    def getPelletables(self):
        """Returns a list of the pelletable characters."""
        return [c for c in self.characterList if c.isPelletable()]

    def getVictimAt(self, x, y):
        """Returns the most suitable character to hit (if any) with a catapult
        pellet at the given coordinates."""
        pelletables = self.getPelletables()
        for adult in [c for c in pelletables if c.isAdult()]:
            if (adult.x, adult.y) == (x, y):
                return adult
        for child in [c for c in pelletables if not c.isAdult()]:
            if (child.x, child.y) == (x, y):
                return child

    def getAdults(self):
        return [c for c in self.characterList if c.isAdult()]

    def getLinesGivers(self):
        return [c for c in self.characterList if c.canGiveLines()]

    def getNearbyCharacters(self, character, candidates, witness):
        """Returns a list of characters from the given list of candidates who
        are close enough to the given character to be visible to him
        (regardless of the direction he's facing)."""
        x0 = character.x - 10
        x1 = character.x + 10
        y0 = character.y - 3
        y1 = character.y + 3
        nearbyCharacters = []
        for c in candidates:
            if x0 <= c.x <= x1 and y0 <= c.y <= y1 and c.hasLineOfSightTo(character):
                if not witness or c.direction * (character.x - c.x) >= 0:
                    nearbyCharacters.append(c)
        return nearbyCharacters

    def getWitnesses(self, character, candidates):
        """Returns a list of characters from the given list of candidates who
        are close enough to the given character (and are facing the right way)
        to be able to see him."""
        return self.getNearbyCharacters(character, candidates, True)

    def getNearbyAdults(self, character):
        """Returns a list of adults who are close enough to the given character
        (and are facing the right way) to be able to see him."""
        return self.getWitnesses(character, self.getAdults())

    def getNearbyLinesGivers(self, character):
        """Returns a list of lines-givers who are close enough to the given
        character (and are facing the right way) to be able to see him."""
        return self.getWitnesses(character, self.getLinesGivers())

    def getPotentialLinesRecipients(self, character):
        linesRecipients = [c for c in self.characterList if c.canReceiveLines()]
        return self.getNearbyCharacters(character, linesRecipients, False)

    def getNearestLinesRecipient(self, character):
        """Return the potential lines recipient nearest to the given
        character."""
        candidates = self.getPotentialLinesRecipients(character)
        if len(candidates) > 0:
            nearest = candidates[0]
            for c in candidates[1:]:
                if abs(c.x - character.x) < abs(nearest.x - character.x):
                    nearest = c
            return nearest

    def expandNames(self, message):
        """Replace occurrences of $BLAH with the name of the character whose
        unique ID is 'BLAH'."""
        index = 0
        marker = '$'
        while message.find(marker, index) >=0:
            start = message.index(marker, index)
            end = start + 1
            while end < len(message) and message[end].isalnum():
                end += 1
            character = self.get(message[start + 1:end])
            if character:
                message = message.replace(message[start:end], character.name, 1)
            index = end
        return message

    def isHome(self, x):
        """Returns whether every character is on the 'home' side of the given
        x-coordinate."""
        for c in self.characterList:
            if not c.isHome(x):
                return False
        return True

    def isStandingOnKid(self, character):
        """Returns whether the character is standing on a kid who's been knocked
        out."""
        for c in self.characterList:
            if c.isKnockedOut() and c.x - 1 <= character.x <= c.x + 1 and character.y == c.y - 1:
                return True
