# Copyright 2008 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import sys
import os
import pygame
from pygame.locals import *
from character import *
from skool import *
from ai import *
from graphics import Screen, Gallery, Font
from sound import Beeper
from skoolbuilder import SkoolBuilder

class Game:
    def __init__(self, scale, iniFile, quickStart):
        builder = SkoolBuilder()
        self.config = builder.getGameConfig(iniFile)
        scale = scale or self.config.get('scale', 2)
        imageDir = self.config['imagedir']
        imageSet = self.config.get('imageset', 'original')
        gallery = Gallery(imageDir, imageSet, scale)

        self.quickStart = quickStart if quickStart is not None else self.config.get('quickstart', False)

        mode = int(self.config.get('graphicsmode', 1))
        background = int(self.config.get('background', 0))
        self.screen = Screen(mode, scale, background, gallery)
        font = Font(gallery.getFont())
        self.screen.setFont(font)
        self.initialiseScreen()

        cast = Cast(gallery.getSprites())
        lessonLength = self.config.get('lessonlength', 1200)
        lessonStartTime = self.config.get('lessonstarttime', 600)
        timetable = Timetable(lessonLength, lessonStartTime)
        beeper = Beeper(self.config.get('soundsdir', 'sounds'))
        self.skool = Skool(self.screen, beeper, gallery, cast, timetable)
        builder.readIniFile(iniFile, self.skool)

        initialColumn = self.config.get('initialcolumn', None)
        self.screen.initialiseColumn(initialColumn, self.skool)
        self.skool.initialiseCast(self)
        self.score = self.lines = self.hiscore = 0
        self.clock = pygame.time.Clock()
        self.lastKeys = pygame.key.get_pressed()
        self.screenshot = 0

    def initialiseScreen(self):
        lessonBoxAttr = int(self.config['lessonboxattribute'])
        lessonBoxCoords = [int(c) for c in self.config['lessonboxpos'].split(',')]
        self.screen.setLessonBoxProperties(lessonBoxAttr, lessonBoxCoords)
        logoCoords = [int(c) for c in self.config['logopos'].split(',')]
        self.screen.addLogo(logoCoords)
        scoreBoxCoords = tuple(int(c) for c in self.config['scoreboxpos'].split(','))
        scoreBoxXOffset = int(self.config['scoreboxxoffset'])
        self.screen.addScoreBox(scoreBoxCoords, scoreBoxXOffset)

    def play(self):
        ringBell = False
        if not self.quickStart:
            ringBell = True
            self.skool.scrollOn(self.clock)

        while True:
            for event in pygame.event.get():
                if event.type == QUIT:
                    return

            if self.skool.tick():
                self.changeLesson(ringBell)
                ringBell = True

            pressed_keys = pygame.key.get_pressed()

            if pressed_keys[K_q]:
                return

            if pressed_keys[K_l] and not self.lastKeys[K_l]:
                self.changeLesson(False)
            if pressed_keys[K_x]:
                self.addLines(10 * random.randrange(1, 9))
            if pressed_keys[K_k] and not self.lastKeys[K_k]:
                self.takeScreenshot()

            # Hold down 'a' for turbo mode
            turboMode = pressed_keys[K_a]
            scroll = self.skool.moveCharacters(pressed_keys, turboMode)
            self.skool.autoShutDoors()

            if turboMode:
                self.skool.tick()
            else:
                self.clock.tick(20)

            self.lastKeys = pressed_keys

            self.skool.draw()
            self.screen.update()
            self.skool.scroll(scroll, self.clock)

    def takeScreenshot(self):
        self.screen.takeScreenshot('pyskool%s.bmp' % str(self.screenshot).zfill(3))
        self.screenshot += 1

    def changeLesson(self, ringBell=True):
        self.skool.nextLesson(ringBell)
        self.signals = {}

    def signal(self, signal):
        self.signals[signal] = True

    def unsignal(self, signal):
        self.signals[signal] = False

    def gotSignal(self, signal):
        return self.signals.get(signal, False)

    def addToScore(self, addend):
        self.score += addend
        self.screen.printNumber(self.score, 1)

    def addLines(self, addend):
        self.lines += addend
        self.screen.printNumber(self.lines, 9)

    def setHiScore(self):
        self.hiscore = self.score
        self.screen.printNumber(self.hiscore, 17)

    def getAnswer(self):
        return self.answer

    def setAnswer(self, answer):
        self.answer = answer
