import os
import pygame

class Screen:
    def __init__(self, mode, scale, background, gallery):
        self.initialiseColours()
        self.mode = mode
        self.scale = scale
        self.screen = pygame.display.set_mode((256 * self.scale, 192 * self.scale))
        self.screen.fill(self.colours[background])
        self.bubble = gallery.getSpeechBubble()
        self.scoreBox = gallery.getScoreBox()
        self.logo = gallery.getLogo()

    def scaleCoords(self, coords):
        return (8 * self.scale * coords[0], 8 * self.scale * coords[1])

    def initialiseColours(self):
        self.colours = []
        self.colours.append((0, 0, 0))       #  0 - black
        self.colours.append((0, 0, 197))     #  1 - blue
        self.colours.append((197, 0, 0))     #  2 - red
        self.colours.append((197, 0, 197))   #  3 - magenta
        self.colours.append((0, 198, 0))     #  4 - green
        self.colours.append((0, 198, 197))   #  5 - cyan
        self.colours.append((197, 198, 0))   #  6 - yellow
        self.colours.append((205, 198, 205)) #  7 - white
        self.colours.append((0, 0, 0))       #  8 - bright black
        self.colours.append((0, 0, 255))     #  9 - bright blue
        self.colours.append((255, 0, 0))     # 10 - bright red
        self.colours.append((255, 0, 255))   # 11 - bright magenta
        self.colours.append((0, 255, 0))     # 12 - bright green
        self.colours.append((0, 255, 255))   # 13 - bright cyan
        self.colours.append((255, 255, 0))   # 14 - bright yellow
        self.colours.append((255, 255, 255)) # 15 - bright white

    def initialiseColumn(self, initialColumn, skool):
        self.maxColumn = skool.getWidth() - 32
        ericX = skool.getEric().x
        defaultInitialColumn = min(self.maxColumn, max((ericX - 16) // 8 * 8, 0))
        column = min(self.maxColumn, defaultInitialColumn if initialColumn is None else initialColumn)
        self.column = max(0, 8 * (column // 8))

    def addScoreBox(self, coords, xOffset):
        self.scoreBoxCoords = coords
        self.scoreBoxXOffset = xOffset
        self.screen.blit(self.scoreBox, self.scaleCoords(self.scoreBoxCoords))

    def addLogo(self, coords):
        self.screen.blit(self.logo, self.scaleCoords(coords))

    def setLessonBoxProperties(self, attr, coords):
        self.lessonBoxAttr = attr
        self.lessonBoxPos = self.scaleCoords(coords)

    def setFont(self, font):
        self.font = font

    def getScrollIncrement(self, x):
        offset = x - self.column
        if offset > 21:
            return 1
        elif offset < 10:
            return -1
        return 0

    def scrollSkool(self, skool, clock):
        self.column -= 32
        background = pygame.Surface((self.screen.get_width(), 8 * self.scale * skool.getHeight()))
        for n in range(32):
            self.column += 1
            skool.draw()
            self.screen.blit(background, (-8 * (n + 1) * self.scale, 0))
            self.update()
            clock.tick(15)

    def scroll(self, inc, skool, clock):
        if self.column == self.maxColumn and inc > 0:
            return
        if self.column == 0 and inc < 0:
            return
        for i in range(0, 8):
            self.column += inc
            skool.draw()
            self.update()
            clock.tick(12)

    def getText(self, words, ink, paper):
        return self.font.render(words, ink, paper)

    def printLesson(self, teacher, room):
        lessonBox = pygame.Surface((self.scale * 64, self.scale * 24))
        paper = self.colours[(self.lessonBoxAttr & 120) / 8]
        lessonBox.fill(paper)
        ink = self.colours[(self.lessonBoxAttr & 64) / 8 + (self.lessonBoxAttr & 7)]
        teacherText = self.getText(teacher, ink, paper)
        roomText = self.getText(room, ink, paper)
        fontHeight = self.scale * 8
        teacherX = (lessonBox.get_width() - teacherText.get_width()) / 2
        teacherY = (lessonBox.get_height() - 2 * fontHeight) / 2
        lessonBox.blit(teacherText, (teacherX, teacherY))
        roomX = (lessonBox.get_width() - roomText.get_width()) / 2
        roomY = teacherY + fontHeight
        lessonBox.blit(roomText, (roomX, roomY))
        self.screen.blit(lessonBox, self.lessonBoxPos)

    def printNumber(self, number, yOffset):
        if number <= 0:
            return

        # Collect the ink and paper colour by inspecting pixels in the score
        # box; maybe these should be game config parameters?
        ink = self.scoreBox.get_at(((self.scoreBoxXOffset + 3) * self.scale, self.scale))
        paper = self.scoreBox.get_at((0, 0))

        numberText = self.getText(str(number), ink, paper)
        width = 24 * self.scale
        background = pygame.Surface((width, 7 * self.scale))
        background.fill(paper)
        background.blit(numberText, (width - numberText.get_width(), 0))
        coords = self.scaleCoords(self.scoreBoxCoords)
        topLeftX = coords[0] + self.scoreBoxXOffset * self.scale - width
        topLeftY = coords[1] + yOffset * self.scale
        self.screen.blit(background, (topLeftX, topLeftY))

    def printLinesBubble(self, x, y, message, paperColour):
        bubbleX = 8 * (x // 8) - self.column
        if bubbleX < 0 or bubbleX > 31:
            return
        bubbleCoords = self.scaleCoords((bubbleX, y))

        linesBubble = pygame.Surface(self.scaleCoords((8, 3)))
        ink = self.colours[7]
        paper = self.colours[paperColour]
        linesBubble.fill(paper)
        text1 = self.getText(message[0], ink, paper)
        text2 = self.getText(message[1], ink, paper)
        text1X = (linesBubble.get_width() - text1.get_width()) / 2
        text1Y = self.scale * 4
        text2X = (linesBubble.get_width() - text2.get_width()) / 2
        text2Y = text1Y + 8 * self.scale
        linesBubble.blit(text1, (text1X, text1Y))
        linesBubble.blit(text2, (text2X, text2Y))
        self.screen.blit(linesBubble, bubbleCoords)
        pygame.display.update()

    def getBubble(self, words, lipPos, shift):
        ink = self.colours[0]
        paper = self.colours[7]
        bubble = pygame.Surface(self.scaleCoords((8, 3)))
        colorkey = (0, 255, 0)
        bubble.fill(colorkey)
        bubble.set_colorkey(colorkey)
        bubble.blit(self.bubble, (0, 0))
        lip = self.bubble.subsurface(self.scaleCoords((8, 0)), self.scaleCoords((1, 1)))
        lipCoords = self.scaleCoords((lipPos, 2))
        bubble.blit(lip, lipCoords)
        openLipByte = 66
        for bit in range(0, 8 * self.scale, self.scale):
            colour = ink if openLipByte & 128 else paper
            for r in range(0, self.scale + 1):
                for c in range(0, self.scale + 1):
                    bubble.set_at((lipCoords[0] + bit + c, lipCoords[1] - 1 - r), colour)
            openLipByte *= 2
        text = self.getText(words, ink, paper)
        insetX = (4 - 8 * min(shift, 0)) * self.scale
        insetY = 4 * self.scale
        textX = max(shift, 0) * 8 * self.scale
        width = min(56 * self.scale, text.get_width() - textX)
        if insetX + width > 56 * self.scale:
            width = 56 * self.scale - insetX
        textWindow = text.subsurface((textX, 0), (width, 8 * self.scale))
        bubble.blit(textWindow, (insetX, insetY))
        return (bubble, textX > text.get_width())

    def update(self):
        pygame.display.update()

    def draw(self, skoolImages, blackboards, cast, others):
        if self.mode == 0:
            self.drawSkool(self.screen, skoolImages[0])
            for x, y, text in blackboards:
                self.drawBlackboard(self.screen, x, y, text)
            for x, y, sprite in cast:
                self.drawImage(self.screen, x, y, sprite)
        else:
            scratch = pygame.Surface(self.scaleCoords((32, 21)))
            self.drawSkool(scratch, skoolImages[1])
            for x, y, text in blackboards:
                self.drawBlackboard(scratch, x, y, text)
            for x, y, sprite in cast:
                self.drawImage(scratch, x, y, sprite)
            self.drawSkool(self.screen, skoolImages[2])
            scratch.set_colorkey((255, 255, 255))
            self.screen.blit(scratch, (0, 0))
        for x, y, image in others:
            self.drawImage(self.screen, x, y, image)

    def drawSkool(self, surface, skool):
        surface.blit(skool, self.scaleCoords((-self.column, 0)))

    def drawBlackboard(self, surface, x, y, text):
        for lineNo in range(len(text)):
            coords = (self.scale * 8 * (x - self.column), self.scale * (8 * (y + lineNo) + 1))
            surface.blit(text[lineNo], coords)

    def drawImage(self, surface, x, y, image):
        if image:
            surface.blit(image, self.scaleCoords((x - self.column, y)))

    def takeScreenshot(self, filename):
        pygame.image.save(self.screen, filename)

class Gallery:
    def __init__(self, imageDir, imageSet, scale):
        self.imageDir = imageDir
        self.imageSet = imageSet
        self.scale = scale

    def getImage(self, fname):
        scaleUp = True
        imageSetDir = os.path.join(self.imageDir, '%sx%i' % (self.imageSet, self.scale))
        if os.path.isdir(imageSetDir):
            scaleUp = False
        else:
            imageSetDir = os.path.join(self.imageDir, '%sx1' % self.imageSet)
        imageFile = os.path.join(imageSetDir, fname)
        if not os.path.exists(imageFile):
            return None
        img = pygame.image.load(imageFile).convert()
        if scaleUp:
            img = pygame.transform.scale(img, (self.scale * img.get_width(), self.scale * img.get_height()))
        return img

    def getSpeechBubble(self):
        return self.getImage('bubble.png')

    def getFont(self):
        return self.getImage('font.png')

    def getInventory(self):
        return self.getImage('inventory.png')

    def getLogo(self):
        return self.getImage('logo.png')

    def getMutables(self):
        return (self.getImage('mutables.png'), self.getImage('mutables_ink.png'), self.getImage('mutables_paper.png'))

    def getScoreBox(self):
        return self.getImage('scorebox.png')

    def getSkool(self, mode):
        if mode == 1:
            return (None, self.getImage('skool_ink.png'), self.getImage('skool_paper.png'))
        else:
            return (self.getImage('skool.png'), None, None)

    def getSprites(self):
        return self.getImage('sprites.png')

class Font:
    def __init__(self, image):
        self.image = image
        self.characterOffsets = {}

    def addCharacterOffset(self, char, offset, width):
        self.characterOffsets[char] = (offset, width)

    def render(self, words, ink, paper):
        characterImages = []
        totalWidth = 0
        height = self.image.get_height()
        scale = height / 8
        for c in words:
            offset, width = self.characterOffsets[c]
            image = self.image.subsurface((scale * offset, 0), (scale * width, height))
            characterImages.append(image)
            totalWidth += width
        text = pygame.Surface((scale * totalWidth, height))
        offset = 0
        for image in characterImages:
            text.blit(image, (offset, 0))
            offset += image.get_width()
        textInk = (0, 1, 2)
        textPaper = (255, 254, 253)
        paperSurface = pygame.Surface((text.get_width(), text.get_height()))
        paperSurface.fill(paper)
        inkSurface = pygame.Surface((text.get_width(), text.get_height()))
        inkSurface.fill(ink)
        text.set_colorkey(textInk)
        inkSurface.blit(text, (0, 0))
        inkSurface.set_colorkey(textPaper)
        paperSurface.blit(inkSurface, (0, 0))
        return paperSurface
