# Copyright 2008 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

from skool import *
from ai import *
from character import *

class SkoolBuilder:
    def getGameConfig(self, iniFile):
        config = {}
        f = open(iniFile, 'r')
        for line in f:
            if line[0] == '[' and line[1:].strip() != 'Game]':
                break
            if line.isspace():
                continue
            elif line.startswith(';'):
                continue
            elif line.startswith('[Game]'):
                continue
            else:
                elements = [e.strip() for e in line.strip().lower().split(':')]
                param = elements[0]
                value = elements[1]
                if value.isdigit():
                    value = int(value)
                config[param] = value
        f.close()
        return config

    def readIniFile(self, iniFile, skool):
        self.skool = skool
        self.cast = skool.cast
        self.timetable = skool.timetable
        self.font = skool.font
        self.beeper = skool.beeper
        f = open(iniFile, 'r')
        for line in f:
            if line.isspace():
                continue
            elif line.startswith(';'):
                continue
            elif line.startswith('[Game]'):
                addConfig = self.doNothing
                params = ()
            elif line.startswith('[Sounds]'):
                addConfig = self.addSound
                params = ()
            elif line.startswith('[SpriteGroup '):
                addConfig = self.addSprite
                groupId = line[13:line.index(']')].strip()
                params = (groupId,)
            elif line.startswith('[Eric]'):
                addConfig = self.addEric
                params = ()
            elif line.startswith('[Characters]'):
                addConfig = self.addCharacter
                params = ()
            elif line.startswith('[CatapultPellets]'):
                addConfig = self.addPellet
                params = ()
            elif line.startswith('[Timetable]'):
                addConfig = self.addLesson
                params = ()
            elif line.startswith('[Lesson '):
                addConfig = self.addCommandList
                details = line[8:line.index(']')]
                index = details.index(' ')
                lessonId = details[:index]
                lessonDetails = [e.strip() for e in details[index + 1:].split(',')]
                roomId = lessonDetails[-1]
                teacherId = lessonDetails[0] if len(lessonDetails) > 1 else ''
                self.timetable.addLessonDetails(lessonId, teacherId, roomId)
                params = (lessonId,)
            elif line.startswith('[RandomLocations]'):
                addConfig = self.addRandomLocations
                params = ()
            elif line.startswith('[SkoolLocations]'):
                addConfig = self.addSkoolLocation
                params = ()
            elif line.startswith('[CommandList '):
                addConfig = self.addCommand
                params = (line[13:line.index(']')],)
            elif line.startswith('[Rooms]'):
                addConfig = self.addRoom
                params = ()
            elif line.startswith('[Chairs]'):
                addConfig = self.addChairs
                params = ()
            elif line.startswith('[Doors]'):
                addConfig = self.addDoor
                params = ()
            elif line.startswith('[DoorImages]'):
                addConfig = self.addDoorImages
                params = ()
            elif line.startswith('[Walls]'):
                addConfig = self.addWall
                params = ()
            elif line.startswith('[Staircases]'):
                addConfig = self.addStaircase
                params = ()
            elif line.startswith('[Floors]'):
                addConfig = self.addFloor
                params = ()
            elif line.startswith('[Routes]'):
                addConfig = self.addRoute
                params = ()
            elif line.startswith('[SitDownMessages]'):
                addConfig = self.addSitDownMessage
                params = ()
            elif line.startswith('[CharacterWidths]'):
                addConfig = self.addCharacterOffset
                params = ()
            elif line.startswith('[AssemblyMessages]'):
                addConfig = self.addAssemblyMessage
                params = ()
            elif line.startswith('[Blackboards]'):
                addConfig = self.addBlackboard
                params = ()
            elif line.startswith('[BlackboardMessages]'):
                addConfig = self.addBlackboardMessage
                params = ()
            elif line.startswith('[QuestionsAndAnswers '):
                addConfig = self.addQA
                teacherId = line[21:line.index(']')].strip()
                params = (teacherId,)
            elif line.startswith('[LinesMessages]'):
                addConfig = self.addLinesMessage
                params = ()
            elif line.startswith('[LessonMessages]'):
                addConfig = self.addLessonMessage
                params = ()
            elif line.startswith('[Shields]'):
                addConfig = self.addShield
                params = ()
            else:
                addConfig(self.getElements(line.rstrip()), *params)
        f.close()

    def doNothing(self, details):
        return

    def addSound(self, elements):
        soundId = elements[0]
        soundFile = elements[1]
        self.beeper.addSound(soundId, soundFile)

    def addSprite(self, elements, groupId):
        spriteId = elements[0]
        spriteIndex = int(elements[1])
        self.cast.addSprite(groupId, spriteId, spriteIndex)

    def addEric(self, elements):
        characterId = elements[0]
        name = elements[1]
        spriteGroupId = elements[2]
        initialAS = elements[3]
        initialLocation = elements[4]
        flags = elements[5].upper()
        self.cast.addEric(characterId, name, spriteGroupId, initialAS, flags)
        self.cast.setLocation(characterId, *initialLocation)

    def addCharacter(self, elements):
        characterId = elements[0]
        name = elements[1]
        spriteGroupId = elements[2]
        initialAS = elements[3]
        initialLocation = elements[4]
        flags = elements[5].upper()
        self.cast.addCharacter(characterId, name, spriteGroupId, initialAS, flags)
        self.cast.setLocation(characterId, *initialLocation)

    def addPellet(self, elements):
        characterId = elements[0]
        spriteGroupId = elements[1]
        tapId = elements[2]
        pelletRange = int(elements[3])
        hitZone = int(elements[4])
        self.cast.addPellet(characterId, spriteGroupId, tapId, pelletRange, hitZone)

    def addLesson(self, elements):
        lessonType = elements[0]
        self.timetable.addLesson(lessonType)

    def addCommandList(self, elements, lessonId):
        characterId = elements[0]
        tapId = elements[1]
        self.cast.addCommandList(characterId, lessonId, tapId)

    def addRandomLocations(self, elements):
        characterId = elements[0]
        locations = elements[1:]
        self.cast.setRandomLocations(characterId, locations)

    def addSkoolLocation(self, elements):
        locationId = elements[0]
        x, y = int(elements[1]), int(elements[2])
        self.skool.addLocation(locationId, (x, y))

    def addCommand(self, elements, tapId):
        commandClass = eval(elements[0])
        params = []
        for e in elements[1:]:
            try:
                params.append(int(e))
            except ValueError:
                params.append(e)
        self.cast.addCommand(tapId, commandClass, *params)

    def addRoom(self, elements):
        roomId = elements[0]
        name = elements[1]
        y = int(elements[2])
        minX = int(elements[3])
        maxX = int(elements[4])
        self.skool.addRoom(roomId, name, y, minX, maxX)

    def addChairs(self, elements):
        roomId = elements[0]
        for x in elements[1:]:
            self.skool.addChair(roomId, int(x))

    def addDoor(self, elements):
        doorId = elements[0]
        x = int(elements[1])
        bottomY = int(elements[2])
        topY = int(elements[3])
        shut = elements[4][0].lower() == 's'
        autoShuts = len(elements) > 5
        self.skool.addDoor(doorId, x, bottomY, topY, shut, autoShuts)

    def addDoorImages(self, elements):
        doorId = elements[0]
        shutTopLeft = elements[1]
        size = elements[2]
        coords = elements[3]
        self.skool.addDoorImages(doorId, shutTopLeft, size, coords)

    def addWall(self, elements):
        x = int(elements[0])
        bottomY = int(elements[1])
        topY = int(elements[2])
        self.skool.addWall(x, bottomY, topY)

    def addStaircase(self, elements):
        staircaseIds = elements[0].partition(':')
        bottom = elements[1]
        top = elements[2]
        force = len(elements) == 4
        self.skool.addStaircase(staircaseIds[0], bottom, top, force, staircaseIds[2])

    def addFloor(self, elements):
        floorId = elements[0]
        minX = int(elements[1])
        maxX = int(elements[2])
        y = int(elements[3])
        self.skool.addFloor(floorId, minX, maxX, y)

    def addRoute(self, elements):
        homeFloorId = elements[0]
        staircaseId = elements[-1]
        destFloorIds = elements[1:-1]
        self.skool.addRoutes(homeFloorId, destFloorIds, staircaseId)

    def addSitDownMessage(self, elements):
        characterId = elements[0]
        message = elements[1]
        self.cast.setSitDownMessage(characterId, message)

    def addCharacterOffset(self, elements):
        char = elements[0]
        offset = int(elements[1])
        width = int(elements[2])
        self.font.addCharacterOffset(char, offset, width)

    def addAssemblyMessage(self, elements):
        generator = self.skool.assemblyMessageGenerator
        if elements[0] == 'MESSAGE':
            generator.setText(elements[1])
        elif elements[0] == 'VERB':
            generator.addVerb(elements[1])
        elif elements[0] == 'NOUN':
            generator.addNoun(elements[1])

    def addBlackboard(self, elements):
        roomId = elements[0]
        x, y = int(elements[1]), int(elements[2])
        self.skool.addBlackboard(roomId, x, y)

    def addBlackboardMessage(self, elements):
        characterId = elements[0]
        message = elements[1]
        self.cast.addBlackboardMessage(characterId, message)

    def addQA(self, elements, teacherId):
        qaGenerator = self.cast.get(teacherId).getQAGenerator()
        entryType = elements[0][:2].lower()
        if entryType == 'qu':
            questionId = elements[1]
            qaGroup = elements[2]
            text = elements[3]
            qaGenerator.addQuestion(questionId, qaGroup, text)
        elif entryType == 'an':
            questionId = elements[1]
            text = elements[2]
            qaGenerator.addAnswer(questionId, text)
        elif entryType == 'qa':
            qaGroup = elements[1]
            qaGenerator.addQAPair(qaGroup, elements[2], elements[3])

    def addLinesMessage(self, elements):
        characterId = elements[0]
        messageId = elements[1]
        message = elements[2]
        self.cast.addLinesMessage(characterId, messageId, message)

    def addLessonMessage(self, elements):
        characterId = elements[0]
        message = elements[1]
        condition = '' if len(elements) < 3 else elements[2]
        self.cast.addLessonMessage(characterId, message, condition)

    def addShield(self, elements):
        x = int(elements[0])
        y = int(elements[1])
        imageIndex = int(elements[2])
        self.skool.addShield(x, y, imageIndex)

    def getElements(self, details):
        elements = []
        index = 0
        while index < len(details):
            if details[index].isspace():
                index += 1
            elif details[index] in '"\'':
                quote = details[index]
                end = details.index(quote, index + 1)
                elements.append(details[index + 1:end])
                index = details.find(',', end + 1)
                if index < 0:
                    index = len(details)
                index += 1
            elif details[index] == '(':
                end = details.index(')', index)
                coords = details[index + 1:end].split(',')
                elements.append((int(coords[0]), int(coords[1])))
                index = details.find(',', end + 1)
                if index < 0:
                    index = len(details)
                index += 1
            else:
                end = details.find(',', index)
                if end < 0:
                    end = len(details)
                elements.append(details[index:end].strip())
                index = end + 1
        return elements
