# Copyright 2008 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

from skool import *
from ai import *
from character import *

class SkoolBuilder:
    def __init__(self, iniFile):
        self.sections = {}
        f = open(iniFile, 'r')
        section = None
        for line in f:
            if line[0] == '[':
                if section:
                    self.sections[sectionName] = section
                sectionName = line[1:line.index(']')].strip()
                section = []
            elif line.isspace():
                continue
            elif line.startswith(';'):
                continue
            else:
                section.append(line.strip())
        self.sections[sectionName] = section
        f.close()

    def buildSkool(self, skool):
        self.parseSounds(skool.beeper)
        self.parseSpriteGroups(skool.cast)
        self.parseCharacters(skool.cast)
        self.parseEric(skool.cast)
        self.parseCatapultPellets(skool.cast)
        self.parseTimetable(skool.timetable)
        self.parseLessons(skool.timetable, skool.cast)
        self.parseRandomLocations(skool.cast)
        self.parseSkoolLocations(skool)
        self.parseCommandLists(skool.cast)
        self.parseRooms(skool)
        self.parseChairs(skool)
        self.parseDoors(skool)
        self.parseDoorImages(skool)
        self.parseWalls(skool)
        self.parseStaircases(skool)
        self.parseFloors(skool)
        self.parseRoutes(skool)
        self.parseNoGoZones(skool)
        self.parseSitDownMessages(skool.cast)
        self.parseCharacterWidths(skool.font)
        self.parseAssemblyMessages(skool)
        self.parseBlackboards(skool)
        self.parseBlackboardMessages(skool.cast)
        self.parseQuestionsAndAnswers(skool.cast)
        self.parseLinesMessages(skool.cast)
        self.parseLessonMessages(skool.cast)
        self.parseShields(skool)
        self.parseSafe(skool)
        self.parseSkoolMessages(skool)

    def getElements(self, details, parseInts=True):
        elements = []
        index = 0
        while index < len(details):
            if details[index].isspace():
                index += 1
            elif details[index] in '"\'':
                quote = details[index]
                end = details.index(quote, index + 1)
                elements.append(details[index + 1:end])
                index = details.find(',', end + 1)
                if index < 0:
                    index = len(details)
                index += 1
            elif details[index] == '(':
                end = details.index(')', index)
                coords = details[index + 1:end].split(',')
                elements.append((int(coords[0]), int(coords[1])))
                index = details.find(',', end + 1)
                if index < 0:
                    index = len(details)
                index += 1
            else:
                end = details.find(',', index)
                if end < 0:
                    end = len(details)
                element = details[index:end].strip()
                if element.isdigit() and parseInts:
                    elements.append(int(element))
                else:
                    elements.append(element)
                index = end + 1
        return elements

    def parseSection(self, name, parseInts=True):
        if self.sections.has_key(name):
            return [self.getElements(line, parseInts) for line in self.sections[name]]
        return []

    def getSections(self, prefix, parseInts=True):
        sections = {}
        for name in self.sections:
            if name.startswith(prefix):
                sections[name[len(prefix):].strip()] = self.parseSection(name, parseInts)
        return sections

    def getConfig(self, name):
        config = {}
        for key, value in self.parseSection(name):
            config[key] = value
        return config

    def getGameConfig(self):
        return self.getConfig('Game')

    def getGalleryConfig(self):
        return self.getConfig('Images')

    def parseSounds(self, beeper):
        for soundId, soundFile in self.parseSection('Sounds'):
            beeper.addSound(soundId, soundFile)

    def parseSpriteGroups(self, cast):
        for groupId, section in self.getSections('SpriteGroup ').items():
            for spriteId, spriteIndex in section:
                cast.addSprite(groupId, spriteId, spriteIndex)

    def parseEric(self, cast):
        for characterId, name, spriteGroup, initialAS, location, flags in self.parseSection('Eric'):
            cast.addEric(characterId, name, spriteGroup, initialAS, flags.upper())
            cast.setLocation(characterId, *location)

    def parseCharacters(self, cast):
        for characterId, name, spriteGroup, initialAS, location, flags in self.parseSection('Characters'):
            cast.addCharacter(characterId, name, spriteGroup, initialAS, flags.upper())
            cast.setLocation(characterId, *location)

    def parseCatapultPellets(self, cast):
        for characterId, spriteGroup, tapId, pelletRange, hitZone in self.parseSection('CatapultPellets'):
            cast.addPellet(characterId, spriteGroup, tapId, int(pelletRange), int(hitZone))

    def parseTimetable(self, timetable):
        for lessonType in self.parseSection('Timetable'):
            timetable.addLesson(*lessonType)

    def parseLessons(self, timetable, cast):
        for details, section in self.getSections('Lesson ').items():
            index = details.index(' ')
            lessonId = details[:index]
            lessonDetails = [e.strip() for e in details[index + 1:].split(',')]
            roomId = lessonDetails[-1]
            teacherId = lessonDetails[0] if len(lessonDetails) > 1 else ''
            timetable.addLessonDetails(lessonId, teacherId, roomId)
            for characterId, tapId in section:
                cast.addCommandList(characterId, lessonId, tapId)

    def parseRandomLocations(self, cast):
        for elements in self.parseSection('RandomLocations'):
            characterId = elements[0]
            locations = elements[1:]
            cast.setRandomLocations(characterId, locations)

    def parseSkoolLocations(self, skool):
        for locationId, x, y in self.parseSection('SkoolLocations'):
            skool.addLocation(locationId, (int(x), int(y)))

    def parseCommandLists(self, cast):
        for tapId, section in self.getSections('CommandList ').items():
            for elements in section:
                commandClass = eval(elements[0])
                params = []
                for e in elements[1:]:
                    try:
                        params.append(int(e))
                    except ValueError:
                        params.append(e)
                cast.addCommand(tapId, commandClass, *params)

    def parseRooms(self, skool):
        for roomId, name, y, minX, maxX in self.parseSection('Rooms'):
            skool.addRoom(roomId, name, int(y), int(minX), int(maxX))

    def parseChairs(self, skool):
        for elements in self.parseSection('Chairs'):
            roomId = elements[0]
            for x in elements[1:]:
                skool.addChair(roomId, int(x))

    def parseDoors(self, skool):
        for elements in self.parseSection('Doors'):
            doorId = elements[0]
            x = int(elements[1])
            bottomY = int(elements[2])
            topY = int(elements[3])
            shut = elements[4][0].lower() == 's'
            autoShuts = len(elements) > 5
            skool.addDoor(doorId, x, bottomY, topY, shut, autoShuts)

    def parseDoorImages(self, skool):
        for doorId, shutTopLeft, size, coords in self.parseSection('DoorImages'):
            skool.addDoorImages(doorId, shutTopLeft, size, coords)

    def parseWalls(self, skool):
        for x, bottomY, topY in self.parseSection('Walls'):
            skool.addWall(int(x), int(bottomY), int(topY))

    def parseStaircases(self, skool):
        for elements in self.parseSection('Staircases'):
            staircaseIds = elements[0].partition(':')
            bottom = elements[1]
            top = elements[2]
            force = len(elements) == 4
            skool.addStaircase(staircaseIds[0], bottom, top, force, staircaseIds[2])

    def parseFloors(self, skool):
        for floorId, minX, maxX, y in self.parseSection('Floors'):
            skool.addFloor(floorId, int(minX), int(maxX), int(y))

    def parseRoutes(self, skool):
        for elements in self.parseSection('Routes'):
            homeFloorId = elements[0]
            staircaseId = elements[-1]
            destFloorIds = elements[1:-1]
            skool.addRoutes(homeFloorId, destFloorIds, staircaseId)

    def parseNoGoZones(self, skool):
        for zoneId, minX, maxX, bottomY, topY in self.parseSection('NoGoZones'):
            skool.addNoGoZone(zoneId, int(minX), int(maxX), int(bottomY), int(topY))

    def parseSitDownMessages(self, cast):
        for characterId, message in self.parseSection('SitDownMessages'):
            cast.setSitDownMessage(characterId, message)

    def parseCharacterWidths(self, font):
        for char, offset, width in self.parseSection('CharacterWidths'):
            font.addCharacterOffset(char, int(offset), int(width))

    def parseAssemblyMessages(self, skool):
        generator = skool.assemblyMessageGenerator
        for elements in self.parseSection('AssemblyMessages'):
            if elements[0] == 'MESSAGE':
                generator.setText(elements[1])
            elif elements[0] == 'VERB':
                generator.addVerb(elements[1])
            elif elements[0] == 'NOUN':
                generator.addNoun(elements[1])

    def parseBlackboards(self, skool):
        for roomId, x, y in self.parseSection('Blackboards'):
            skool.addBlackboard(roomId, int(x), int(y))

    def parseBlackboardMessages(self, cast):
        for characterId, message in self.parseSection('BlackboardMessages'):
            cast.addBlackboardMessage(characterId, message)

    def parseQuestionsAndAnswers(self, cast):
        for teacherId, section in self.getSections('QuestionsAndAnswers ', False).items():
            qaGenerator = cast.get(teacherId).getQAGenerator()
            for elements in section:
                entryType = elements[0]
                if entryType == 'SpecialGroup':
                    qaGenerator.setSpecialGroup(*elements[1:])
                elif entryType == 'SpecialQuestion':
                    qaGenerator.setSpecialQuestion(elements[1])
                elif entryType == 'SpecialAnswer':
                    qaGenerator.setSpecialAnswer(elements[1])
                elif entryType == 'Question':
                    questionId = elements[1]
                    qaGroup = elements[2]
                    text = elements[3]
                    qaGenerator.addQuestion(questionId, qaGroup, text)
                elif entryType == 'Answer':
                    questionId = elements[1]
                    text = elements[2]
                    qaGenerator.addAnswer(questionId, text)
                elif entryType == 'QAPair':
                    qaGroup = elements[1]
                    qaGenerator.addQAPair(qaGroup, elements[2], elements[3])

    def parseLinesMessages(self, cast):
        for characterId, messageId, message in self.parseSection('LinesMessages'):
            cast.addLinesMessage(characterId, messageId, message)

    def parseLessonMessages(self, cast):
        for elements in self.parseSection('LessonMessages'):
            characterId = elements[0]
            message = elements[1]
            condition = '' if len(elements) < 3 else elements[2]
            cast.addLessonMessage(characterId, message, condition)

    def parseShields(self, skool):
        for x, y, score, imageIndex in self.parseSection('Shields'):
            skool.addShield(x, y, score, imageIndex)

    def parseSafe(self, skool):
        for elements in self.parseSection('Safe'):
            x = elements[0]
            y = elements[1]
            score = elements[2]
            imageIndex = None if len(elements) < 4 else elements[3]
            skool.addSafe(x, y, score, imageIndex)

    def parseSkoolMessages(self, skool):
        for messageId, message in self.parseSection('SkoolMessages'):
            skool.addMessage(messageId, message)
