# Copyright 2008 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import random
import pygame
from character import Character
from eric import Eric
from pellet import Pellet
from ai import CommandListTemplate

class Cast:
    def __init__(self, sprites):
        self.sprites = sprites
        self.spriteGroups = {}
        self.eric = None
        self.all = []           # Everything that must be drawn
        self.characters = {}    # Humans
        self.characterList = [] # Ordered list of humans
        self.movables = []      # All computer-controlled things
        self.taps = {}

    def addSprite(self, groupId, spriteId, spriteIndex):
        if not self.spriteGroups.has_key(groupId):
            self.spriteGroups[groupId] = [{}, {}]
        sprite = self.getAnimatoryState(spriteIndex)
        self.spriteGroups[groupId][0][spriteId] = sprite
        self.spriteGroups[groupId][1][spriteId] = pygame.transform.flip(sprite, True, False)

    def addEric(self, characterId, name, spriteGroupId, initialAS, flags):
        self.eric = Eric(characterId, name, flags)
        self.eric.setAnimatoryStates(*self.spriteGroups[spriteGroupId])
        self.eric.initialiseAnimatoryState(initialAS)
        self.all.append(self.eric)
        self.characters[characterId] = self.eric
        self.characterList.append(self.eric)

    def addCharacter(self, characterId, nameAndTitle, spriteGroupId, initialAS, flags):
        names = nameAndTitle.partition('/')
        name, title = names[0], names[2]
        character = Character(characterId, name, flags)
        character.setTitle(title)
        character.setAnimatoryStates(*self.spriteGroups[spriteGroupId])
        character.initialiseAnimatoryState(initialAS)
        self.all.append(character)
        self.characters[characterId] = character
        self.characterList.append(character)
        self.movables.append(character)

    def addPellet(self, characterId, spriteGroupId, tapId, pelletRange, hitZone):
        pelletId = characterId + '-PELLET'
        pellet = Pellet(pelletId, tapId, pelletRange, hitZone)
        pellet.setAnimatoryStates(*self.spriteGroups[spriteGroupId])
        pellet.x = -3
        self.all.append(pellet)
        self.movables.append(pellet)
        self.get(characterId).setPellet(pellet)

    def getAnimatoryState(self, state):
        width = self.sprites.get_width() / 16
        height = self.sprites.get_height() / 8
        surface = self.sprites.subsurface((width * (state % 16), height * (state / 16)), (width, height))
        surface.set_colorkey((0, 255, 0))
        return surface

    def initialise(self, skool, screen, beeper):
        for character in self.all:
            character.setComponents(self, skool, screen, beeper)

    def reinitialise(self):
        for c in self.all:
            c.reinitialise()

    def initialiseSecrets(self):
        letters = ''
        for character in self.characterList:
            character.initialiseSpecialAnswer()
            if character.hasSecret():
                letters += character.initialiseSecret()
        return letters

    def addCommand(self, tapId, commandClass, *params):
        if not self.taps.has_key(tapId):
            self.taps[tapId] = CommandListTemplate(tapId)
        self.taps[tapId].addCommand(commandClass, *params)

    def setRandomLocations(self, characterId, locations):
        self.characters[characterId].setRandomLocations(locations)

    def addCommandList(self, characterId, lessonId, tapId):
        self.characters[characterId].addCommandList(lessonId, tapId)

    def setLocation(self, characterId, x, y):
        self.get(characterId).setInitialLocation(x, y)

    def setSitDownMessage(self, characterId, message):
        self.characters[characterId].setSitDownMessage(message)

    def getEric(self):
        return self.eric

    def get(self, characterId):
        return self.characters.get(characterId, None)

    def addBlackboardMessage(self, characterId, message):
        self.characters[characterId].addBlackboardMessage(message)

    def addLinesMessage(self, characterId, messageId, message):
        messageLines = message.split('^')
        if characterId == '*':
            for c in self.getLinesGivers():
                c.addLinesMessage(messageId, messageLines)
        else:
            self.get(characterId).addLinesMessage(messageId, messageLines)

    def addLessonMessage(self, characterId, message, condition):
        if characterId == '*':
            for c in self.characterList:
                if c.isATeacher():
                    c.addLessonMessage(message, condition)
        else:
            self.get(characterId).addLessonMessage(message, condition)

    def move(self, keyboard, moveNow):
        for movable in self.movables:
            movable.move(moveNow)
        return self.eric.move(keyboard, moveNow)

    def getImages(self):
        return [thing.getImage() for thing in self.all]

    def getSpeechBubbles(self):
        return [character.getSpeechBubble() for character in self.characterList]

    def setLesson(self, lessonId):
        for movable in self.movables:
            tapId = movable.getTapId(lessonId)
            movable.setCommandListTemplate(self.taps[tapId])
            movable.removeBubble()
        self.eric.unfreeze()

    def getFacingCharacters(self, character, offset):
        """Returns a list of any characters facing the given character from a
        distance equal to offset."""
        targetX = character.x + offset * character.direction
        targetY = character.y
        targetDirection = -1 * character.direction
        facingCharacters = []
        for c in self.getPunchables():
            if (c.x, c.y, c.direction) == (targetX, targetY, targetDirection) and c.isDeckable():
                facingCharacters.append(c)
        return facingCharacters

    def getPunchables(self):
        """Returns a list of the punchable characters."""
        return [c for c in self.characterList if c.isPunchable()]

    def getPelletables(self):
        """Returns a list of the pelletable characters."""
        return [c for c in self.characterList if c.isPelletable()]

    def getVictimAt(self, x, y):
        """Returns the most suitable character to hit (if any) with a catapult
        pellet at the given coordinates."""
        pelletables = self.getPelletables()
        for adult in [c for c in pelletables if c.isAdult()]:
            if (adult.x, adult.y) == (x, y):
                return adult
        for child in [c for c in pelletables if not c.isAdult()]:
            if (child.x, child.y) == (x, y):
                return child

    def getAdults(self):
        return [c for c in self.characterList if c.isAdult()]

    def getLinesGivers(self):
        return [c for c in self.characterList if c.canGiveLines()]

    def getNearbyCharacters(self, character, candidates, witness):
        """Returns a list of characters from the given list of candidates who
        are close enough to the given character to be visible to him
        (regardless of the direction he's facing)."""
        x0 = character.x - 10
        x1 = character.x + 10
        y0 = character.y - 3
        y1 = character.y + 3
        nearbyCharacters = []
        for c in candidates:
            if x0 <= c.x <= x1 and y0 <= c.y <= y1 and c.hasLineOfSightTo(character):
                if not witness or c.direction * (character.x - c.x) >= 0:
                    nearbyCharacters.append(c)
        return nearbyCharacters

    def getWitnesses(self, character, candidates):
        """Returns a list of characters from the given list of candidates who
        are close enough to the given character (and are facing the right way)
        to be able to see him."""
        return self.getNearbyCharacters(character, candidates, True)

    def getNearbyAdults(self, character):
        """Returns a list of adults who are close enough to the given character
        (and are facing the right way) to be able to see him."""
        return self.getWitnesses(character, self.getAdults())

    def getNearbyLinesGivers(self, character):
        """Returns a list of lines-givers who are close enough to the given
        character (and are facing the right way) to be able to see him."""
        return self.getWitnesses(character, self.getLinesGivers())

    def getPotentialLinesRecipients(self, character):
        linesRecipients = [c for c in self.characterList if c.canReceiveLines()]
        return self.getNearbyCharacters(character, linesRecipients, False)

    def getNearestLinesRecipient(self, character):
        """Return the potential lines recipient nearest to the given
        character."""
        candidates = self.getPotentialLinesRecipients(character)
        if len(candidates) > 0:
            nearest = candidates[0]
            for c in candidates[1:]:
                if abs(c.x - character.x) < abs(nearest.x - character.x):
                    nearest = c
            return nearest

    def expandNames(self, message):
        """Replace occurrences of $BLAH with the name of the character whose
        unique ID is 'BLAH'."""
        index = 0
        marker = '$'
        while message.find(marker, index) >=0:
            start = message.index(marker, index)
            end = start + 1
            while end < len(message) and message[end].isalnum():
                end += 1
            character = self.get(message[start + 1:end])
            if character:
                message = message.replace(message[start:end], character.name, 1)
            index = end
        return message

    def isHome(self, x):
        """Returns whether every character is on the 'home' side of the given
        x-coordinate."""
        for c in self.characterList:
            if not c.isHome(x):
                return False
        return True

    def isStandingOnKid(self, character):
        """Returns whether the character is standing on a kid who's been knocked
        out."""
        for c in self.characterList:
            if c.isKnockedOut() and c.x - 1 <= character.x <= c.x + 1 and character.y == c.y - 1:
                return True

    def somebodyNearDoor(self, door):
        """Returns True if somebody is standing near the door, False
        otherwise."""
        for c in self.characterList:
            if door.x - 2 <= c.x <= door.x + 1 and door.topY <= c.y <= door.bottomY:
                return True

    def isBesideEric(self, character):
        """Returns whether the character is beside Eric (and so need go no
        further to find him)."""
        ericX, ericY = self.getLocationOfEric()
        return abs(character.x - ericX) <= 4 and character.y == ericY

    def getLocationOfEric(self):
        """Returns the non-staircase location closest to Eric."""
        # TODO Return the top or bottom of the staircase if Eric is on one
        return (self.eric.x, 3 + 7 * (self.eric.y // 7))

    def expelEric(self):
        for c in self.characterList:
            if c.canExpelEric():
                c.findAndExpelEric()
                return

    def freezeEric(self):
        self.eric.freeze()

    def unfreezeEric(self):
        self.eric.unfreeze()

    def ericUnderstood(self):
        return self.eric.understoodMessage()

    def isTouchingEric(self, character):
        return (character.x, character.y) == (self.eric.x, self.eric.y)

    def isEricAbsent(self):
        return self.eric.isAbsent()

    def tripPeopleUpAt(self, x, y):
        # Assume trippability == pelletability
        for trippable in self.getPelletables():
            if (trippable.x, trippable.y) == (x, y) and trippable.isDeckable():
                trippable.deck()

    ############################################################################
    # Grass config
    ############################################################################
    def setHitters(self, hitters):
        self.hitters = hitters

    def setWriters(self, writers):
        self.writers = writers

    def setHitTale(self, hitTale):
        self.hitTale = hitTale

    def setWriteTale(self, writeTale):
        self.writeTale = writeTale

    def setAbsentTale(self, absentTale):
        self.absentTale = absentTale

    def expandTitle(self, message, character):
        return message.replace('$TITLE', character.getTitle())

    def getHitTale(self, teacher):
        hitterId = self.hitters[random.randrange(len(self.hitters))]
        message = self.expandNames(self.hitTale.replace('$1', '$%s' % hitterId))
        return hitterId, self.expandTitle(message, teacher)

    def getWriteTale(self, writerId, teacher):
        if writerId in self.writers:
            culprit = self.writers[random.randrange(len(self.writers))]
            message = self.expandNames(self.writeTale.replace('$1', '$%s' % writerId))
            return writerId, self.expandTitle(message, teacher)
        return None, None

    def getAbsentTale(self, teacher):
        return self.expandTitle(self.expandNames(self.absentTale), teacher)
