# Copyright 2008 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import pygame
import random

class Skool:
    def __init__(self, game, screen, beeper, cast, timetable, gallery):
        self.game = game
        self.screen = screen
        self.beeper = beeper
        self.scale = screen.scale
        self.mode = screen.mode
        self.skool, self.ink, self.paper = gallery.getSkool(self.mode)
        self.mutables, self.mutables_ink, self.mutables_paper = gallery.getMutables()
        self.font = screen.font
        self.cast = cast
        self.timetable = timetable
        self.scoreboard = Scoreboard(screen)
        self.locations = {}
        self.rooms = {}
        self.doors = {}
        self.walls = []
        self.staircases = {}
        self.floors = {}
        self.routes = {}
        self.noGoZones = []
        self.assemblyMessageGenerator = AssemblyMessageGenerator()
        self.shields = []
        self.shieldMode = 1
        self.safeCombination = None
        self.drawIndex = 0
        self.signals = Signals()
        self.messages = {}
        self.expelled = False

    def addLocation(self, locationId, coords):
        self.locations[locationId] = coords

    def addRoom(self, roomId, name, y, minX, maxX):
        self.rooms[roomId] = Room(roomId, name, y, minX, maxX)

    def addChair(self, roomId, x):
        self.rooms[roomId].addChair(x)

    def addDoor(self, doorId, x, bottomY, topY, shut, autoShuts):
        self.doors[doorId] = Door(doorId, x, bottomY, topY, shut, autoShuts)

    def addDoorImages(self, doorId, shutTopLeft, size, coords):
        size = self.scaleCoords(size)
        shutTopLeft = self.scaleCoords(shutTopLeft)
        openTopLeft = (shutTopLeft[0] + size[0], shutTopLeft[1])
        if self.mode == 0:
            shutImages = (self.mutables.subsurface(shutTopLeft, size),)
            openImages = (self.mutables.subsurface(openTopLeft, size),)
        else:
            shutImageInk = self.mutables_ink.subsurface(shutTopLeft, size)
            shutImagePaper = self.mutables_paper.subsurface(shutTopLeft, size)
            shutImages = (shutImageInk, shutImagePaper)
            openImageInk = self.mutables_ink.subsurface(openTopLeft, size)
            openImagePaper = self.mutables_paper.subsurface(openTopLeft, size)
            openImages = (openImageInk, openImagePaper)
        self.doors[doorId].setImages(openImages, shutImages, coords)

    def addWall(self, x, bottomY, topY):
        self.walls.append(Wall(x, bottomY, topY))

    def addStaircase(self, staircaseId, bottom, top, force, alias):
        staircase = Staircase(bottom, top, force)
        self.staircases[staircaseId] = staircase
        if alias:
            self.staircases[alias] = staircase

    def addFloor(self, floorId, minX, maxX, y):
        self.floors[floorId] = Floor(floorId, minX, maxX, y)

    def addRoutes(self, homeFloorId, destFloorIds, staircaseId):
        if not self.routes.has_key(homeFloorId):
            self.routes[homeFloorId] = {}
        for destFloorId in destFloorIds:
            self.routes[homeFloorId][destFloorId] = staircaseId

    def addBlackboard(self, roomId, x, y):
        self.rooms[roomId].addBlackboard(x, y, self.font, self.scale)

    def addNoGoZone(self, zoneId, minX, maxX, bottomY, topY):
        self.noGoZones.append(NoGoZone(zoneId, minX, maxX, bottomY, topY))

    def getFlashable(self, subclass, x, y, score, imageIndex):
        if imageIndex is not None:
            imageX = 16 * imageIndex * self.scale
            imageSize = self.scaleCoords((1, 1))
            image = self.mutables.subsurface((imageX, 0), imageSize)
            inverseImage = self.mutables.subsurface((imageX + 8 * self.scale, 0), imageSize)
            return subclass(x, y, score, image, inverseImage)
        return subclass(x, y, score)

    def addShield(self, x, y, score, imageIndex):
        shield = self.getFlashable(Shield, x, y, score, imageIndex)
        self.shields.append(shield)

    def addSafe(self, x, y, score, imageIndex):
        self.safe = self.getFlashable(Safe, x, y, score, imageIndex)

    def addMessage(self, messageId, message):
        self.messages[messageId] = message

    def allShieldsFlashed(self):
        for shield in self.shields:
            if not shield.isFlashing():
                return False
        return True

    def allShieldsUnflashed(self):
        for shield in self.shields:
            if shield.isFlashing():
                return False
        return True

    def hitShield(self, x, y):
        for shield in self.shields:
            if (shield.x, shield.y) == (x, y):
                hit = False
                if self.shieldMode == 3 and shield.isFlashing():
                    shield.unflash()
                    hit = True
                elif self.shieldMode == 1 and not shield.isFlashing():
                    shield.flash()
                    hit = True
                if hit:
                    self.addToScore(shield.getScore())
                    self.beeper.makeShieldSound()
                    if self.shieldMode == 1 and self.allShieldsFlashed():
                        self.shieldMode = 2
                        self.addToScore(200)
                        self.beeper.playAllShieldsTune()
                    elif self.shieldMode == 3 and self.allShieldsUnflashed():
                        self.shieldMode = 1
                        self.safe.unflash()
                        self.addToScore(500)
                        messageLines = self.messages['NEXT_YEAR'].split('^')
                        self.screen.printLesson(*messageLines)
                        self.initialiseSecrets()
                        self.beeper.playTune()
                        self.timetable.upAYear()
                return True

    def getBlackboards(self):
        return [room.blackboard for room in self.rooms.values() if room.blackboard]

    def gotCombination(self):
        for board in self.getBlackboards():
            if board.shows(self.safeCombination):
                return True

    def checkSafe(self, x, y):
        if self.shieldMode == 2 and (self.safe.x, self.safe.y) == (x, y) and self.gotCombination():
            self.shieldMode += 1
            self.safe.flash()
            self.addToScore(100)
            self.beeper.playAllShieldsTune()

    def canRevealSecret(self):
        return self.shieldMode == 2

    def getRoom(self, roomId):
        return self.rooms.get(roomId, None)

    def getDoor(self, doorId):
        return self.doors.get(doorId, None)

    def visibleBlackboard(self, character):
        """Returns the blackboard that is in the character's line of sight, or
        None if there is none."""
        for room in self.rooms.values():
            board = room.blackboard
            if board and room.y == character.y and character.isFacing(board) and self.lineOfSightBetween(character, board):
                return board

    def besideBlackboard(self, character):
        for room in self.rooms.values():
            if room.besideBlackboard(character):
                return True

    def room(self, character):
        for room in self.rooms.values():
            if room.contains(character):
                return room

    def staircase(self, character):
        for staircase in self.staircases.values():
            if staircase.contains(character):
                return staircase

    def nextStaircase(self, homeFloor, destFloor):
        if homeFloor is destFloor:
            return None
        routes = self.routes[homeFloor.floorId]
        staircaseId = routes.get(destFloor.floorId, routes['*'])
        return self.staircases[staircaseId]

    def floor(self, location):
        for floor in self.floors.values():
            if floor.supports(location):
                return floor

    def onFloor(self, character):
        for floor in self.floors.values():
            if floor.supports(character):
                return True
        return False

    def inNoGoZone(self, character):
        for zone in self.noGoZones:
            if zone.contains(character):
                return True

    def barrier(self, character):
        for barrier in self.walls + self.doors.values():
            if barrier.impedes(character):
                return barrier

    def chair(self, character):
        room = self.room(character)
        if room:
            return room.chair(character)

    def scaleCoords(self, coords):
        return (8 * self.scale * coords[0], 8 * self.scale * coords[1])

    def isDoorShut(self, doorId):
        return self.doors[doorId].isShut()

    def moveDoor(self, doorId, shut):
        images, coords = self.doors[doorId].move(shut)
        scaledCoords = self.scaleCoords(coords)
        if self.mode == 0:
            self.skool.blit(images[0], scaledCoords)
        else:
            self.ink.blit(images[0], scaledCoords)
            self.paper.blit(images[1], scaledCoords)

    def autoShutDoors(self):
        for door in self.doors.values():
            if door.autoShut() and not self.cast.somebodyNearDoor(door):
                self.moveDoor(door.doorId, True)

    def resolveLocationId(self, locationId):
        locationMarker = 'Location:'
        if locationId.startswith(locationMarker):
            character = self.getCharacter(locationId[len(locationMarker):])
            return Location((character.x, 3 + 7 * (character.y // 7)))
        return Location(self.locations[locationId])

    def getWidth(self):
        surface = self.skool or self.ink
        return surface.get_width() / 8 / self.scale

    def getHeight(self):
        surface = self.skool or self.ink
        return surface.get_height() / 8 / self.scale

    def draw(self):
        skoolImages = (self.skool, self.ink, self.paper)
        blackboardImages = [room.getBlackboardImages() for room in self.rooms.values()]
        castImages = self.cast.getImages()
        inverse = self.drawIndex > 4
        otherImages = [shield.getImages(inverse) for shield in self.shields]
        otherImages += [self.safe.getImages(inverse)]
        otherImages += self.cast.getSpeechBubbles()
        self.screen.draw(skoolImages, blackboardImages, castImages, otherImages)
        self.drawIndex = (self.drawIndex + 1) % 10

    def scrollOn(self, clock):
        self.screen.scrollSkool(self, clock)
        self.beeper.playTune()

    def scroll(self, inc, clock):
        if inc != 0:
            self.screen.scroll(inc, self, clock)

    def getAssemblyMessage(self):
        return self.assemblyMessageGenerator.generateMessage()

    def lineOfSightBetween(self, a, b):
        """Returns whether there is a clear line of sight between a and b (i.e.
        there are no walls between them."""
        for wall in self.walls:
            if wall.separates(a, b):
                return False
        return True

    def wipeBoard(self, blackboard, column):
        if blackboard is None:
            sys.stderr.write('Cannot wipe non-existent blackboard')
            return
        if column == 0:
            blackboard.clear()
            return
        wipedBit = pygame.Surface((8 * self.scale, 8 * self.scale))
        blackboard.wipe(wipedBit, 8 * column * self.scale)

    def writeOnBoard(self, character, blackboard, message, index=1):
        if blackboard is None:
            sys.stderr.write('Cannot write on non-existent blackboard')
            return
        blackboard.write(message[index - 1])
        blackboard.setWriter(character)
        return index == len(message)

    def initialiseSecrets(self):
        letters = self.cast.initialiseSecrets()
        if letters:
            self.safeCombination = letters[0]
            indexes = [n for n in range(1, len(letters))]
            while indexes:
                self.safeCombination += letters[indexes.pop(random.randrange(len(indexes)))]

    def initialiseCast(self):
        self.cast.initialise(self, self.screen, self.beeper)
        self.initialiseSecrets()

    def reinitialise(self):
        self.expelled = False
        self.screen.reinitialise()
        self.signals.clear()
        self.cast.reinitialise()
        self.initialiseSecrets()
        self.timetable.reinitialise()
        self.scoreboard.reinitialise()
        for room in self.rooms.values():
            room.wipeBlackboard()

    def getEric(self):
        return self.cast.getEric()

    def getCharacter(self, characterId):
        return self.cast.get(characterId)

    def moveCharacters(self, keyboard, turboMode):
        return self.cast.move(keyboard, turboMode)

    def tick(self):
        return self.timetable.tick()

    def isTimeToStartLesson(self):
        return self.timetable.isTimeToStartLesson()

    def isTeachingEric(self, character):
        return self.timetable.isTeachingEric(character)

    def getTeacher(self):
        """Returns Eric's teacher (or None if it's Playtime or Revision
        Library)."""
        return self.cast.get(self.timetable.getTeacherId())

    def setHomeRoom(self):
        """Sets the room Eric's supposed to be in at the moment."""
        self.homeRoom = self.getRoom(self.timetable.getRoomId())

    def getHomeRoom(self):
        """Returns the room Eric's supposed to be in at the moment."""
        return self.homeRoom

    def shouldGetAlong(self, eric):
        """Returns whether Eric is somewhere other than he should be."""
        if self.homeRoom:
            return not self.homeRoom.contains(eric)
        destination = self.getRoom(self.timetable.getRoomId())
        dinnerHall = self.rooms.get('DinnerHall', None)
        if self.timetable.isTime(200):
            # Give Eric time to leave a classroom
            return self.room(eric) not in (None, dinnerHall, destination)

    def getLessonDesc(self):
        teacherId = self.timetable.getTeacherId()
        if self.timetable.hideTeacher():
            teacherId = ''
        roomId = self.timetable.getRoomId()
        room = self.getRoom(roomId)
        teacher = self.getCharacter(teacherId)
        return teacher.name if teacher else teacherId, room.name if room else roomId

    def printLesson(self):
        teacher, room = self.getLessonDesc()
        if not teacher:
            elements = room.split(' ')
            if len(elements) > 1:
                teacher = elements[0]
                room = elements[1]
        self.screen.printLesson(teacher, room)

    def nextLesson(self, ringBell):
        self.timetable.nextLesson()
        self.cast.setLesson(self.timetable.getLessonId())
        self.printLesson()
        self.signals.clear()
        self.homeRoom = None
        if ringBell:
            self.beeper.ringBell()

    def setLesson(self, lesson):
        self.lesson = lesson

    def getLesson(self):
        return self.lesson

    def addToScore(self, addend):
        self.scoreboard.addToScore(addend)

    def addLines(self, addend):
        if not self.expelled:
            self.scoreboard.addLines(addend)
            if self.scoreboard.getLines() > self.game.getMaxLines():
                self.expelled = True
                self.stopClock()
                self.cast.expelEric()

    def isEricExpelled(self):
        return self.expelled

    def stopClock(self):
        self.timetable.stop()

    def startClock(self, ticks):
        self.timetable.resume(ticks)

    def getTooManyLinesMessage(self):
        return self.cast.expandNames(self.messages['TOO_MANY_LINES'])

    def getEricHasMumpsMessage(self):
        return self.cast.expandNames(self.messages['ERIC_HAS_MUMPS'])

    def getUnderstandMessage(self):
        return self.messages['UNDERSTOOD']

    def signal(self, signal):
        self.signals.signal(signal)

    def unsignal(self, signal):
        self.signals.unsignal(signal)

    def gotSignal(self, signal):
        return self.signals.isRaised(signal)

    def endGame(self):
        self.game.end()

class Location:
    def __init__(self, coords):
        self.x = coords[0]
        self.y = coords[1]

    def __str__(self):
        return '(%s, %s)' % (self.x, self.y)

class Staircase:
    def __init__(self, bottom, top, force=False):
        self.bottom = Location(bottom)
        self.top = Location(top)
        # Whether character *must* go up or down staircase when at bottom or
        # top (instead of continuing left or right)
        self.force = force
        self.direction = 1 if self.bottom.x < self.top.x else -1

    def contains(self, character):
        """
        Returns whether the character is one of the following:
            i) on this staircase
           ii) at the bottom of this staircase facing the top
          iii) at the top of this staircase facing the bottom
        """
        if self.containsLocation(character.x, character.y):
            if character.x == self.bottom.x:
                return character.direction == self.direction
            if character.x == self.top.x:
                return character.direction != self.direction
            return True
        return False

    def containsLocation(self, x, y):
        """
        Returns whether the given location (x, y) is at the bottom, at the top,
        or on this staircase.
        """
        nx = self.direction * x
        if self.direction * self.bottom.x <= nx <= self.direction * self.top.x:
            return y == self.bottom.y - abs(x - self.bottom.x)
        return False

    def supports(self, character):
        """Returns whether the character is on a step of this staircase."""
        if not self.containsLocation(character.x, character.y):
            return False
        return character.x not in (self.bottom.x, self.top.x)

class Floor:
    def __init__(self, floorId, leftX, rightX, y):
        self.floorId = floorId
        self.leftX = leftX
        self.rightX = rightX
        self.y = y

    def supports(self, character):
        return self.y == character.y and self.leftX <= character.x <= self.rightX

class Barrier:
    def __init__(self, x, bottomY, topY):
        self.x = x
        self.bottomY = bottomY
        self.topY = topY

    def impedes(self, character):
        if not self.isShut():
            return False
        if self.bottomY >= character.y >= self.topY:
            return self.x - 2 <= character.x <= self.x
        return False

    def isOpenable(self):
        return False

    def isShut(self):
        return True

class Wall(Barrier):
    def separates(self, a, b):
        """Returns whether this wall blocks the view from a to b."""
        minX = min(a.x, b.x)
        maxX = max(a.x, b.x)
        if minX < self.x <= maxX:
            return self.topY <= max(a.y, b.y) and self.bottomY >= min(a.y, b.y)

class Door(Barrier):
    def __init__(self, doorId, x, bottomY, topY, shut, autoShuts=True):
        Barrier.__init__(self, x, bottomY, topY)
        self.doorId = doorId
        self.shut = shut
        self.autoShuts = autoShuts
        self.openImages = None
        self.shutImages = None
        self.topLeft = None
        self.autoShutTimer = 0
        self.autoShutDelay = 40

    def impedes(self, character):
        return self.shut and Barrier.impedes(self, character)

    def isShut(self):
        return self.shut

    def isOpenable(self):
        return True

    def move(self, shut):
        self.shut = shut
        if not shut:
            self.autoShutTimer = self.autoShutDelay
        images = self.shutImages if shut else self.openImages
        return (images, self.topLeft)

    def setImages(self, openImages, shutImages, topLeft):
        self.openImages = openImages
        self.shutImages = shutImages
        self.topLeft = topLeft

    def autoShut(self):
        if not self.shut and self.autoShuts:
            self.autoShutTimer -= 1
        return self.autoShutTimer < 0

class Room:
    def __init__(self, roomId, name, y, minX, maxX):
        self.roomId = roomId
        self.name = name
        self.y = y
        self.minX = minX
        self.maxX = maxX
        self.chairs = []
        self.blackboard = None

    def getBlackboard(self):
        return self.blackboard

    def getBlackboardWriter(self):
        if self.blackboard:
            return self.blackboard.getWriter()

    def hasBlackboard(self):
        return self.blackboard is not None

    def addBlackboard(self, x, y, font, scale):
        self.blackboard = Blackboard(x, y, font, scale)

    def blackboardDirty(self):
        if self.blackboard:
            return self.blackboard.isDirty()

    def wipeBlackboard(self):
        if self.blackboard:
            self.blackboard.clear()

    def getBlackboardImages(self):
        if self.blackboard:
            return self.blackboard.getImages()
        return (0, 0, [])

    def addChair(self, x):
        self.chairs.append(Chair(self, x))

    def besideBlackboard(self, character):
        if self.blackboard:
            return self.y == character.y and self.blackboard.beside(character)

    def contains(self, character):
        return self.y - 3 <= character.y <= self.y and self.minX <= character.x <= self.maxX

    def chair(self, character):
        for chair in self.chairs:
            if character.direction < 0 and (character.x, character.y) == (chair.x, chair.y):
                return chair
        return None

    def getNextChair(self, character, moveAlong):
        """Returns the chair nearest to the character, or the chair he should
        sit in if he was dethroned (moveAlong=True) or has just started looking
        for a chair (in which case he's facing right)."""
        if character.direction > 0 or (moveAlong and self.chairs[0].x == character.x):
            return self.chairs[-1]
        minDistance = 100
        nextChair = None
        for chair in self.chairs:
            distance = character.x - chair.x
            if distance == 0 and moveAlong:
                continue
            if 0 <= distance < minDistance:
                minDistance = distance
                nextChair = chair
        return nextChair

class Chair:
    def __init__(self, room, x):
        self.x = x
        self.y = room.y
        self.room = room
        self.occupant = None

    def seat(self, character):
        self.occupant = character

    def vacate(self):
        self.occupant = None

class Blackboard:
    def __init__(self, x, y, font, scale):
        self.x = x
        self.y = y
        self.font = font
        self.width = 64 * scale
        self.clear()

    def setWriter(self, character):
        self.writer = character

    def getWriter(self):
        return self.writer

    def write(self, char):
        if char == '^':
            if len(self.lines) == 1:
                self.lines.append('')
            return
        self.lines[-1] += char
        self.images = []
        for line in self.lines:
            textImage = self.font.render(line, (255, 255, 255), (0, 0, 0))
            if textImage.get_width() > self.width:
                textImage = textImage.subsurface((0, 0), (self.width, textImage.get_height()))
            self.images.append(textImage)

    def wipe(self, wipedBit, x):
        for image in self.images:
            image.blit(wipedBit, (x, 0))

    def isDirty(self):
        return len(self.images) > 0

    def getImages(self):
        return (self.x, self.y, self.images)

    def clear(self):
        self.lines = ['']
        self.images = []
        self.writer = None

    def beside(self, character):
        armX = character.x + character.direction + 1
        return self.x + 1 <= armX <= self.x + 6

    def shows(self, string):
        return self.lines[0].lower().startswith(string.lower())

class NoGoZone:
    def __init__(self, zoneId, minX, maxX, bottomY, topY):
        self.zoneId = zoneId
        self.minX = minX
        self.maxX = maxX
        self.bottomY = bottomY
        self.topY = topY

    def contains(self, character):
        if not (self.minX <= character.x <= self.maxX):
            return False
        return self.topY <= character.y <= self.bottomY

class Flashable:
    def __init__(self, x, y, score, image=None, inverseImage=None):
        self.x = x
        self.y = y
        self.score = score
        self.image = image
        self.inverseImage = inverseImage
        self.flashing = False

    def getScore(self):
        return self.score // 10

    def isFlashing(self):
        return self.flashing

    def flash(self):
        self.flashing = True

    def unflash(self):
        self.flashing = False

    def getImages(self, inverse):
        if self.flashing:
            image = self.inverseImage if inverse else self.image
            return (self.x, self.y, image)
        return (0, 0, None)

class Shield(Flashable):
    pass

class Safe(Flashable):
    pass

class Timetable:
    def __init__(self, lessonLength, lessonStartTime):
        self.index = -1
        self.lessonId = None
        self.lessons = []
        self.lessonDetails = {}
        self.counter = 0
        self.ticking = True
        self.lessonLength = lessonLength
        self.lessonStartTime = lessonStartTime
        self.specialPlaytimes = []

    def reinitialise(self):
        self.counter = 0
        self.ticking = True
        self.index = (self.index & 48) + 15

    def addLesson(self, lessonId):
        self.lessons.append(lessonId)

    def addLessonDetails(self, lessonId, hideTeacher, teacherId, roomId):
        self.lessonDetails[lessonId] = (hideTeacher, teacherId, roomId)

    def addSpecialPlaytime(self, lessonId):
        self.specialPlaytimes.append(lessonId)

    def nextLesson(self):
        self.ticking = True
        self.counter = self.lessonLength
        self.index = (self.index + 1) % len(self.lessons)
        self.lessonId = self.lessons[self.index]
        if self.lessonId.startswith('Playtime') and self.specialPlaytimes and random.randrange(8) < 3:
            self.lessonId = self.specialPlaytimes[random.randrange(len(self.specialPlaytimes))]

    def getLessonId(self):
        return self.lessonId

    def hideTeacher(self):
        return self.lessonDetails[self.getLessonId()][0]

    def getTeacherId(self):
        return self.lessonDetails[self.getLessonId()][1]

    def getRoomId(self):
        return self.lessonDetails[self.getLessonId()][2]

    def tick(self):
        if self.ticking:
            self.counter -= 1
        return self.counter < 0

    def isTime(self, time):
        return self.counter + time < self.lessonLength

    def isTimeToStartLesson(self):
        return self.isTime(self.lessonStartTime)

    def isTeachingEric(self, character):
        return self.getTeacherId() == character.characterId

    def upAYear(self):
        self.counter = self.lessonLength // 2

    def stop(self):
        self.ticking = False

    def resume(self, ticks):
        self.counter = ticks
        self.ticking = True

class AssemblyMessageGenerator:
    def __init__(self):
        self.text = None
        self.verbs = []
        self.nouns = []

    def setText(self, text):
        self.text = text

    def addVerb(self, verb):
        self.verbs.append(verb)

    def addNoun(self, noun):
        self.nouns.append(noun)

    def generateMessage(self):
        verb = self.verbs[random.randrange(len(self.verbs))]
        noun = self.nouns[random.randrange(len(self.nouns))]
        message = self.text.replace('$VERB', verb)
        return message.replace('$NOUN', noun)

class Scoreboard:
    def __init__(self, screen):
        self.screen = screen
        self.hiscore = 0
        self.reset()

    def reset(self):
        self.score = self.lines = 0

    def addToScore(self, addend):
        self.score += addend
        self.screen.printScore(self.score)

    def addLines(self, addend):
        self.lines += addend
        self.screen.printLines(self.lines)

    def reinitialise(self):
        if self.score > self.hiscore:
            self.hiscore = self.score
            self.screen.printHiScore(self.hiscore)
        self.reset()
        self.screen.printScore(self.score)
        self.screen.printLines(self.lines)

    def getLines(self):
        return self.lines

class Signals:
    ASSEMBLY_FINISHED = 'AssemblyFinished'
    TIME_FOR_ASSEMBLY = 'TimeForAssembly'
    SWOT_READY = 'SwotReady'

    def __init__(self):
        self.clear()

    def clear(self):
        self.signals = {}

    def isRaised(self, signal):
        return self.signals.get(signal, False)

    def signal(self, signal):
        self.signals[signal] = True

    def unsignal(self, signal):
        self.signals[signal] = False
