# Copyright 2008 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import sys
import random
from skool import Location
from ai import *

class Character:
    COME_ALONG_PREFIX = 'COME_ALONG'

    def __init__(self, characterId, name, flags=''):
        self.characterId = characterId
        self.name = name
        self.flags = flags
        self.commandList = CommandList()
        self.x, self.y = -3, 0
        self.randomLocations = ()
        self.direction = 1 # 1 = facing right, -1 = facing left
        self.verticalDirection = 0 # 1 = down, 0 = neither up nor down
        self.action = 0 # 0 = walking, 1 = still, 2 = writing, 3 = other
        self.walkStates = ['WALK0', 'WALK1', 'WALK2', 'WALK3']
        self.speed = 1 # 1 = walk, 2 = run
        self.speedChangeDelay = 1
        self.walkDelay = random.randrange(3, 6)
        self.actionDelay = 1
        self.staircase = None
        self.barrier = None
        self.room = None
        self.commandLists = {}
        self.sitDownMessage = None
        self.bubble = None
        self.blackboardMessages = []
        self.qaGenerator = None
        self.specialAnswer = None
        self.wipingBoard = False
        self.pellet = None
        self.linesMessages = {}
        self.comeAlongMessages = []
        self.comeAlongIndex = 0
        self.lessonMessages = []
        self.changesSeats = True

    def setComponents(self, cast, skool, screen, beeper):
        self.cast = cast
        self.skool = skool
        self.screen = screen
        self.beeper = beeper

    def setTitle(self, title):
        self.title = title

    def getTitle(self):
        return self.title or self.name

    def setInitialLocation(self, x, y):
        self.initialLocation = Location((x, y))
        self.x = x
        self.y = y

    def initialiseSpecialAnswer(self):
        if self.qaGenerator:
            self.specialAnswer = self.qaGenerator.initialiseSpecialAnswer()

    def setSitDownMessage(self, message):
        self.sitDownMessage = message

    def setAnimatoryStates(self, asDictL, asDictR):
        self.asDictL = asDictL
        self.asDictR = asDictR

    def initialiseAnimatoryState(self, initialAS):
        self.direction = self.initialDirection = -1 if initialAS[0].upper() == 'L' else 1
        self.animatoryState = self.initialAS = self.walkStates[int(initialAS[1:])]

    def reinitialise(self):
        self.x, self.y = self.initialLocation.x, self.initialLocation.y
        self.direction = self.initialDirection
        self.verticalDirection = 0
        self.animatoryState = self.initialAS
        self.bubble = None
        self.wipingBoard = False
        self.action = 0
        self.run(False)
        self.speedChangeDelay = 1
        self.commandList = CommandList()

    def setControllingCommand(self, command):
        self.commandList.setControllingCommand(command)

    def isAdult(self):
        return 'A' in self.flags

    def isConkerable(self):
        return 'C' in self.flags

    def canOpenDoors(self):
        return 'D' in self.flags

    def canExpelEric(self):
        return 'E' in self.flags

    def isPunchable(self):
        return 'F' in self.flags

    def canGiveLines(self):
        return 'L' in self.flags

    def isScaredOfMice(self):
        return 'M' in self.flags

    def isPelletable(self):
        return 'P' in self.flags

    def canReceiveLines(self):
        return 'R' in self.flags

    def hasSecret(self):
        return 'S' in self.flags

    def isATeacher(self):
        return 'T' in self.flags

    def sometimesRuns(self):
        return 'W' not in self.flags

    def isHome(self, x):
        if 'G' in self.flags:
            return self.x > x
        elif 'B' in self.flags:
            return self.x < x - 1
        return True

    def getImage(self):
        asDict = self.asDictR if self.direction > 0 else self.asDictL
        return (self.x, self.y, asDict[self.animatoryState])

    def getSpeechBubble(self):
        if self.bubble:
            bubbleX = 8 * ((self.x + 1) // 8)
            bubbleY = self.y - (3 if self.isAdult() else 2)
            return (bubbleX, bubbleY, self.bubble)
        return (0, 0, None)

    def setRandomLocations(self, locations):
        self.randomLocations = locations

    def getRandomDestination(self):
        dest = (self.x, self.y)
        choice = len(self.randomLocations)
        if choice == 0:
            return dest
        elif choice == 1:
            return self.randomLocations[0]
        while dest == (self.x, self.y):
            dest = self.randomLocations[random.randrange(choice)]
        return dest

    def addCommandList(self, lessonId, tapId):
        self.commandLists[lessonId] = tapId

    def getTapId(self, lessonId):
        return self.commandLists[lessonId]

    def setCommandListTemplate(self, template):
        self.commandList.setTemplate(template)
        self.lesson = None
        self.triggerSpeedChange()

    def restartTable(self):
        self.commandList.restart()

    def triggerSpeedChange(self):
        self.speedChangeDelay = 1

    def resetWalkDelay(self):
        self.walkDelay = 7 - 2 * self.speed

    def isTimeToMove(self):
        if self.bubble:
            self.actionDelay = (self.actionDelay + 1) % 3
            return self.actionDelay == 0
        if self.action in (1, 2):
            # Writing or not moving
            self.actionDelay = (self.actionDelay + 1) % 5
            return self.actionDelay == 0
        if self.action == 3:
            # Perform hitting, firing etc. quickly
            self.actionDelay = (self.actionDelay + 1) % 2
            return self.actionDelay == 0
        self.walkDelay -= 1
        if self.walkDelay > 0:
            return False
        self.speedChangeDelay -= 1
        if self.speedChangeDelay == 0:
            self.speedChangeDelay = random.randrange(16, 32)
            if self.sometimesRuns():
                self.run(self.speedChangeDelay & 1)
            else:
                self.run(False)
        self.resetWalkDelay()
        return True

    def move(self, moveNow):
        if not (self.isTimeToMove() or moveNow):
            return
        if self.midstride():
            return self.walk()
        self.staircase = self.skool.staircase(self)
        self.barrier = self.skool.barrier(self)
        self.room = self.skool.room(self)
        self.commandList.command(self)

    def left(self):
        if self.direction > 0:
            self.turn()
        else:
            if self.barrier and self.barrier.x <= self.x:
                if self.canOpenDoors() and self.barrier.isOpenable():
                    return self.openDoor()
                return
            onStairs = False
            if self.staircase:
                if self.staircase.direction < 0:
                    onStairs = self.x != self.staircase.bottom.x or self.staircase.force
                else:
                    onStairs = self.x != self.staircase.top.x or self.staircase.force
            self.walk(onStairs)

    def right(self):
        if self.direction > 0:
            if self.barrier and self.x < self.barrier.x:
                if self.canOpenDoors() and self.barrier.isOpenable():
                    return self.openDoor()
                return
            onStairs = False
            if self.staircase:
                if self.staircase.direction > 0:
                    onStairs = self.x != self.staircase.bottom.x or self.staircase.force
                else:
                    onStairs = self.x != self.staircase.top.x or self.staircase.force
            self.walk(onStairs)
        else:
            self.turn()

    def up(self):
        if self.staircase:
            if self.direction == self.staircase.direction or (self.x == self.staircase.top.x and self.staircase.force):
                return self.walk(True)
            elif self.x != self.staircase.top.x:
                return self.turn()
        if self.direction > 0:
            self.right()
        else:
            self.left()

    def down(self):
        if self.staircase:
            if self.direction != self.staircase.direction or (self.x == self.staircase.bottom.x and self.staircase.force):
                return self.walk(True)
            elif self.x != self.staircase.bottom.x:
                return self.turn()
        if self.direction > 0:
            self.right()
        else:
            self.left()

    def turn(self):
        self.direction *= -1

    def getWalkStateIndex(self):
        if self.animatoryState in self.walkStates:
            return self.walkStates.index(self.animatoryState)
        return -1

    def walk(self, onStairs=False):
        if onStairs:
            if self.direction == self.staircase.direction:
                if not self.midstride():
                    self.y -= 1
            else:
                self.verticalDirection = 1
        walkState = (self.getWalkStateIndex() + 1) % 4
        if walkState % 2 == 0:
            self.x += self.direction
            self.y += self.verticalDirection
            if self.wipingBoard:
                # Animatory state sequence is 0, 1, 0, 1... when wiping board
                walkState = 0
            self.verticalDirection = 0
        self.animatoryState = self.walkStates[walkState]
        self.action = 0

    def chair(self):
        return self.skool.chair(self)

    def sit(self):
        if self.isSitting():
            self.getUp()
            return
        self.previousAS = self.animatoryState
        chair = self.chair()
        if chair:
            self.animatoryState = 'SITTING_ON_CHAIR'
            occupant = chair.occupant
            chair.seat(self)
            if occupant:
                occupant.dethrone()
        else:
            staircase = self.skool.staircase(self)
            if staircase and staircase.supports(self):
                if self.direction * staircase.direction < 0:
                    self.animatoryState = 'SITTING_ON_CHAIR'
            else:
                self.sitOnFloor()
        self.action = 1

    def standUp(self):
        self.animatoryState = 'WALK0'
        self.resetWalkDelay()

    def dethrone(self):
        self.sitOnFloor()
        self.addCommand(FindSeat())
        self.addCommand(Dethroned())

    def keepSeat(self):
        self.changesSeats = False

    def sitOnFloor(self):
        self.animatoryState = 'SITTING_ON_FLOOR'

    def raiseArm(self):
        self.animatoryState = 'ARM_UP'

    def lowerArm(self):
        self.animatoryState = 'WALK0'

    def knockOver(self):
        self.previousAS = self.animatoryState
        if self.isAdult():
            self.animatoryState = 'KNOCKED_OVER'
        else:
            self.animatoryState = 'KNOCKED_OUT'
        self.action = 1

    def raiseFist(self):
        self.animatoryState = 'HITTING0'
        self.action = 3

    def punch(self):
        self.animatoryState = 'HITTING1'

    def lowerFist(self):
        self.animatoryState = 'HITTING0'

    def raiseCatapult(self):
        self.animatoryState = 'CATAPULT0'
        self.action = 3

    def fireCatapult(self):
        self.animatoryState = 'CATAPULT1'

    def lowerCatapult(self):
        self.animatoryState = 'CATAPULT0'

    def completeAction(self):
        self.animatoryState = 'WALK0'
        self.action = 0

    def isStanding(self):
        return self.animatoryState in ('WALK0', 'WALK1', 'WALK2', 'WALK3')

    def isSitting(self):
        return self.animatoryState in ('SITTING_ON_FLOOR', 'SITTING_ON_CHAIR')

    def isSittingOnFloor(self):
        return self.animatoryState == 'SITTING_ON_FLOOR'

    def isSittingOnChair(self):
        return self.animatoryState == 'SITTING_ON_CHAIR' and self.chair()

    def isSittingOnStairs(self):
        return self.animatoryState == 'SITTING_ON_CHAIR' and not self.chair()

    def isKnockedOver(self):
        return self.animatoryState == 'KNOCKED_OVER'

    def hasArmRaised(self):
        return self.animatoryState == 'ARM_UP'

    def isKnockedOut(self):
        return self.animatoryState == 'KNOCKED_OUT'

    def isRaisingCatapult(self):
        return self.animatoryState == 'CATAPULT0'

    def isFiringCatapult(self):
        return self.animatoryState == 'CATAPULT1'

    def isLoweringCatapult(self):
        return self.animatoryState == 'CATAPULT0'

    def isRaisingFist(self):
        return self.animatoryState == 'HITTING0'

    def isPunching(self):
        return self.animatoryState == 'HITTING1'

    def isLoweringFist(self):
        return self.animatoryState == 'HITTING0'

    def isDeckable(self):
        """Returns whether this character is in an animatory state and location
        amenable to being knocked over."""
        if self.getFloor() is None:
            return
        if self.isAdult():
            return self.isStanding() or self.animatoryState == 'ARM_UP'
        return self.isStanding() or self.isSittingOnChair()

    def getUp(self):
        if self.isSittingOnChair():
            self.chair().vacate()
        self.animatoryState = self.previousAS

    def midstride(self):
        return self.animatoryState in ('WALK1', 'WALK3')

    def getFloor(self, object=None):
        return self.skool.floor(object or self)

    def onStairs(self):
        if self.staircase:
            return self.x not in (self.staircase.bottom.x, self.staircase.top.x)
        return False

    def getNextStaircase(self, destination):
        if self.getFloor() is None:
            sys.stderr.write('%s at %i, %i has no home floor\n' % (self.name, self.x, self.y))
        if self.getFloor(destination) is None:
            sys.stderr.write('%s at %i, %i going to %i, %i has no destination floor\n' % (self.name, self.x, self.y, destination.x, destination.y))
        return self.skool.nextStaircase(self.getFloor(), self.getFloor(destination))

    def openDoor(self):
        return OpenDoor(self.barrier.doorId)

    def moveDoor(self, doorId, shut):
        self.skool.moveDoor(doorId, shut)

    def getNextChair(self, moveAlong):
        return self.room.getNextChair(self, moveAlong and self.changesSeats)

    def signal(self, signal):
        self.skool.signal(signal)

    def unsignal(self, signal):
        self.skool.unsignal(signal)

    def gotSignal(self, signal):
        return self.skool.gotSignal(signal)

    def isTimeToStartLesson(self):
        return self.skool.isTimeToStartLesson()

    def say(self, words, shift):
        lipPos = (self.x + 1) % 8
        self.bubble, done = self.screen.getBubble(words, lipPos, shift)
        return done

    def removeBubble(self):
        self.bubble = None

    def getAssemblyMessage(self):
        return self.skool.getAssemblyMessage()

    def setWipingBoard(self, wiping):
        self.wipingBoard = wiping

    def wipeBoard(self, column):
        self.skool.wipeBoard(self.room.blackboard, column)

    def writeOnBoard(self, message, index):
        self.action = 2
        return self.skool.writeOnBoard(self, self.room.getBlackboard(), message, index)

    def resolveLocationId(self, locationId):
        return self.skool.resolveLocationId(locationId)

    def isTeachingEric(self):
        return self.skool.isTeachingEric(self)

    def getQAGenerator(self):
        if self.qaGenerator is None:
            self.qaGenerator = QAGenerator()
        return self.qaGenerator

    def addBlackboardMessage(self, message):
        self.blackboardMessages.append(message)

    def getBlackboardMessage(self):
        message = self.blackboardMessages[random.randrange(len(self.blackboardMessages))]
        return self.cast.expandNames(message)

    def run(self, run):
        self.speed = 2 if run else 1

    def isFacing(self, object):
        return self.direction * (object.x - self.x) >= 0

    def getFacingCharacters(self, offset):
        """Returns a list of any characters facing this character from a
        distance equal to offset."""
        return self.cast.getFacingCharacters(self, offset)

    def addCommand(self, command):
        self.commandList.addCommand(command)

    def deck(self):
        self.knockOver()
        if self.isAdult():
            self.addCommand(KnockedOver())
        else:
            if self.previousAS == 'SITTING_ON_CHAIR':
                self.chair().vacate()
                self.addCommand(FindSeat())
            self.addCommand(KnockedOut())

    def setPellet(self, pellet):
        self.pellet = pellet

    def fire(self):
        """Launches a catapult pellet."""
        self.pellet.launch(self.x + self.direction, self.y, self.direction)

    def getNearbyAdults(self):
        """Returns a list of adults who are close enough to this character
        (and are facing the right way) to be able to see him."""
        return self.cast.getNearbyAdults(self)

    def setRestartPoint(self):
        """Discard the current and all previous commands in the command list (so
        any command that restarts the command list will see the next command as
        the first)."""
        self.commandList.setRestartPoint()

    def jumpIfShut(self, doorId, offset):
        """Jump forwards (or backwards) in the command list if the given door is
        shut."""
        if self.skool.isDoorShut(doorId):
            self.commandList.jump(offset)

    def jumpIfOpen(self, doorId, offset):
        """Jump forwards (or backwards) in the command list if the given door is
        open."""
        if not self.skool.isDoorShut(doorId):
            self.commandList.jump(offset)

    def checkDoorStatus(self, doorId, shut):
        return self.skool.isDoorShut(doorId) == shut

    def stalk(self, characterId):
        """Sets this character's destination equal to that of another
        character's destination (if they are both GoTo-ing)."""
        self.commandList.setGoToDestination(self.getDestination(characterId))

    def getGoToDestination(self):
        """Returns the destination of this character, or None if they are not
        under the control of the GoTo command."""
        return self.commandList.getGoToDestination()

    def getDestination(self, characterId):
        return self.cast.get(characterId).getGoToDestination()

    def waitAtDoor(self, doorId):
        """Returns whether the characters are on the correct side of the given
        door."""
        return self.cast.isHome(self.skool.getDoor(doorId).x)

    def isEricExpelled(self):
        return self.skool.isEricExpelled()

    def getComeAlongMessageId(self):
        messageId = self.comeAlongMessages[self.comeAlongIndex]
        self.comeAlongIndex = (self.comeAlongIndex + 1) % len(self.comeAlongMessages)
        if self.comeAlongIndex == 0 and len(self.comeAlongMessages) > 2:
            self.comeAlongIndex = 1
        return messageId

    def printLinesBubble(self, message, ink, paper, sound):
        y = self.y - 2 if self.isKnockedOver() else self.y - 3
        self.screen.printLinesBubble(self.x, y, message, ink, paper)
        self.beeper.makeLinesSound(sound)

    def giveLines(self, recipientId, messageId):
        """Makes this character give lines."""
        if not self.screen.contains(self):
            return
        recipient = self.cast.get(recipientId)
        admonition = self.linesMessages.get(messageId, None)
        if not (recipient and admonition):
            return
        if recipient.isEric() and self.isEricExpelled():
            return
        numLines = 10 * random.randrange(1, 9)
        message = ('%i0 LINES' % numLines, recipient.name)
        if recipient.isEric():
            paper = 2
            self.skool.addLines(numLines)
        else:
            paper = 4
            self.skool.addToScore(numLines)
        self.printLinesBubble(message, 7, paper, 1)
        self.printLinesBubble(admonition, 7, paper, 2)

    def canSeeSpecialAnswer(self):
        """Returns whether this character can see his special answer written on
        a blackboard."""
        if self.specialAnswer:
            blackboard = self.skool.visibleBlackboard(self)
            if blackboard:
                return blackboard.shows(self.specialAnswer)

    def revealSecret(self, decked):
        """Makes this character reveal his secret (if he has one, and conditions
        are right)."""
        if not self.hasSecret() or not self.skool.canRevealSecret():
            return
        if not self.screen.contains(self):
            return
        if decked:
            reveal = self.specialAnswer is None
        else:
            reveal = self.canSeeSpecialAnswer()
        if reveal:
            self.printLinesBubble(('', self.secret), 7, 0, 2)

    def initialiseSecret(self):
        if self.hasSecret():
            letters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ'
            self.secret = letters[random.randrange(len(letters))]
        return self.secret

    def isEric(self):
        return False

    def addLinesMessage(self, messageId, messageLines):
        if not self.linesMessages.has_key(messageId):
            self.linesMessages[messageId] = messageLines
        if messageId.startswith(Character.COME_ALONG_PREFIX):
            self.comeAlongMessages.append(messageId)

    def addLessonMessage(self, message, condition):
        self.lessonMessages.append((message, condition))

    def lessonMessageCondition(self, condition):
        if condition == 'BoardDirty':
            return self.room.blackboardDirty()
        return True

    def getLessonMessage(self):
        while True:
            message, condition = self.lessonMessages[random.randrange(len(self.lessonMessages))]
            if self.lessonMessageCondition(condition):
                break
        randNum = random.randrange(99, 1000)
        return message.replace('$n', str(randNum))

    def reprimand(self):
        """Make character give lines to the nearest lines recipient for knocking
        him over."""
        recipient = self.cast.getNearestLinesRecipient(self)
        if recipient:
            self.giveLines(recipient.characterId, 'NEVER_AGAIN')

    def checkShieldsAt(self, x, y):
        return self.skool.hitShield(x, y)

    def hasLineOfSightTo(self, character):
        """Returns whether this character has a line of sight to the given
        character, i.e. there are no walls between them."""
        return self.skool.lineOfSightBetween(self, character)

    def getLocationOfEric(self):
        return self.cast.getLocationOfEric()

    def isFacingEric(self):
        return self.isFacing(self.cast.getEric())

    def isBesideEric(self):
        """Returns whether this character is beside Eric."""
        return self.cast.isBesideEric(self)

    def isEricAbsent(self):
        return self.cast.isEricAbsent()

    def findAndExpelEric(self):
        self.bubble = None
        template = CommandListTemplate('FindAndExpelEric')
        template.addCommand(FindEric)
        template.addCommand(TellEric, self.skool.getTooManyLinesMessage())
        template.addCommand(EndGame)
        self.setCommandListTemplate(template)

    def freezeEric(self):
        self.cast.freezeEric()

    def unfreezeEric(self):
        self.cast.unfreezeEric()

    def ericUnderstood(self):
        return self.cast.ericUnderstood()

    def expandNames(self, message):
        return self.cast.expandNames(message)

    def getEricHasMumpsMessage(self):
        return self.skool.getEricHasMumpsMessage()

    def getUnderstandMessage(self):
        return self.skool.getUnderstandMessage()

    def stopClock(self):
        self.skool.stopClock()

    def startClock(self, ticks):
        self.skool.startClock(ticks)

    def isTouchingEric(self):
        return self.cast.isTouchingEric(self)

    def addLines(self, lines):
        self.skool.addLines(lines // 10)

    def endGame(self):
        self.skool.endGame()

    def tripPeopleUp(self):
        self.cast.tripPeopleUpAt(self.x, self.y)

    ############################################################################
    # Lesson coordination methods
    ############################################################################
    def startLesson(self):
        """Called by the Swot to start a lesson."""
        self.lesson = Lesson(self.cast, self, self.room)
        self.skool.setLesson(self.lesson)

    def joinLesson(self, withQA):
        """Called by the teacher to join the lesson started by the Swot."""
        self.lesson = self.skool.getLesson()
        self.lesson.join(self, self.qaGenerator if withQA else None)

    def nextSwotAction(self):
        """Returns the next command to be executed by the swot (which may be
        None)."""
        return self.lesson.nextSwotAction()

    def nextTeacherAction(self):
        """Returns the next command to be executed by the teacher (which may be
        None)."""
        return self.lesson.nextTeacherAction()

    def finishedSpeaking(self):
        """Called by the swot or the teacher when he's finished speaking."""
        self.lesson.finishedSpeaking()

    def setHomeRoom(self):
        self.skool.setHomeRoom()

    def resetComeAlongIndex(self):
        self.comeAlongIndex = 0

class Lesson:
    def __init__(self, cast, swot, room):
        self.cast = cast
        self.swot = swot
        self.room = room
        self.hitterId = None
        self.writerId = None
        self.teacher = None
        self.qaGenerator = None
        self.answer = None
        self.askedSpecial = False
        self.actor = None
        self.swotAction = self.checkEricInitial
        self.teacherAction = None
        self.baseAction = self.tellClassWhatToDo
        self.baseLocation = None
        self.grassed = False
        self.absenceMessageIds = ('BE_PUNCTUAL', 'STAY_IN_CLASS')
        self.absenceIndex = 0

    def join(self, teacher, qaGenerator):
        self.teacher = teacher
        self.qaGenerator = qaGenerator
        if qaGenerator:
            self.baseAction = self.askQuestion
        self.actor = self.swot
        self.baseLocation = (teacher.x, teacher.y)

    def nextSwotAction(self):
        while self.actor is self.swot:
            nextAction = self.swotAction()
            if nextAction:
                return nextAction

    def checkEricInitial(self):
        self.teacher.setHomeRoom()
        self.swotAction = self.grassForHitting
        if self.isEricAbsent():
            self.teacherAction = self.fetchEric
            return Say(self.cast.getAbsentTale(self.teacher), True)

    def grassForHitting(self):
        self.swotAction = self.grassForWriting
        self.teacherAction = self.giveLinesForHitting
        if random.randrange(256) < 36:
            self.hitterId, tale = self.cast.getHitTale(self.teacher)
            return Say(tale, True)
        self.switch()

    def grassForWriting(self):
        self.grassed = True
        self.teacherAction = self.giveLinesForWriting
        writer = self.room.getBlackboardWriter()
        if writer:
            self.writerId, tale = self.cast.getWriteTale(writer.characterId, self.teacher)
            if tale:
                return Say(tale, True)
        self.switch()

    def checkEric(self):
        if self.isEricAbsent():
            self.teacherAction = self.fetchEric
            return Say(self.cast.getAbsentTale(self.teacher), True)
        self.switch(self.baseAction)

    def answerQuestion(self):
        self.swotAction = self.checkEric
        return Say(self.answer)

    def nextTeacherAction(self):
        while self.actor is self.teacher:
            nextAction = self.teacherAction()
            if nextAction:
                return nextAction

    def fetchEric(self):
        if random.randrange(64) < 21:
            self.teacher.giveLines(self.swot.characterId, 'NO_TALES')
        if self.isEricAbsent():
            self.teacherAction = self.returnToBase
            self.teacher.resetComeAlongIndex()
            return FetchEric()
        linesMessageId = self.absenceMessageIds[self.absenceIndex]
        self.teacher.giveLines(self.cast.getEric().characterId, linesMessageId)
        self.switch()

    def returnToBase(self):
        if (self.teacher.x, self.teacher.y) != self.baseLocation:
            return GoToXY(*self.baseLocation)
        if self.room.hasBlackboard() and not self.grassed:
            if self.teacher.direction > 0:
                # Turn teacher round before continuing
                return GoTowardsXY(self.teacher.x - 1, self.teacher.y)
            else:
                self.switch()
                return
        if self.qaGenerator:
            if self.teacher.direction < 0:
                # Turn teacher round before continuing
                return GoTowardsXY(self.teacher.x + 1, self.teacher.y)
            else:
                self.teacherAction = self.askQuestion
        else:
            self.teacherAction = self.walkUpOrDown
            if self.teacher.direction < 0:
                return GoToXY(self.teacher.x - 3, self.teacher.y)

    def giveLines(self, victimId, messageId):
        if victimId:
            victimPresent = self.room.contains(self.cast.get(victimId))
            punishSwot = random.randrange(64) < 21
            if punishSwot or not victimPresent:
                victimId, messageId = self.swot.characterId, 'NO_TALES'
            self.teacher.giveLines(victimId, messageId)

    def giveLinesForHitting(self):
        self.absenceIndex = 1
        self.giveLines(self.hitterId, 'NO_HITTING')
        self.switch()

    def giveLinesForWriting(self):
        self.giveLines(self.writerId, 'NO_WRITING')
        self.teacherAction = self.wipeBoard

    def wipeBoard(self):
        if self.room.hasBlackboard():
            self.teacherAction = self.writeOnBoard
            return WipeBoard()
        self.teacherAction = self.baseAction

    def writeOnBoard(self):
        self.baseLocation = (self.teacher.x, self.teacher.y)
        self.teacherAction = self.baseAction
        if random.randrange(256) > 184:
            return WriteOnBoard(self.teacher.getBlackboardMessage())

    def askQuestion(self):
        self.swotAction = self.answerQuestion
        return Say(self.getQuestion(), True)

    def tellClassWhatToDo(self):
        self.baseAction = self.walkUpOrDown
        self.teacherAction = self.walkUpOrDown
        return TellClassWhatToDo()

    def walkUpOrDown(self):
        self.switch(self.checkEric)
        return WalkUpOrDown()

    def getQuestion(self):
        generator = self.qaGenerator.prepareQA
        if not self.askedSpecial:
            self.askedSpecial = True
            if self.qaGenerator.hasSpecialQuestion():
                generator = self.qaGenerator.prepareSpecialQA
        question, self.answer = generator()
        return question

    def switch(self, action=None):
        if self.actor is self.swot:
            self.actor = self.teacher
            self.teacherAction = action or self.teacherAction
        else:
            self.actor = self.swot
            self.swotAction = action or self.swotAction

    def finishedSpeaking(self):
        self.switch()

    def isEricAbsent(self):
        return not self.room.contains(self.cast.getEric())

class QAGenerator:
    def __init__(self):
        self.questions = []
        self.answers = {}
        self.qaPairs = {}
        self.specialQAGroup = None
        self.remaining = {}

    def setSpecialGroup(self, qaGroup, index):
        self.specialQAGroup = qaGroup
        self.specialQAPairIndex = int(index)

    def setSpecialQuestion(self, text):
        self.specialQuestion = text

    def setSpecialAnswer(self, text):
        self.specialAnswer = text

    def initialiseSpecialAnswer(self):
        if self.specialQAGroup:
            self.specialAnswerIndex = random.randrange(len(self.qaPairs[self.specialQAGroup]))
            return self.qaPairs[self.specialQAGroup][self.specialAnswerIndex][self.specialQAPairIndex]

    def hasSpecialQuestion(self):
        return self.specialQAGroup is not None

    def addQuestion(self, questionId, qaGroup, text):
        self.questions.append((questionId, qaGroup, text))

    def addAnswer(self, questionId, text):
        self.answers[questionId] = text

    def addQAPair(self, qaGroup, word1, word2):
        if not self.qaPairs.has_key(qaGroup):
            self.qaPairs[qaGroup] = []
            self.remaining[qaGroup] = []
        self.qaPairs[qaGroup].append((word1, word2))

    def expand(self, template, word1, word2):
        return template.replace('$1', word1).replace('$2', word2)

    def prepareSpecialQA(self):
        word1, word2 = self.qaPairs[self.specialQAGroup][self.specialAnswerIndex]
        return self.specialQuestion, self.expand(self.specialAnswer, word1, word2)

    def prepareQA(self):
        questionId, qaGroup, question = self.questions[random.randrange(len(self.questions))]
        answer = self.answers[questionId]
        if not self.remaining[qaGroup]:
            self.remaining[qaGroup] = list(range(len(self.qaPairs[qaGroup])))
        randomIndex = self.remaining[qaGroup].pop(random.randrange(len(self.remaining[qaGroup])))
        word1, word2 = self.qaPairs[qaGroup][randomIndex]
        return self.expand(question, word1, word2), self.expand(answer, word1, word2)
