# Copyright 2008 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import pygame
from character import Character
from ai import Write, Jump, Hit, FireCatapult, Freeze

class Eric(Character):
    def __init__(self, characterId, name, flags):
        Character.__init__(self, characterId, name, flags)
        self.controller = None
        self.linesDelay = 40
        self.frozen = False
        self.understood = False

    def reinitialise(self):
        Character.reinitialise(self)
        self.frozen = False
        self.understood = False
        self.controller = None
        self.linesDelay = 40

    def move(self, keyboard, moveNow):
        self.keyboard = keyboard
        self.getLines()
        self.walkDelay -= 1
        if self.walkDelay > 0 and not moveNow:
            return 0
        self.keyboard.pump()
        self.walkDelay = 2
        if self.controller:
            if self.controller.command(self) is self.controller:
                self.controller = None
            return 0
        if self.midstride():
            self.makeWalkingSound(self.getWalkStateIndex())
            return self.walk()
        if self.isKnockedOut() or self.isSitting():
            if self.keyboard.wasPressed(pygame.K_s):
                self.getUp()
                self.beeper.makeSittingSound()
            return 0
        if self.keyboard.wasPressed(pygame.K_s):
            self.sit()
            self.beeper.makeSittingSound()
            return 0
        if self.isSitting():
            return 0
        if self.keyboard.wasPressed(pygame.K_f) and not self.pellet.isAirborne():
            self.controller = FireCatapult()
            return 0
        if self.keyboard.wasPressed(pygame.K_h):
            self.controller = Hit()
            return 0
        if self.keyboard.wasPressed(pygame.K_j):
            self.controller = Jump()
            return 0
        if self.keyboard.wasPressed(pygame.K_w) and self.skool.besideBlackboard(self):
            self.raiseArm()
            self.controller = Write()
            self.keyboard.startWriting()
            return 0
        self.staircase = self.skool.staircase(self)
        if not (self.staircase or self.cast.isStandingOnKid(self) or self.skool.onFloor(self)):
            self.y += 1
            return 0
        self.barrier = self.skool.barrier(self)
        oldWS, oldAS, oldDirection = self.getWalkStateIndex(), self.animatoryState, self.direction
        if self.keyboard.pressed(pygame.K_LEFT):
            self.left()
        elif self.keyboard.pressed(pygame.K_RIGHT):
            self.right()
        elif self.keyboard.pressed(pygame.K_UP):
            self.up()
        elif self.keyboard.pressed(pygame.K_DOWN):
            self.down()
        if (self.animatoryState, self.direction) != (oldAS, oldDirection):
            self.makeWalkingSound(oldWS)
        return 0

    def getLines(self):
        self.linesDelay = max(0, self.linesDelay - 1)
        if self.linesDelay > 0:
            return
        gaveLines = False
        homeRoom = self.skool.getHomeRoom()
        if self.isSittingOnStairs():
            gaveLines = self.alertLinesGivers('NO_SITTING_ON_STAIRS')
        elif self.skool.inNoGoZone(self):
            gaveLines = self.alertLinesGivers('GET_OUT')
        elif self.shouldGetAlong():
            gaveLines = self.alertLinesGivers('GET_ALONG', True)
        elif homeRoom and homeRoom.contains(self) and not self.isSittingOnChair():
            gaveLines = self.alertLinesGivers('SIT_DOWN')
        elif self.isSittingOnFloor() or self.isKnockedOut():
            gaveLines = self.alertLinesGivers('GET_UP')
        if gaveLines:
            self.linesDelay = 40

    def isAbsent(self):
        """Returns whether Eric is playing truant."""
        return self.skool.getHomeRoom() not in (None, self.skool.room(self))

    def shouldGetAlong(self):
        """Returns whether Eric is somewhere other than he should be."""
        return self.skool.shouldGetAlong(self)

    def makeWalkingSound(self, index):
        self.beeper.makeWalkingSound(index)

    def walk(self, onStairs=False):
        Character.walk(self, onStairs)
        if not self.midstride():
            return self.screen.getScrollIncrement(self.x)
        return 0

    def dethrone(self):
        # Allow time for Eric to be drawn on the floor before lines are given
        self.linesDelay = max(self.linesDelay, 2)
        self.sitOnFloor()
        self.beeper.makeKnockedOutSound()

    def deck(self):
        # Allow time for Eric to be drawn on the floor before lines are given
        self.linesDelay = max(self.linesDelay, 2)
        self.knockOver()
        self.beeper.makeKnockedOutSound()

    def alertLinesGivers(self, messageId, truant=False):
        nearbyLinesGivers = self.cast.getNearbyLinesGivers(self)
        if nearbyLinesGivers:
            teacher = self.skool.getTeacher()
            if truant and teacher in nearbyLinesGivers:
                teacher.giveLines(self.characterId, teacher.getComeAlongMessageId())
            else:
                nearbyLinesGivers[0].giveLines(self.characterId, messageId)
        return len(nearbyLinesGivers) > 0

    def fireCatapult(self):
        Character.fireCatapult(self)
        self.alertLinesGivers('NO_CATAPULTS')
        self.beeper.makeCatapultSound()

    def punch(self):
        Character.punch(self)
        self.alertLinesGivers('NO_HITTING')

    def jump(self):
        self.previousAS = self.animatoryState
        self.y -= 1
        self.walkDelay = 4
        self.animatoryState = 'ARM_UP'

    def getLocationAboveHand(self):
        return (self.x + self.direction + 1, self.y)

    def descend(self):
        """Makes Eric finish his jump."""
        self.alertLinesGivers('NO_JUMPING')
        self.skool.checkSafe(*self.getLocationAboveHand())
        self.checkShields()
        if not self.cast.isStandingOnKid(self):
            self.y += 1
        self.animatoryState = self.previousAS

    def checkShields(self):
        self.checkShieldsAt(*self.getLocationAboveHand())

    def write(self):
        """Controls Eric when he's writing on a board. Returns True if Eric has
        finished writing."""
        if self.hasArmRaised():
            self.lowerArm()
            return
        blackboard = self.skool.room(self).getBlackboard()
        keyDownEvents = self.keyboard.getKeyDownEvents()
        if keyDownEvents:
            firstEvent = keyDownEvents[0]
            if firstEvent.key == pygame.K_RETURN:
                self.raiseArm()
                self.keyboard.finishWriting()
                return True
            char = firstEvent.unicode
            if char:
                self.raiseArm()
                self.skool.writeOnBoard(self, blackboard, char)
        else:
            self.walkDelay = 1 # Maximise responsiveness (hack)

    def isEric(self):
        return True

    def freeze(self):
        self.controller = Freeze()
        self.frozen = True

    def unfreeze(self):
        if self.frozen:
            self.controller = None
            self.frozen = False
            self.understood = False

    def checkUnderstanding(self):
        if self.keyboard.wasPressed(pygame.K_u):
            self.understood = True

    def understoodMessage(self):
        return self.understood
