# Copyright 2008 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import sys
import os
import pygame
import random
from cast import Cast
from skool import Skool, Timetable
from graphics import Screen, Gallery
from sound import Beeper
from input import Keyboard
from skoolbuilder import SkoolBuilder

class Game:
    def __init__(self, pyskoolDir, scale, iniFile, quickStart):
        builder = SkoolBuilder(iniFile)
        config = builder.getGameConfig()

        imagesDir = os.path.join(pyskoolDir, config['ImagesDir'])
        imageSet = config['ImageSet']
        scale = scale or config['Scale']
        galleryConfig = builder.getGalleryConfig()
        gallery = Gallery(imagesDir, imageSet, scale, galleryConfig)

        self.createScreen(config, scale, gallery)
        self.createSkool(pyskoolDir, config, gallery, builder)
        self.screen.initialiseColumn(config['InitialColumn'], self.skool)

        self.maxLines = config['MaxLines'] // 10
        self.quickStart = quickStart if quickStart is not None else config['QuickStart']
        self.turboMode = False

    def createScreen(self, config, scale, gallery):
        self.screen = Screen(config['GraphicsMode'], scale, config['Background'], gallery)
        self.screen.setLessonBoxProperties(config['LessonBoxAttribute'], config['LessonBoxPos'])
        self.screen.addLogo(config['LogoPos'])
        self.screen.addScoreBox(config['ScoreBoxPos'], config['ScoreBoxXOffset'], config['ScoreBoxAttribute'])

    def createSkool(self, pyskoolDir, config, gallery, builder):
        beeper = Beeper(os.path.join(pyskoolDir, config['SoundsDir']))
        cast = Cast(gallery.getSprites())
        timetable = Timetable(config['LessonLength'], config['LessonStartTime'])
        self.skool = Skool(self, self.screen, beeper, cast, timetable, gallery)
        builder.buildSkool(self.skool)
        self.skool.initialiseCast()

    def checkCheatKeys(self, keyboard):
        if keyboard.wasPressed(pygame.K_l):
            self.skool.nextLesson(False)
        self.turboMode = keyboard.isPressed(pygame.K_a)
        if keyboard.wasPressed(pygame.K_2):
            for shield in self.skool.shields:
                shield.flash()
            self.skool.safe.unflash()
            self.skool.shieldMode = 2
        if keyboard.wasPressed(pygame.K_3):
            for shield in self.skool.shields[:-1]:
                shield.unflash()
            self.skool.shields[-1].flash()
            self.skool.safe.flash()
            self.skool.shieldMode = 3
        if keyboard.wasPressed(pygame.K_n):
            for shield in self.skool.shields[:-1]:
                shield.flash()
        if keyboard.wasPressed(pygame.K_m):
            for shield in self.skool.shields[:-1]:
                shield.unflash()
        if keyboard.wasPressed(pygame.K_x):
            self.skool.addLines(10 * random.randrange(1, 9))
        if keyboard.wasPressed(pygame.K_c):
            sys.stderr.write('Safe: %s\n' % self.skool.safeCombination)
        if keyboard.wasPressed(pygame.K_y):
            for c in self.skool.cast.characterList:
                if c.specialAnswer:
                    sys.stderr.write('%s: %s\n' % (c.name, c.specialAnswer))

    def play(self, cheat=False):
        clock = pygame.time.Clock()

        self.ringBell = False

        self.screenshot = 0
        self.paused = False
        keyboard = Keyboard()
        while True:
            if not self.quickStart:
                self.ringBell = True
                self.skool.scrollOn(clock)

            self.quickStart = False
            self.gameOver = False

            while not self.gameOver:
                if self.mainLoop(keyboard, clock, cheat):
                    return

            self.skool.reinitialise()

    def mainLoop(self, keyboard, clock, cheat):
        if keyboard.gotQuit() or keyboard.wasPressed(pygame.K_q):
            return True

        spacePressed = keyboard.wasPressed(pygame.K_SPACE)
        if self.paused:
            self.paused = not spacePressed
        else:
            self.paused = spacePressed
        if self.paused:
            clock.tick(20)
            keyboard.pump()
            return False

        if self.skool.tick():
            self.skool.nextLesson(self.ringBell)
            self.ringBell = True

        if cheat:
            self.checkCheatKeys(keyboard)

        if keyboard.wasPressed(pygame.K_k):
            self.screen.takeScreenshot('pyskool%s.bmp' % str(self.screenshot).zfill(3))
            self.screenshot += 1

        scroll = self.skool.moveCharacters(keyboard, self.turboMode)
        self.skool.autoShutDoors()

        if self.turboMode:
            self.skool.tick()
        else:
            clock.tick(20)

        self.skool.draw()
        self.screen.update()
        self.skool.scroll(scroll, clock)

    def getMaxLines(self):
        return self.maxLines

    def end(self):
        self.gameOver = True
