# -*- coding: utf-8 -*-
# Copyright 2008, 2010 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

from character import Character
from animatorystates import *
from ai import Write, Jump, Hit, FireCatapult, Freeze, Catch, FireWaterPistol, DumpWaterPistol, DropStinkbomb, RideBike, FallToFloor, JumpOutOfWindow, ClimbOverSkoolGate, Kiss, ReleaseMice
import lines
import items
import bike
import keys

LINES_DELAY = 60

class Eric(Character):
    def __init__(self, character_id, name, flags):
        Character.__init__(self, character_id, name, flags)
        self._reinitialise()

    def reinitialise(self):
        Character.reinitialise(self)
        self._reinitialise()

    def _reinitialise(self):
        self.writing = False
        self.frozen = False
        self.understood = False
        self.controller = None
        self.lines_delay = LINES_DELAY
        self.inventory = set()
        self.mice = 0
        self.last_bike_key = None
        self.bike_key = None
        self.on_bike = False
        self.started_pedalling = False
        self.sitting_on_saddle = False

    def move(self, keyboard):
        self.keyboard = keyboard
        self.get_lines()
        self.walk_delay -= 1
        if self.walk_delay > 0:
            return 0
        self.barrier = self.skool.barrier(self)
        self.floor = self.get_floor()
        self.keyboard.pump()
        self.walk_delay = 2
        if self.controller:
            next_controller = self.controller.command(self)
            if next_controller is self.controller:
                self.controller = None
            elif next_controller is not None:
                self.controller = next_controller
            return self.screen.get_scroll_increment(self.x)
        if self.midstride():
            self.make_walking_sound(self.get_walk_state_index())
            return self.walk()
        if self.is_knocked_out() or self.is_sitting():
            if self.keyboard.was_pressed(keys.SIT_STAND):
                if self.is_sitting_on_chair():
                    self.chair().vacate()
                self.animatory_state = WALK0
                self.beeper.make_sitting_sound()
            elif self.is_sitting_on_chair() and self.keyboard.was_pressed(keys.OPEN_DESK):
                self.open_desk(self.skool.desk(self))
            return 0
        if self.keyboard.was_pressed(keys.SIT_STAND):
            self.sit()
            self.beeper.make_sitting_sound()
            return 0
        if self.keyboard.was_pressed(keys.FIRE_CATAPULT) and self.can_fire_catapult():
            self.controller = FireCatapult()
            return 0
        if self.keyboard.was_pressed(keys.FIRE_WATER_PISTOL) and self.can_fire_water_pistol():
            self.controller = FireWaterPistol()
            return 0
        if self.keyboard.was_pressed(keys.DUMP_WATER_PISTOL) and self.can_dump_water_pistol():
            self.bend_over()
            self.controller = DumpWaterPistol()
            return 0
        if self.keyboard.was_pressed(keys.DROP_STINKBOMB) and self.can_drop_stinkbomb():
            self.controller = DropStinkbomb()
            return 0
        if self.keyboard.was_pressed(keys.HIT):
            self.controller = Hit()
            return 0
        if self.keyboard.was_pressed(keys.JUMP):
            self.controller = Jump()
            return 0
        if self.keyboard.was_pressed(keys.WRITE) and self.skool.beside_blackboard(self):
            self.raise_arm()
            self.controller = Write()
            self.keyboard.start_writing()
            self.writing = True
            return 0
        if self.keyboard.was_pressed(keys.CATCH) and self.can_bend_over():
            self.bend_over()
            self.controller = Catch()
            return 0
        if self.keyboard.was_pressed(keys.MOUNT_BIKE) and self.can_mount_bike():
            self.mount_bike()
            self.controller = RideBike(self.cast.bike)
            return 0
        if self.keyboard.was_pressed(keys.KISS) and self.can_kiss():
            self.previous_as = self.animatory_state
            self.controller = Kiss()
            return 0
        if self.keyboard.was_pressed(keys.RELEASE_MICE) and self.can_release_mice():
            self.bend_over()
            self.controller = ReleaseMice()
            return 0
        self.staircase = self.skool.staircase(self)
        if not self.is_supported():
            self.y += 1
            return 0
        old_ws, old_as, old_direction = self.get_walk_state_index(), self.animatory_state, self.direction
        plant = self.cast.plant(self)
        if self.keyboard.pressed(keys.LEFT):
            if plant and plant.is_fully_grown():
                barrier = self.skool.barrier(self, 1)
                if barrier and barrier.barrier_id == 'SkoolGate' and barrier.is_shut():
                    self.controller = ClimbOverSkoolGate()
                    return 0
            self.left()
        elif self.keyboard.pressed(keys.RIGHT):
            if plant and plant.is_fully_grown():
                window = self.skool.window(self)
                if window and not window.is_shut():
                    self.controller = JumpOutOfWindow(window.barrier_id)
                    return 0
                elif self.barrier and self.barrier.barrier_id == 'SkoolGate' and self.barrier.is_shut():
                    self.controller = ClimbOverSkoolGate()
                    return 0
            self.right()
        elif self.keyboard.pressed(keys.UP):
            self.up()
        elif self.keyboard.pressed(keys.DOWN):
            self.down()
        if (self.animatory_state, self.direction) != (old_as, old_direction):
            self.make_walking_sound(old_ws)
        return 0

    def can_fire_catapult(self):
        return Character.can_fire_catapult(self) and not self.cast.conker_falling()

    def can_sit_on_stairs(self):
        """Return whether Eric can sit on the stairs."""
        return self.cast.can_get_lines(lines.NO_SITTING_ON_STAIRS)

    def is_supported(self):
        """Return whether Eric is standing on something that prevents him from falling."""
        return (self.staircase
            or self.cast.is_standing_on_kid(self)
            or self.skool.on_floor(self)
            or self.skool.plant_pot(self.x, self.y)
            or self.cast.plant(self))

    def can_open_door(self, door):
        if door.barrier_id == 'ScienceLabDoor' and items.STOREROOM_KEY in self.inventory:
            return True
        return Character.can_open_door(self, door)

    def open_door(self):
        self.skool.move_door(self.barrier.barrier_id, False)
        return 0

    def get_lines(self):
        self.lines_delay = max(0, self.lines_delay - 1)
        if self.lines_delay > 0:
            return
        gave_lines = False
        home_room = self.skool.get_home_room()
        if self.is_sitting_on_stairs():
            gave_lines = self.alert_lines_givers(lines.NO_SITTING_ON_STAIRS)
        elif self.skool.in_no_go_zone(self):
            gave_lines = self.alert_lines_givers(lines.GET_OUT)
        elif self.should_get_along():
            gave_lines = self.alert_lines_givers(lines.GET_ALONG, True)
        elif home_room and home_room.contains(self) and home_room.chairs and not self.is_sitting_on_chair():
            gave_lines = self.alert_lines_givers(lines.SIT_DOWN)
        elif self.is_sitting_on_floor() or self.is_knocked_out():
            gave_lines = self.alert_lines_givers(lines.GET_UP)
        elif self.on_bike:
            gave_lines = self.alert_lines_givers(lines.NO_BIKES)
        if gave_lines:
            self.lines_delay = LINES_DELAY

    def is_absent(self):
        """Return whether Eric is playing truant."""
        return self.skool.get_home_room() not in (None, self.skool.room(self))

    def should_get_along(self):
        """Return whether Eric is somewhere other than he should be."""
        return self.skool.should_get_along(self)

    def make_walking_sound(self, index):
        self.beeper.make_walking_sound(index)

    def walk(self, on_stairs=False):
        Character.walk(self, on_stairs)
        if not self.midstride():
            return self.screen.get_scroll_increment(self.x)
        return 0

    def dethrone(self):
        self.sit_on_floor()
        self.beeper.make_knocked_out_sound()

    def deck(self):
        if self.is_sitting_on_chair():
            self.chair().vacate()
        self.knock_over()
        if self.writing:
            self.writing = False
            self.keyboard.finish_writing()
            self.controller = None
        self.beeper.make_knocked_out_sound()

    def alert_lines_givers(self, message_id, truant=False):
        nearby_lines_givers = self.cast.get_nearby_lines_givers(self)
        if nearby_lines_givers:
            teacher = self.skool.get_teacher()
            if truant and teacher in nearby_lines_givers:
                teacher.give_lines(self.character_id, teacher.get_come_along_message_id())
            else:
                nearby_lines_givers[0].give_lines(self.character_id, message_id)
        return len(nearby_lines_givers) > 0

    def aim_catapult(self):
        Character.aim_catapult(self)
        self.alert_lines_givers(lines.NO_CATAPULTS)
        self.beeper.make_catapult_sound()

    def has_water_pistol(self):
        return items.WATER_PISTOL in self.inventory or items.SHERRY_PISTOL in self.inventory

    def can_fire_water_pistol(self):
        if self.has_water_pistol():
            return Character.can_fire_water_pistol(self)

    def can_dump_water_pistol(self):
        return self.has_water_pistol() and self.can_bend_over()

    def dump_water_pistol(self):
        pistol = None
        if items.WATER_PISTOL in self.inventory:
            pistol = items.WATER_PISTOL
        elif items.SHERRY_PISTOL in self.inventory:
            pistol = items.SHERRY_PISTOL
        if pistol:
            self.inventory.remove(pistol)
            self.print_inventory()
            self.skool.hide_in_desk(items.WATER_PISTOL)

    def fire_water_pistol(self):
        liquid = 'SHERRY' if items.SHERRY_PISTOL in self.inventory else 'WATER'
        Character.fire_water_pistol(self, liquid)
        self.alert_lines_givers(lines.NO_WATERPISTOLS)
        self.beeper.make_catapult_sound()

    def can_drop_stinkbomb(self):
        return self.floor and self.has_stinkbomb() and Character.can_drop_stinkbomb(self)

    def has_stinkbomb(self):
        return any([i in self.inventory for i in (items.STINKBOMBS1, items.STINKBOMBS2, items.STINKBOMBS3)])

    def drop_stinkbomb(self):
        Character.drop_stinkbomb(self)
        if items.STINKBOMBS3 in self.inventory:
            self.inventory.remove(items.STINKBOMBS3)
            self.inventory.add(items.STINKBOMBS2)
        elif items.STINKBOMBS2 in self.inventory:
            self.inventory.remove(items.STINKBOMBS2)
            self.inventory.add(items.STINKBOMBS1)
        elif items.STINKBOMBS1 in self.inventory:
            self.inventory.remove(items.STINKBOMBS1)
        self.print_inventory()
        self.alert_lines_givers(lines.NO_STINKBOMBS)

    def open_desk(self, desk):
        if desk:
            self.cast.open_desk(self, desk)

    def collect_desk_contents(self, desk):
        if desk.contents:
            self.inventory.add(desk.contents)
            if desk.contents == items.WATER_PISTOL:
                desk.empty()
            self.beeper.make_bingo_sound()
            self.print_inventory()

    def punch(self):
        Character.punch(self)
        self.alert_lines_givers(lines.NO_HITTING)

    def jump(self):
        self.previous_as = self.animatory_state
        self.y -= 1
        self.walk_delay = 4
        self.animatory_state = ARM_UP
        self.alert_lines_givers(lines.NO_JUMPING)

    def can_bend_over(self):
        return BENDING_OVER in self.as_dict_L

    def bend_over(self):
        self.previous_as = self.animatory_state
        self.walk_delay = 1
        self.animatory_state = BENDING_OVER

    def catch_animal(self):
        animal = self.cast.get_animal(self)
        if animal:
            if animal.is_mouse():
                self.beeper.make_mouse_sound()
                self.mice += 1
                self.print_mouse_inventory()
                self.cast.caught_mouse(animal)
            elif animal.is_frog():
                animal.hide()
                self.beeper.make_bingo_sound()
                self.inventory.add(items.FROG)
                self.print_inventory()
        self.walk_delay = 4

    def print_mouse_inventory(self):
        self.screen.print_mice(self.mice, self.skool.get_mouse_image())

    def can_release_mice(self):
        return self.floor and self.mice > 0 and self.can_bend_over()

    def release_mice(self):
        if self.mice > 0:
            num_mice = min(self.mice, 5)
            self.mice -= num_mice
            self.print_mouse_inventory()
            self.cast.release_mice(num_mice, self.x + self.direction, self.y)

    def print_inventory(self):
        self.screen.print_inventory(self.skool.get_inventory_images(self.inventory))

    def stand_up(self):
        self.animatory_state = self.previous_as
        self.walk_delay = 1

    def get_location(self):
        """Return the on-floor location that is closest to Eric."""
        staircase = self.skool.staircase(self)
        if staircase and staircase.supports(self):
            if staircase.bottom.y - self.y < self.y - staircase.top.y:
                return staircase.bottom.coords()
            return staircase.top.coords()
        return self.x, self.skool.floor_below(self).y

    def get_location_above_hand(self):
        return (self.x + self.direction + 1, self.y)

    def descend(self):
        """Make Eric finish his jump."""
        x, y = self.get_location_above_hand()
        self.skool.check_safe(x, y, items.SAFE_KEY in self.inventory)
        self.check_shields_at(x, y)
        if self.skool.check_drinks_cabinet(x, y) and items.WATER_PISTOL in self.inventory:
            self.inventory.remove(items.WATER_PISTOL)
            self.inventory.add(items.SHERRY_PISTOL)
            self.print_inventory()
            self.beeper.make_sherry_sound()
        if not self.cast.is_standing_on_kid(self) and not self.skool.plant_pot(self.x, self.y):
            self.y += 1
        self.animatory_state = self.previous_as

    def write(self):
        """Controls Eric when he's writing on a board. Returns True if Eric has
        finished writing."""
        blackboard = self.skool.room(self).get_blackboard()
        if self.has_arm_raised():
            self.lower_arm()
            self.writing = self.keyboard.writing
            if self.writing:
                self.alert_lines_givers(lines.NO_WRITING)
            else:
                self.check_combinations(blackboard)
            return not self.writing
        key_down_events = self.keyboard.key_down_events
        if key_down_events:
            first_event = key_down_events[0]
            if first_event.key == keys.ENTER:
                self.raise_arm()
                self.keyboard.finish_writing()
                return
            char = first_event.unicode
            if self.skool.font.has_char(char):
                self.raise_arm()
                self.skool.write_on_board(self, blackboard, char)
        else:
            self.walk_delay = 1 # Maximise responsiveness (hack)

    def check_combinations(self, blackboard):
        if self.skool.got_bike_combination(blackboard):
            self.skool.unchain_bike()
            self.beeper.make_bingo_sound()
        elif self.skool.got_storeroom_combination(blackboard):
            self.cast.release_frogs()
            self.inventory.add(items.STOREROOM_KEY)
            self.print_inventory()
            self.beeper.make_bingo_sound()

    def is_eric(self):
        return True

    def freeze(self):
        if not self.writing:
            self.controller = Freeze()
            self.frozen = True
        return self.frozen

    def unfreeze(self):
        if self.frozen:
            self.controller = None
            self.frozen = False
            self.understood = False

    def check_understanding(self):
        if self.keyboard.was_pressed(keys.UNDERSTOOD):
            self.understood = True

    def understood_message(self):
        return self.understood

    def take_safe_key(self):
        self.beeper.make_safe_key_sound()
        self.inventory.add(items.SAFE_KEY)
        self.print_inventory()

    def can_kiss(self):
        return self.cast.has_kissees()

    def kissee(self):
        return self.cast.kissee()

    def kiss(self):
        self.skool.add_lines(-100)
        self.beeper.make_bingo_sound()

    def can_mount_bike(self):
        return RIDING_BIKE0 in self.as_dict_L and self.cast.is_beside_bike(self)

    def mount_bike(self):
        self.last_bike_key = None
        self.on_bike = True
        self.started_pedalling = False
        self.sitting_on_saddle = True
        self.previous_as = self.animatory_state
        self.animatory_state = RIDING_BIKE0

    def check_bike_keys(self):
        self.walk_delay = bike.SPEED
        self.bike_key = None
        for key in (keys.LEFT, keys.RIGHT, keys.UP, keys.DOWN, keys.JUMP):
            if self.keyboard.was_pressed(key):
                self.bike_key = key
                break

    def rode_into_barrier(self, barrier_id=None):
        if self.barrier:
            if barrier_id:
                return self.barrier.barrier_id == barrier_id
            return True
        return False

    def pedalled(self):
        if self.sitting_on_saddle:
            pedalled = self.bike_key in (keys.LEFT, keys.RIGHT) and self.bike_key != self.last_bike_key
            if pedalled:
                self.started_pedalling = True
                self.pedal()
                self.last_bike_key = self.bike_key
            if self.started_pedalling:
                self.move_bike()
            return pedalled
        return False

    def pedal(self):
        if self.animatory_state == RIDING_BIKE0:
            self.animatory_state = RIDING_BIKE1
        else:
            self.animatory_state = RIDING_BIKE0

    def move_bike(self):
        self.x += self.direction

    def dismounted(self):
        return self.sitting_on_saddle and self.bike_key == keys.DOWN

    def dismount(self):
        self.on_bike = False

    def stood_on_saddle(self):
        if self.sitting_on_saddle and self.bike_key == keys.UP:
            self.last_bike_key = keys.UP
            self.sitting_on_saddle = False
        return not self.sitting_on_saddle

    def stand_on_saddle(self, bike):
        self.animatory_state = WALK0
        self.x = bike.x
        self.y = bike.y - 1

    def got_back_on_saddle(self):
        if not self.sitting_on_saddle and self.bike_key == keys.DOWN:
            self.last_bike_key = keys.DOWN
            self.sitting_on_saddle = True
        return self.sitting_on_saddle

    def jumped(self):
        return not self.sitting_on_saddle and self.bike_key in (keys.JUMP, keys.UP)

    def check_cup(self):
        if items.FROG in self.inventory:
            cup = self.skool.cup(*self.get_location_above_hand())
            if cup:
                self.cast.insert_frog(cup)
                self.inventory.remove(items.FROG)
                self.print_inventory()
                self.beeper.make_bingo_sound()

    def fall_to_floor(self):
        self.sit_on_floor()
        self.beeper.make_knocked_out_sound()

    def fall_off_plant(self):
        self.controller = FallToFloor()

    def fly(self, x_inc, y_inc):
        self.x += x_inc * self.direction
        self.y += y_inc
        self.make_walking_sound(self.x % 4)

    def end_flight(self):
        if not self.is_standing():
            self.beeper.make_knocked_out_sound()
