# -*- coding: utf-8 -*-
# Copyright 2008-2010 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import pygame
import random
import items

class Skool:
    def __init__(self, game, screen, beeper, cast, timetable, gallery):
        self.game = game
        self.screen = screen
        self.beeper = beeper
        self.scale = screen.scale
        self.mode = screen.mode
        self.skool, self.ink, self.paper = gallery.get_skool(self.mode)
        self.mutables, self.mutables_ink, self.mutables_paper = gallery.get_mutables()
        self.inventory = gallery.get_inventory()
        self.inventory_images = {}
        self.inventory_item_ids = []
        self.font = screen.font
        self.cast = cast
        self.timetable = timetable
        self.scoreboard = Scoreboard(screen)
        self.locations = {}
        self.rooms = {}
        self.doors = {}
        self.windows = {}
        self.walls = {}
        self.barriers = {} # Doors, windows and walls
        self.staircases = {}
        self.floors = {}
        self.routes = {}
        self.no_go_zones = []
        self.assembly_message_generator = AssemblyMessageGenerator()
        self.shields = []
        self.shield_mode = 1
        self.safe_combination = None
        self.bike_combination = None
        self.storeroom_combination = None
        self.draw_index = 0
        self.signals = Signals()
        self.messages = {}
        self.expelled = False
        self.cups = {}
        self.plant_pots = {}
        self.bike = None
        self.clear_score = True

    def add_location(self, location_id, coords):
        self.locations[location_id] = coords

    def add_room(self, room_id, name, y, min_x, max_x, get_along):
        self.rooms[room_id] = Room(room_id, name, y, min_x, max_x, get_along)

    def add_chair(self, room_id, x):
        self.rooms[room_id].add_chair(x)

    def add_desk(self, room_id, x):
        self.rooms[room_id].add_desk(x)

    def add_door(self, door_id, x, bottom_y, top_y, initially_shut, auto_shuts, shut_top_left, size, coords):
        door = Door(door_id, x, bottom_y, top_y, initially_shut, auto_shuts)
        self.doors[door_id] = door
        self.barriers[door_id] = door
        size = self.scale_coords(size)
        shut_top_left = self.scale_coords(shut_top_left)
        open_top_left = (shut_top_left[0] + size[0], shut_top_left[1])
        if self.mode == 0:
            shut_images = (self.mutables.subsurface(shut_top_left, size),)
            open_images = (self.mutables.subsurface(open_top_left, size),)
        else:
            shut_image_ink = self.mutables_ink.subsurface(shut_top_left, size)
            shut_image_paper = self.mutables_paper.subsurface(shut_top_left, size)
            shut_images = (shut_image_ink, shut_image_paper)
            open_image_ink = self.mutables_ink.subsurface(open_top_left, size)
            open_image_paper = self.mutables_paper.subsurface(open_top_left, size)
            open_images = (open_image_ink, open_image_paper)
        door.set_images(open_images, shut_images, coords)

    def add_window(self, window_id, x, bottom_y, top_y, initially_shut, opener_coords, shut_top_left, size, coords):
        window = Window(window_id, x, bottom_y, top_y, initially_shut, opener_coords)
        self.windows[window_id] = window
        self.barriers[window_id] = window
        size = self.scale_coords(size)
        shut_top_left = self.scale_coords(shut_top_left)
        open_top_left = (shut_top_left[0] + size[0], shut_top_left[1])
        if self.mode == 0:
            shut_images = (self.mutables.subsurface(shut_top_left, size),)
            open_images = (self.mutables.subsurface(open_top_left, size),)
        else:
            shut_image_ink = self.mutables_ink.subsurface(shut_top_left, size)
            shut_image_paper = self.mutables_paper.subsurface(shut_top_left, size)
            shut_images = (shut_image_ink, shut_image_paper)
            open_image_ink = self.mutables_ink.subsurface(open_top_left, size)
            open_image_paper = self.mutables_paper.subsurface(open_top_left, size)
            open_images = (open_image_ink, open_image_paper)
        window.set_images(open_images, shut_images, coords)

    def add_inventory_item(self, item_id, top_left, size):
        top_left = self.scale_coords(top_left)
        size = self.scale_coords(size)
        self.inventory_images[item_id] = self.inventory.subsurface(top_left, size)
        self.inventory_item_ids.append(item_id)

    def get_inventory_images(self, item_ids):
        images = []
        for item_id in self.inventory_item_ids:
            if item_id in item_ids:
                images.append(self.inventory_images.get(item_id))
        return images

    def get_mouse_image(self):
        return self.inventory_images.get(items.MOUSE)

    def add_wall(self, wall_id, x, bottom_y, top_y):
        wall = Wall(wall_id, x, bottom_y, top_y)
        self.walls[wall_id] = wall
        self.barriers[wall_id] = wall

    def add_staircase(self, staircase_id, bottom, top, force, alias):
        staircase = Staircase(bottom, top, force)
        self.staircases[staircase_id] = staircase
        if alias:
            self.staircases[alias] = staircase

    def add_floor(self, floor_id, min_x, max_x, y):
        self.floors[floor_id] = Floor(floor_id, min_x, max_x, y)

    def add_routes(self, home_floor_id, dest_floor_ids, staircase_id):
        if home_floor_id not in self.routes:
            self.routes[home_floor_id] = {}
        for dest_floor_id in dest_floor_ids:
            self.routes[home_floor_id][dest_floor_id] = staircase_id

    def add_blackboard(self, room_id, x, y):
        self.rooms[room_id].add_blackboard(x, y, self.font, self.scale)

    def add_no_go_zone(self, zone_id, min_x, max_x, bottom_y, top_y):
        self.no_go_zones.append(NoGoZone(zone_id, min_x, max_x, bottom_y, top_y))

    def get_flashable(self, subclass, x, y, score, image_index):
        if image_index is not None:
            image_x = 16 * image_index * self.scale
            image_size = self.scale_coords((1, 1))
            image = self.mutables.subsurface((image_x, 0), image_size)
            inverse_image = self.mutables.subsurface((image_x + 8 * self.scale, 0), image_size)
            return subclass(x, y, score, image, inverse_image)
        return subclass(x, y, score)

    def add_shield(self, x, y, score, image_index):
        shield = self.get_flashable(Shield, x, y, score, image_index)
        self.shields.append(shield)

    def add_safe(self, x, y, score, image_index):
        self.safe = self.get_flashable(Safe, x, y, score, image_index)

    def add_cup(self, cup_id, empty_top_left, size, coords):
        cup = Cup(cup_id, coords)
        self.cups[cup_id] = cup
        size = self.scale_coords(size)
        empty_top_left = self.scale_coords(empty_top_left)
        water_top_left = (empty_top_left[0] + size[0], empty_top_left[1])
        sherry_top_left = (water_top_left[0] + size[0], empty_top_left[1])
        if self.mode == 0:
            empty_images = (self.mutables.subsurface(empty_top_left, size),)
            water_images = (self.mutables.subsurface(water_top_left, size),)
            sherry_images = (self.mutables.subsurface(sherry_top_left, size),)
        else:
            empty_image_ink = self.mutables_ink.subsurface(empty_top_left, size)
            empty_image_paper = self.mutables_paper.subsurface(empty_top_left, size)
            empty_images = (empty_image_ink, empty_image_paper)
            water_image_ink = self.mutables_ink.subsurface(water_top_left, size)
            water_image_paper = self.mutables_paper.subsurface(water_top_left, size)
            water_images = (water_image_ink, water_image_paper)
            sherry_image_ink = self.mutables_ink.subsurface(sherry_top_left, size)
            sherry_image_paper = self.mutables_paper.subsurface(sherry_top_left, size)
            sherry_images = (sherry_image_ink, sherry_image_paper)
        cup.set_images(empty_images, water_images, sherry_images)

    def add_plant(self, plant_id, sprite_group_id, x, y, tap_id):
        plant = self.cast.add_plant(plant_id, sprite_group_id, x, y, tap_id)
        self.plant_pots[plant_id] = PlantPot(plant_id, plant, x, y - 1)

    def add_bike(self, bike_id, sprite_group_id, initial_as, location, tap_id, top_left, size, coords):
        self.cast.add_bike(bike_id, sprite_group_id, initial_as, location, tap_id)
        self.bike = Bike(*coords)
        size = self.scale_coords(size)
        unchained_top_left = self.scale_coords(top_left)
        chained_top_left = (unchained_top_left[0] + size[0], unchained_top_left[1])
        if self.mode == 0:
            unchained_images = (self.mutables.subsurface(unchained_top_left, size),)
            chained_images = (self.mutables.subsurface(chained_top_left, size),)
        else:
            unchained_image_ink = self.mutables_ink.subsurface(unchained_top_left, size)
            unchained_image_paper = self.mutables_paper.subsurface(unchained_top_left, size)
            unchained_images = (unchained_image_ink, unchained_image_paper)
            chained_image_ink = self.mutables_ink.subsurface(chained_top_left, size)
            chained_image_paper = self.mutables_paper.subsurface(chained_top_left, size)
            chained_images = (chained_image_ink, chained_image_paper)
        self.bike.set_images(unchained_images, chained_images)

    def add_message(self, message_id, message):
        self.messages[message_id] = message

    def all_shields_flashed(self):
        return all(shield.is_flashing() for shield in self.shields)

    def all_shields_unflashed(self):
        return not any(shield.is_flashing() for shield in self.shields)

    def hit_shield(self, x, y):
        for shield in self.shields:
            if (shield.x, shield.y) == (x, y):
                hit = False
                if self.shield_mode == 3 and shield.is_flashing():
                    shield.unflash()
                    hit = True
                elif self.shield_mode == 1 and not shield.is_flashing():
                    shield.flash()
                    hit = True
                if hit:
                    self.add_to_score(shield.get_score())
                    self.beeper.make_shield_sound()
                    if self.shield_mode == 1 and self.all_shields_flashed():
                        self.shield_mode = 2
                        self.add_to_score(200)
                        self.beeper.play_all_shields_tune()
                    elif self.shield_mode == 3 and self.all_shields_unflashed():
                        self.up_a_year('SkoolDaze')
                return True

    def up_a_year(self, game_id):
        if game_id == 'SkoolDaze':
            self.shield_mode = 1
            self.safe.unflash()
            self.add_to_score(500)
            self.print_up_a_year_message()
            self.initialise_safe_combo()
            self.beeper.play_tune()
            self.timetable.up_a_year()
        elif game_id == 'BackToSkool':
            self.add_to_score(self.safe.score // 10)
            self.print_up_a_year_message()
            self.beeper.play_up_a_year_tune()
            self.end_game(False)

    def print_up_a_year_message(self):
        message_lines = self.messages['NEXT_YEAR'].split('^')
        self.screen.print_lesson(*message_lines)

    def get_blackboards(self):
        return [room.blackboard for room in self.rooms.values() if room.blackboard]

    def got_combination(self):
        for board in self.get_blackboards():
            if board.shows(self.safe_combination):
                return True

    def check_safe(self, x, y, got_key):
        if (self.safe.x, self.safe.y) == (x, y):
            if got_key:
                # Must be the Back to Skool safe
                self.up_a_year('BackToSkool')
            elif self.shield_mode == 2 and self.got_combination():
                self.shield_mode += 1
                self.safe.flash()
                self.add_to_score(100)
                self.beeper.play_all_shields_tune()

    def check_drinks_cabinet(self, x, y):
        door = self.doors.get('DrinksCabinet')
        if door:
            return (x, y) == door.top_left and not door.shut
        return False

    def got_bike_combination(self, blackboard):
        return blackboard.shows(self.bike_combination, False)

    def got_storeroom_combination(self, blackboard):
        return blackboard.shows(self.storeroom_combination, False)

    def can_reveal_safe_secret(self):
        return self.shield_mode == 2

    def get_room(self, room_id):
        return self.rooms.get(room_id, None)

    def get_door(self, door_id):
        return self.doors.get(door_id, None)

    def visible_blackboard(self, character):
        """Returns the blackboard that is in the character's line of sight, or
        None if there is none."""
        for room in self.rooms.values():
            board = room.blackboard
            if board and room.y == character.y and character.is_facing(board) and self.line_of_sight_between(character, board):
                return board

    def beside_blackboard(self, character):
        for room in self.rooms.values():
            if room.beside_blackboard(character):
                return True

    def room(self, character):
        for room in self.rooms.values():
            if room.contains(character):
                return room

    def staircase(self, character, distance=0):
        for staircase in self.staircases.values():
            if staircase.contains(character, distance):
                return staircase

    def next_staircase(self, home_floor, dest_floor):
        if home_floor is dest_floor:
            return None
        routes = self.routes[home_floor.floor_id]
        staircase_id = routes.get(dest_floor.floor_id, routes['*'])
        return self.staircases[staircase_id]

    def floor(self, thing):
        for floor in self.floors.values():
            if floor.supports(thing):
                return floor

    def floor_below(self, character):
        floor = None
        for f in self.floors.values():
            if f.below(character):
                if floor is None or f.y < floor.y:
                    floor = f
        return floor

    def on_floor(self, character):
        return any(floor.supports(character) for floor in self.floors.values())

    def floor_at(self, x, y):
        for floor in self.floors.values():
            if floor.contains_location(x, y):
                return floor

    def in_no_go_zone(self, character):
        return any(zone.contains(character) for zone in self.no_go_zones)

    def barrier(self, character, distance=0):
        for barrier in self.barriers.values():
            if barrier.impedes(character, distance):
                return barrier

    def chair(self, character, direction=-1):
        room = self.room(character)
        if room:
            return room.chair(character, direction)

    def desk(self, character):
        room = self.room(character)
        if room:
            return room.desk(character)

    def cup(self, x, y):
        for cup in self.cups.values():
            if (cup.x, cup.y) == (x, y):
                return cup

    def plant_pot(self, x, y):
        for plant_pot in self.plant_pots.values():
            if (plant_pot.x, plant_pot.y) == (x, y):
                return plant_pot

    def window(self, character):
        """Returns the window (if any) in front of a character."""
        for window in self.windows.values():
            if window.impedes(character, force_shut=True):
                return window

    def nearby_window(self, character):
        for window in self.windows.values():
            x, y = window.opener_coords
            if character.y == y and abs(character.x - x) <= 16:
                return window

    def scale_coords(self, coords):
        return (8 * self.scale * coords[0], 8 * self.scale * coords[1])

    def is_door_shut(self, barrier_id):
        return self.barriers[barrier_id].is_shut()

    def move_door(self, barrier_id, shut):
        images, coords = self.barriers[barrier_id].move(shut)
        self.draw_mutable(images, coords)

    def auto_shut_doors(self):
        for barrier_id, door in self.doors.items():
            if door.auto_shut() and not self.cast.somebody_near_door(door):
                self.move_door(barrier_id, True)

    def fill_cup(self, cup_id, contents):
        images, coords = self.cups[cup_id].fill(contents)
        self.draw_mutable(images, coords)

    def unchain_bike(self):
        if self.bike:
            images, coords = self.bike.unchain()
            self.draw_mutable(images, coords)
            self.cast.unchain_bike()

    def chain_bike(self):
        if self.bike:
            images, coords = self.bike.chain()
            self.draw_mutable(images, coords)

    def draw_mutable(self, images, coords):
        scaled_coords = self.scale_coords(coords)
        if self.mode == 0:
            self.skool.blit(images[0], scaled_coords)
        else:
            self.ink.blit(images[0], scaled_coords)
            self.paper.blit(images[1], scaled_coords)

    def resolve_location_id(self, location_id):
        location_marker = 'Location:'
        if location_id.startswith(location_marker):
            character = self.get_character(location_id[len(location_marker):])
            return Location((character.x, 3 + 7 * (character.y // 7)))
        return Location(self.locations[location_id])

    def get_width(self):
        surface = self.skool or self.ink
        return surface.get_width() // (8 * self.scale)

    def get_height(self):
        surface = self.skool or self.ink
        return surface.get_height() // (8 * self.scale)

    def draw(self):
        skool_images = (self.skool, self.ink, self.paper)
        blackboard_images = [room.get_blackboard_images() for room in self.rooms.values()]
        cast_images = self.cast.get_images()
        inverse = self.draw_index > 4
        other_images = [shield.get_images(inverse) for shield in self.shields]
        other_images += [self.safe.get_images(inverse)]
        other_images += self.cast.get_speech_bubbles()
        self.screen.draw(skool_images, blackboard_images, cast_images, other_images)
        self.draw_index = (self.draw_index + 1) % 10

    def scroll_on(self, clock):
        self.screen.scroll_skool(self, clock)
        self.beeper.play_tune()

    def scroll(self, inc, clock):
        if inc != 0:
            self.screen.scroll(inc, self, clock)

    def get_assembly_message(self):
        return self.assembly_message_generator.generate_message()

    def line_of_sight_between(self, a, b):
        """Return whether there is a clear line of sight between two points (i.e. there are no walls between them)."""
        return not any(wall.separates(a, b) for wall in self.walls.values())

    def wipe_board(self, blackboard, column):
        if blackboard:
            if column == 0:
                blackboard.clear()
                return
            wiped_bit = pygame.Surface((8 * self.scale, 8 * self.scale))
            blackboard.wipe(wiped_bit, 8 * column * self.scale)

    def write_on_board(self, character, blackboard, message, index=1):
        if blackboard:
            blackboard.write(message[index - 1])
            blackboard.set_writer(character)
        return index == len(message)

    def initialise_safe_combo(self):
        letters = self.cast.initialise_safe_combo()
        if letters:
            self.safe_combination = letters[0]
            indexes = [n for n in range(1, len(letters))]
            while indexes:
                self.safe_combination += letters[indexes.pop(random.randrange(len(indexes)))]

    def initialise_bike_combo(self):
        self.bike_combination = self.cast.initialise_bike_combo()

    def initialise_storeroom_combo(self):
        self.storeroom_combination = self.cast.initialise_storeroom_combo()

    def initialise_cast(self):
        self.cast.initialise(self, self.screen, self.beeper)
        self.initialise_safe_combo()
        self.initialise_bike_combo()
        self.initialise_storeroom_combo()

    def reinitialise(self):
        self.expelled = False
        self.screen.reinitialise()
        self.signals.clear()
        self.cast.reinitialise()
        self.initialise_safe_combo()
        self.initialise_bike_combo()
        self.initialise_storeroom_combo()
        self.timetable.reinitialise()
        if self.clear_score:
            self.scoreboard.reinitialise()
        for room in self.rooms.values():
            room.wipe_blackboard()
        for shield in self.shields:
            shield.unflash()
        self.fill_desks()
        for cup_id in self.cups:
            self.fill_cup(cup_id, None)
        for door in self.doors.values():
            self.move_door(door.barrier_id, door.initially_shut)
        for window in self.windows.values():
            self.move_door(window.barrier_id, door.initially_shut)
        self.chain_bike()

    def get_desks(self):
        desks = []
        for room in self.rooms.values():
            desks.extend(room.desks)
        return desks

    def hide_in_desk(self, item):
        desks = self.get_desks()
        while desks:
            desk = desks.pop(random.randrange(len(desks)))
            if not desk.contents:
                desk.insert(item)
                break

    def fill_desks(self):
        desks = self.get_desks()
        if not desks:
            return
        for desk in desks:
            desk.empty()
        desks.pop(random.randrange(len(desks))).insert(items.WATER_PISTOL)
        if desks:
            random.choice(desks).insert(items.STINKBOMBS3)

    def get_eric(self):
        return self.cast.eric

    def get_character(self, character_id):
        return self.cast.get(character_id)

    def move_characters(self, keyboard):
        return self.cast.move(keyboard)

    def tick(self):
        return self.timetable.tick()

    def is_time_to_start_lesson(self):
        return self.timetable.is_time_to_start_lesson()

    def is_time_remaining(self, time):
        return self.timetable.is_time_remaining(time)

    def is_teaching_eric(self, character):
        return self.timetable.is_teaching_eric(character)

    def get_teacher(self):
        """Returns Eric's teacher (or None if it's Playtime or Revision
        Library)."""
        return self.cast.get(self.timetable.get_teacher_id())

    def set_home_room(self):
        """Sets the room Eric's supposed to be in at the moment."""
        self.home_room = self.get_room(self.timetable.get_room_id())

    def get_home_room(self):
        """Returns the room Eric's supposed to be in at the moment."""
        return self.home_room

    def should_get_along(self, eric):
        """Return whether Eric is somewhere other than he should be."""
        if self.home_room:
            return not self.home_room.contains(eric)
        if self.timetable.is_time(200):
            # Give Eric time to leave a classroom
            room = self.room(eric)
            if room:
                destination = self.get_room(self.timetable.get_room_id())
                return room != destination and room.get_along
        return False

    def get_lesson_desc(self):
        teacher_id = self.timetable.get_teacher_id()
        if self.timetable.hide_teacher():
            teacher_id = ''
        room_id = self.timetable.get_room_id()
        room = self.get_room(room_id)
        teacher = self.get_character(teacher_id)
        return teacher.name if teacher else teacher_id, room.name if room else room_id

    def print_lesson(self):
        teacher, room = self.get_lesson_desc()
        if not teacher:
            elements = room.split(' ')
            if len(elements) > 1:
                teacher = elements[0]
                room = elements[1]
        self.screen.print_lesson(teacher, room)

    def next_lesson(self, ring_bell):
        self.timetable.next_lesson()
        self.cast.set_lesson(self.timetable.get_lesson_id())
        self.print_lesson()
        self.signals.clear()
        self.home_room = None
        if ring_bell:
            self.beeper.ring_bell()

    def set_lesson(self, lesson):
        self.lesson = lesson

    def get_lesson(self):
        return self.lesson

    def add_to_score(self, addend):
        self.scoreboard.add_to_score(addend)

    def add_lines(self, addend):
        if not self.expelled:
            self.scoreboard.add_lines(addend)
            if self.scoreboard.get_lines() > self.game.get_max_lines():
                self.expelled = True
                self.stop_clock()
                self.cast.expel_eric()

    def is_eric_expelled(self):
        return self.expelled

    def stop_clock(self):
        self.timetable.stop()

    def start_clock(self, ticks):
        self.timetable.resume(ticks)

    def rewind_clock(self, ticks):
        self.timetable.rewind(ticks)

    def get_too_many_lines_message(self):
        return self.cast.expand_names(self.messages['TOO_MANY_LINES'])

    def get_understand_message(self):
        return self.messages['UNDERSTOOD']

    def signal(self, signal):
        self.signals.signal(signal)

    def unsignal(self, signal):
        self.signals.unsignal(signal)

    def got_signal(self, signal):
        return self.signals.is_raised(signal)

    def end_game(self, clear_score=True):
        self.clear_score = clear_score
        self.game.end()

class Location:
    def __init__(self, coords):
        self.x = coords[0]
        self.y = coords[1]

    def coords(self):
        return self.x, self.y

    def __str__(self):
        return '(%s, %s)' % (self.x, self.y)

class Staircase:
    def __init__(self, bottom, top, force=False):
        self.bottom = Location(bottom)
        self.top = Location(top)
        # Whether character *must* go up or down staircase when at bottom or
        # top (instead of continuing left or right)
        self.force = force
        self.direction = 1 if self.bottom.x < self.top.x else -1

    def contains(self, character, distance=0):
        """
        Returns whether the character is one of the following:
            i) on a step of this staircase
           ii) at the bottom of this staircase facing the top
          iii) at the top of this staircase facing the bottom
        """
        for x, y in [(character.x + d * character.direction, character.y) for d in range(distance + 1)]:
            if self.contains_location(x, y):
                if x == self.bottom.x:
                    return character.direction == self.direction
                if x == self.top.x:
                    return character.direction != self.direction
                return True
        return False

    def contains_location(self, x, y):
        """
        Returns whether the given location (x, y) is at the bottom, at the top,
        or on a step of this staircase.
        """
        if self.direction * self.bottom.x <= self.direction * x <= self.direction * self.top.x:
            if (x, y) in (self.bottom.coords(), self.top.coords()):
                return True
            return self.top.y < y < self.bottom.y
        return False

    def supports(self, character):
        """Returns whether the character is on a step of this staircase."""
        if not self.contains_location(character.x, character.y):
            return False
        return (character.x, character.y) not in (self.bottom.coords(), self.top.coords())

class Floor:
    def __init__(self, floor_id, left_x, right_x, y):
        self.floor_id = floor_id
        self.left_x = left_x
        self.right_x = right_x
        self.y = y

    def supports(self, character):
        return self.y == character.y and self.left_x <= character.x <= self.right_x

    def below(self, character):
        return self.y >= character.y and self.left_x <= character.x <= self.right_x

    def contains_location(self, x, y):
        return self.y == y and self.left_x <= x <= self.right_x

class Barrier:
    def __init__(self, barrier_id, x, bottom_y, top_y):
        self.barrier_id = barrier_id
        self.x = x
        self.bottom_y = bottom_y
        self.top_y = top_y

    def impedes(self, character, distance=0, force_shut=False):
        if not (force_shut or self.is_shut()):
            return False
        if character.impeded(self.bottom_y, self.top_y):
            if character.x < self.x and character.direction > 0:
                return character.x + distance >= self.x - 2
            elif character.x >= self.x and character.direction < 0:
                return character.x - distance <= self.x
        return False

    def is_shut(self):
        return True

    def is_door(self):
        return False

class Wall(Barrier):
    def separates(self, a, b):
        """Returns whether this wall blocks the view from a to b."""
        min_x = min(a.x, b.x)
        max_x = max(a.x, b.x)
        if min_x < self.x <= max_x:
            return self.top_y <= max(a.y, b.y) and self.bottom_y >= min(a.y, b.y)

class Door(Barrier):
    def __init__(self, door_id, x, bottom_y, top_y, shut, auto_shuts):
        Barrier.__init__(self, door_id, x, bottom_y, top_y)
        self.shut = self.initially_shut = shut
        self.auto_shuts = auto_shuts
        self.open_images = None
        self.shut_images = None
        self.top_left = None
        self.auto_shut_timer = 0
        self.auto_shut_delay = 40

    def is_shut(self):
        return self.shut

    def move(self, shut):
        self.shut = shut
        if not shut:
            self.auto_shut_timer = self.auto_shut_delay
        images = self.shut_images if shut else self.open_images
        return (images, self.top_left)

    def set_images(self, open_images, shut_images, top_left):
        self.open_images = open_images
        self.shut_images = shut_images
        self.top_left = top_left

    def auto_shut(self):
        if not self.shut and self.auto_shuts:
            self.auto_shut_timer -= 1
        return self.auto_shut_timer < 0

    def is_door(self):
        return True

class Window(Door):
    def __init__(self, window_id, x, bottom_y, top_y, shut, opener_coords):
        Door.__init__(self, window_id, x, bottom_y, top_y, shut, False)
        self.opener_coords = opener_coords

class Room:
    def __init__(self, room_id, name, y, min_x, max_x, get_along):
        self.room_id = room_id
        self.name = name
        self.y = y
        self.min_x = min_x
        self.max_x = max_x
        self.get_along = get_along
        self.chairs = []
        self.desks = []
        self.blackboard = None

    def get_blackboard(self):
        return self.blackboard

    def get_blackboard_writer(self):
        if self.blackboard:
            return self.blackboard.get_writer()

    def has_blackboard(self):
        return self.blackboard is not None

    def add_blackboard(self, x, y, font, scale):
        self.blackboard = Blackboard(x, y, font, scale)

    def blackboard_dirty(self):
        if self.blackboard:
            return self.blackboard.is_dirty()

    def wipe_blackboard(self):
        if self.blackboard:
            self.blackboard.clear()

    def get_blackboard_images(self):
        if self.blackboard:
            return self.blackboard.get_images()
        return (0, 0, [])

    def add_chair(self, x):
        self.chairs.append(Chair(self, x))

    def add_desk(self, x):
        self.desks.append(Desk(self, x))

    def beside_blackboard(self, character):
        if self.blackboard:
            return self.y == character.y and self.blackboard.beside(character)

    def contains(self, character):
        return self.y - 3 <= character.y <= self.y and self.min_x <= character.x <= self.max_x

    def chair(self, character, direction=-1):
        for chair in self.chairs:
            if (direction is None or character.direction == direction) and (character.x, character.y) == (chair.x, chair.y):
                return chair

    def desk(self, character):
        if character.is_sitting_on_chair():
            for desk in self.desks:
                if (character.x, character.y) == (desk.x, desk.y):
                    return desk

    def get_next_chair(self, character, move_along):
        """Returns the chair nearest to the character, or the chair he should
        sit in if he was dethroned (move_along=True) or has just started looking
        for a chair (in which case he's facing right)."""
        if character.direction > 0 or (move_along and self.chairs[0].x == character.x):
            return self.chairs[-1]
        min_distance = 100
        next_chair = None
        for chair in self.chairs:
            distance = character.x - chair.x
            if distance == 0 and move_along:
                continue
            if 0 <= distance < min_distance:
                min_distance = distance
                next_chair = chair
        return next_chair

class Chair:
    def __init__(self, room, x):
        self.x = x
        self.y = room.y
        self.room = room
        self.occupant = None

    def seat(self, character):
        self.occupant = character

    def vacate(self):
        self.occupant = None

class Desk:
    def __init__(self, room, x):
        self.x = x
        self.y = room.y
        self.room = room
        self.empty()

    def insert(self, item):
        self.contents = item

    def empty(self):
        self.contents = None

class Blackboard:
    def __init__(self, x, y, font, scale):
        self.x = x
        self.y = y
        self.font = font
        self.width = 64 * scale
        self.clear()

    def set_writer(self, character):
        self.writer = character

    def get_writer(self):
        return self.writer

    def write(self, char):
        if char == '^':
            if len(self.lines) == 1:
                self.lines.append('')
            return
        self.lines[-1] += char
        self.images = []
        for line in self.lines:
            text_image = self.font.render(line, (255, 255, 255), (0, 0, 0))
            if text_image.get_width() > self.width:
                text_image = text_image.subsurface((0, 0), (self.width, text_image.get_height()))
            self.images.append(text_image)

    def wipe(self, wiped_bit, x):
        for image in self.images:
            image.blit(wiped_bit, (x, 0))

    def is_dirty(self):
        return len(self.images) > 0

    def get_images(self):
        return (self.x, self.y, self.images)

    def clear(self):
        self.lines = ['']
        self.images = []
        self.writer = None

    def beside(self, character):
        arm_x = character.x + character.direction + 1
        return self.x + 1 <= arm_x <= self.x + 6

    def shows(self, text, in_order=True):
        if in_order:
            return self.lines[0].lower().startswith(text.lower())
        return set(text.lower()) == set(self.lines[0][:len(text)].lower())

class NoGoZone:
    def __init__(self, zone_id, min_x, max_x, bottom_y, top_y):
        self.zone_id = zone_id
        self.min_x = min_x
        self.max_x = max_x
        self.bottom_y = bottom_y
        self.top_y = top_y

    def contains(self, character):
        if self.min_x <= character.x <= self.max_x:
            return self.top_y <= character.y <= self.bottom_y
        return False

class Flashable:
    def __init__(self, x, y, score, image=None, inverse_image=None):
        self.x = x
        self.y = y
        self.score = score
        self.image = image
        self.inverse_image = inverse_image
        self.flashing = False

    def get_score(self):
        return self.score // 10

    def is_flashing(self):
        return self.flashing

    def flash(self):
        self.flashing = True

    def unflash(self):
        self.flashing = False

    def get_images(self, inverse):
        if self.flashing:
            image = self.inverse_image if inverse else self.image
            return (self.x, self.y, image)
        return (0, 0, None)

class Shield(Flashable):
    pass

class Safe(Flashable):
    pass

class Cup:
    def __init__(self, cup_id, coords):
        self.cup_id = cup_id
        self.x, self.y = coords
        self.contents = None
        self.frogs = []

    def set_images(self, empty_images, water_images, sherry_images):
        self.empty_images = empty_images
        self.water_images = water_images
        self.sherry_images = sherry_images

    def is_empty(self):
        return not (self.contents or self.frogs)

    def fill(self, contents):
        self.contents = contents
        if contents == 'WATER':
            images = self.water_images
        elif contents == 'SHERRY':
            images = self.sherry_images
        else:
            images = self.empty_images
        return images, (self.x, self.y)

    def insert_frog(self, frog):
        self.frogs.append(frog)
        frog.insert_into_cup(self)

    def remove_frog(self, frog):
        if frog in self.frogs:
            self.frogs.remove(frog)

class PlantPot:
    def __init__(self, plant_id, plant, x, y):
        self.plant_id = plant_id
        self.plant = plant
        self.x = x
        self.y = y

class Bike:
    def __init__(self, x, y):
        self.x = x
        self.y = y

    def set_images(self, unchained_images, chained_images):
        self.unchained_images = unchained_images
        self.chained_images = chained_images

    def unchain(self):
        return self.unchained_images, (self.x, self.y)

    def chain(self):
        return self.chained_images, (self.x, self.y)

class Timetable:
    def __init__(self, lesson_length, lesson_start_time):
        self.index = -1
        self.lesson_id = None
        self.lessons = []
        self.lesson_details = {}
        self.counter = 0
        self.ticking = True
        self.lesson_length = lesson_length
        self.lesson_start_time = lesson_start_time
        self.special_playtimes = []

    def reinitialise(self):
        self.counter = 0
        self.ticking = True
        self.index = (self.index & 48) + 15

    def add_lesson(self, lesson_id):
        self.lessons.append(lesson_id)

    def add_lesson_details(self, lesson_id, hide_teacher, teacher_id, room_id):
        self.lesson_details[lesson_id] = (hide_teacher, teacher_id, room_id)

    def add_special_playtime(self, lesson_id):
        self.special_playtimes.append(lesson_id)

    def next_lesson(self):
        self.ticking = True
        self.counter = self.lesson_length
        self.index = (self.index + 1) % len(self.lessons)
        self.lesson_id = self.lessons[self.index]
        if self.lesson_id.startswith('Playtime') and self.special_playtimes and random.randrange(8) < 3:
            self.lesson_id = random.choice(self.special_playtimes)

    def get_lesson_id(self):
        return self.lesson_id

    def hide_teacher(self):
        return self.lesson_details[self.get_lesson_id()][0]

    def get_teacher_id(self):
        return self.lesson_details[self.get_lesson_id()][1]

    def get_room_id(self):
        return self.lesson_details[self.get_lesson_id()][2]

    def tick(self):
        if self.ticking:
            self.counter -= 1
        return self.counter < 0

    def is_time(self, time):
        return self.counter + time < self.lesson_length

    def is_time_remaining(self, time):
        return self.counter < time

    def is_time_to_start_lesson(self):
        return self.is_time(self.lesson_start_time)

    def is_teaching_eric(self, character):
        return self.get_teacher_id() == character.character_id

    def up_a_year(self):
        self.counter = self.lesson_length // 2

    def stop(self):
        self.ticking = False

    def resume(self, ticks):
        self.counter = ticks
        self.ticking = True

    def rewind(self, ticks):
        self.counter += ticks

class AssemblyMessageGenerator:
    def __init__(self):
        self.text = None
        self.verbs = []
        self.nouns = []

    def set_text(self, text):
        self.text = text

    def add_verb(self, verb):
        self.verbs.append(verb)

    def add_noun(self, noun):
        self.nouns.append(noun)

    def generate_message(self):
        verb = random.choice(self.verbs)
        noun = random.choice(self.nouns)
        message = self.text.replace('$VERB', verb)
        return message.replace('$NOUN', noun)

class Scoreboard:
    def __init__(self, screen):
        self.screen = screen
        self.hiscore = 0
        self.reset()

    def reset(self):
        self.score = self.lines = 0

    def add_to_score(self, addend):
        self.score += addend
        self.screen.print_score(self.score)

    def add_lines(self, addend):
        self.lines = max(self.lines + addend, 0)
        self.screen.print_lines(self.lines)

    def reinitialise(self):
        if self.score > self.hiscore:
            self.hiscore = self.score
            self.screen.print_hi_score(self.hiscore)
        self.reset()
        self.screen.print_score(self.score)
        self.screen.print_lines(self.lines)

    def get_lines(self):
        return self.lines

class Signals:
    ASSEMBLY_FINISHED = 'AssemblyFinished'
    TIME_FOR_ASSEMBLY = 'TimeForAssembly'
    SWOT_READY = 'SwotReady'

    def __init__(self):
        self.clear()

    def clear(self):
        self.signals = {}

    def is_raised(self, signal):
        return self.signals.get(signal, False)

    def signal(self, signal):
        self.signals[signal] = True

    def unsignal(self, signal):
        self.signals[signal] = False
