# -*- coding: utf-8 -*-
# Copyright 2008-2010 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

class Barrier:
    def __init__(self, barrier_id, x, bottom_y, top_y):
        self.barrier_id = barrier_id
        self.x = x
        self.bottom_y = bottom_y
        self.top_y = top_y

    def impedes(self, character, distance=0, force_shut=False):
        if not (force_shut or self.is_shut()):
            return False
        if character.impeded(self.bottom_y, self.top_y):
            if character.x < self.x and character.direction > 0:
                return character.x + distance >= self.x - 2
            elif character.x >= self.x and character.direction < 0:
                return character.x - distance <= self.x
        return False

    def is_shut(self):
        return True

    def is_door(self):
        return False

class Wall(Barrier):
    def separates(self, a, b):
        """Returns whether this wall blocks the view from a to b."""
        min_x = min(a.x, b.x)
        max_x = max(a.x, b.x)
        if min_x < self.x <= max_x:
            return self.top_y <= max(a.y, b.y) and self.bottom_y >= min(a.y, b.y)

class Door(Barrier):
    def __init__(self, door_id, x, bottom_y, top_y, shut, auto_shuts):
        Barrier.__init__(self, door_id, x, bottom_y, top_y)
        self.shut = self.initially_shut = shut
        self.auto_shuts = auto_shuts
        self.open_images = None
        self.shut_images = None
        self.top_left = None
        self.auto_shut_timer = 0
        self.auto_shut_delay = 40

    def is_shut(self):
        return self.shut

    def move(self, shut):
        self.shut = shut
        if not shut:
            self.auto_shut_timer = self.auto_shut_delay
        images = self.shut_images if shut else self.open_images
        return (images, self.top_left)

    def set_images(self, open_images, shut_images, top_left):
        self.open_images = open_images
        self.shut_images = shut_images
        self.top_left = top_left

    def auto_shut(self):
        if not self.shut and self.auto_shuts:
            self.auto_shut_timer -= 1
        return self.auto_shut_timer < 0

    def is_door(self):
        return True

class Window(Door):
    def __init__(self, window_id, x, bottom_y, top_y, shut, opener_coords):
        Door.__init__(self, window_id, x, bottom_y, top_y, shut, False)
        self.opener_coords = opener_coords
