# -*- coding: utf-8 -*-
# Copyright 2008-2010 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

class Room:
    def __init__(self, room_id, name, y, min_x, max_x, get_along):
        self.room_id = room_id
        self.name = name
        self.y = y
        self.min_x = min_x
        self.max_x = max_x
        self.get_along = get_along
        self.chairs = []
        self.desks = []
        self.blackboard = None

    def get_blackboard(self):
        return self.blackboard

    def get_blackboard_writer(self):
        if self.blackboard:
            return self.blackboard.get_writer()

    def has_blackboard(self):
        return self.blackboard is not None

    def add_blackboard(self, x, y, font, scale):
        self.blackboard = Blackboard(x, y, font, scale)

    def blackboard_dirty(self):
        if self.blackboard:
            return self.blackboard.is_dirty()

    def wipe_blackboard(self):
        if self.blackboard:
            self.blackboard.clear()

    def get_blackboard_images(self):
        if self.blackboard:
            return self.blackboard.get_images()
        return (0, 0, [])

    def add_chair(self, x):
        self.chairs.append(Chair(self, x))

    def add_desk(self, x):
        self.desks.append(Desk(self, x))

    def beside_blackboard(self, character):
        if self.blackboard:
            return self.y == character.y and self.blackboard.beside(character)

    def contains(self, character):
        return self.y - 3 <= character.y <= self.y and self.min_x <= character.x <= self.max_x

    def chair(self, character, direction=-1):
        for chair in self.chairs:
            if (direction is None or character.direction == direction) and (character.x, character.y) == (chair.x, chair.y):
                return chair

    def desk(self, character):
        if character.is_sitting_on_chair():
            for desk in self.desks:
                if (character.x, character.y) == (desk.x, desk.y):
                    return desk

    def get_next_chair(self, character, move_along):
        """Returns the chair nearest to the character, or the chair he should
        sit in if he was dethroned (move_along=True) or has just started looking
        for a chair (in which case he's facing right)."""
        if character.direction > 0 or (move_along and self.chairs[0].x == character.x):
            return self.chairs[-1]
        min_distance = 100
        next_chair = None
        for chair in self.chairs:
            distance = character.x - chair.x
            if distance == 0 and move_along:
                continue
            if 0 <= distance < min_distance:
                min_distance = distance
                next_chair = chair
        return next_chair

class Chair:
    def __init__(self, room, x):
        self.x = x
        self.y = room.y
        self.room = room
        self.occupant = None

    def seat(self, character):
        self.occupant = character

    def vacate(self):
        self.occupant = None

class Desk:
    def __init__(self, room, x):
        self.x = x
        self.y = room.y
        self.room = room
        self.empty()

    def insert(self, item):
        self.contents = item

    def empty(self):
        self.contents = None

class Blackboard:
    def __init__(self, x, y, font, scale):
        self.x = x
        self.y = y
        self.font = font
        self.width = 64 * scale
        self.clear()

    def set_writer(self, character):
        self.writer = character

    def get_writer(self):
        return self.writer

    def write(self, char):
        if char == '^':
            if len(self.lines) == 1:
                self.lines.append('')
            return
        self.lines[-1] += char
        self.images = []
        for line in self.lines:
            text_image = self.font.render(line, (255, 255, 255), (0, 0, 0))
            if text_image.get_width() > self.width:
                text_image = text_image.subsurface((0, 0), (self.width, text_image.get_height()))
            self.images.append(text_image)

    def wipe(self, wiped_bit, x):
        for image in self.images:
            image.blit(wiped_bit, (x, 0))

    def is_dirty(self):
        return len(self.images) > 0

    def get_images(self):
        return (self.x, self.y, self.images)

    def clear(self):
        self.lines = ['']
        self.images = []
        self.writer = None

    def beside(self, character):
        arm_x = character.x + character.direction + 1
        return self.x + 1 <= arm_x <= self.x + 6

    def shows(self, text, in_order=True):
        if in_order:
            return self.lines[0].lower().startswith(text.lower())
        return set(text.lower()) == set(self.lines[0][:len(text)].lower())

class NoGoZone:
    def __init__(self, zone_id, min_x, max_x, bottom_y, top_y):
        self.zone_id = zone_id
        self.min_x = min_x
        self.max_x = max_x
        self.bottom_y = bottom_y
        self.top_y = top_y

    def contains(self, character):
        if self.min_x <= character.x <= self.max_x:
            return self.top_y <= character.y <= self.bottom_y
        return False
