# -*- coding: utf-8 -*-
# Copyright 2010 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

from character import Character
from animatorystates import *
from location import Location

# Minimum distance away Eric must be to be considered nearby (used to decide
# whether the frog should hop)
ERIC_PROXIMITY = 3

# Coordinates of the frog within its SIT sprite (used for placement in a cup
# and detecting collision with a character's head)
FROG_X, FROG_Y = 1, 3

class Animal(Character):
    def __init__(self, animal_id, tap_id, initial_as, location):
        Character.__init__(self, animal_id, animal_id)
        self.x, self.y = location
        self.initial_location = Location(location)
        self.tap_id = tap_id
        self.animatory_state = self.initial_as = initial_as
        self.direction = self.initial_direction = -1

    def get_tap_id(self, lesson_id):
        return self.tap_id

    def is_mouse(self):
        return False

    def is_frog(self):
        return False

    def is_blocked(self, distance=0):
        """Return True if the animal can proceed no further in the current direction, False otherwise."""
        # Check walls and closed doors
        if self.skool.barrier(self, distance):
            return True
        # Check staircases that must be ascended or descended
        staircase = self.skool.staircase(self, distance)
        if staircase and staircase.force:
            return True
        # Check whether there is a floor in front of the animal
        floor = self.skool.floor_at(self.x + self.direction * max(1, distance), self.y)
        return floor is None

    def is_eric_nearby(self):
        eric_x, eric_y = self.get_location_of_eric()
        return self.y == eric_y and abs(self.x - eric_x) <= ERIC_PROXIMITY

class Mouse(Animal):
    def __init__(self, mouse_id, tap_id, initial_as, location, immortal=True):
        Animal.__init__(self, mouse_id, tap_id, initial_as, location)
        self.immortal = immortal

    def is_mouse(self):
        return True

    def is_sitting(self):
        return True

    def scare_people(self):
        self.cast.scare_musophobes(self)

    def die(self):
        self.cast.kill_mouse(self)

class Frog(Animal):
    def __init__(self, frog_id, tap_id, initial_as, location):
        Animal.__init__(self, frog_id, tap_id, initial_as, location)
        self.falling = False
        self.trapped = False

    def is_frog(self):
        return True

    def is_sitting(self):
        return self.animatory_state == SIT

    def insert_into_cup(self, cup):
        self.trapped = True
        self.animatory_state = SIT
        self.x, self.y = cup.x - FROG_X, cup.y - FROG_Y

    def fall_from_cup(self, cup):
        cup.remove_frog(self)
        self.trapped = False
        self.falling = True

    def check_heads(self):
        return self.cast.check_heads(self)
