# -*- coding: utf-8 -*-
# Copyright 2008, 2010 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import random
import pygame
import character as char
from desklid import DeskLid
import animal
from bike import Bike
import droppable
from plant import Plant
from eric import Eric
from pellet import Pellet
from water import Water
from stinkbomb import Stinkbomb
from ai import CommandListTemplate
from location import Location
from animatorystates import *

# The transparent colour in sprites.png
SPRITE_COLORKEY = (0, 254, 0)

# Size of the sprite matrix in sprites.png
SPRITE_ROWS = 8
SPRITE_COLUMNS = 16

# Distance in front of Eric a character must be in order to be kissable
KISS_DISTANCE = 2

# Maximum distances at which a character is considered nearby (for lines-giving
# purposes)
NEARBY_X_RANGE = 10
NEARBY_Y_RANGE = 3

# Maximum distance from Eric at which a character is considered beside him
BESIDE_ERIC_X_RANGE = 4

# Locations at which a new immortal mouse may be released (after Eric catches
# one)
MOUSE_LOCATIONS = (
    (10, 3),  (88, 3),  (170, 3),
    (10, 10), (88, 10), (170, 10),
    (10, 17), (88, 17), (170, 17)
)
# ID of the command list used by mortal mice
MOUSE_COMMAND_LIST = 'Mouse'
# ID of the sprite group used by mortal mice
MOUSE_SPRITE_GROUP = 'MOUSE'

# Marker replaced by a teacher's title in EINSTEIN's speech
TITLE_MARKER = '$TITLE'
# Marker replaced by the grassee's name in EINSTEIN's speech
GRASSEE_MARKER = '$1'

# Character that will be replaced by a newline in lines messages
LINES_MESSAGE_NEWLINE = '^'

class Cast:
    def __init__(self, sprites):
        self.sprites = sprites
        self.skool = None
        self.screen = None
        self.beeper = None
        self.sprite_groups = {}
        self.eric = None
        self.everything = []     # Everything that must be drawn
        self.characters = {}     # Humans
        self.character_list = [] # Ordered list of humans
        self.movables = []       # All computer-controlled things
        self.taps = {}
        # We store animals (mice and frogs) separately so that we can check
        # whether Eric is next to one
        self.animals = []
        # We store plants separately so that we can check whether Eric is
        # standing on one
        self.plants = []
        # We store the bike separately so that we can release it and check
        # whether Eric is beside it
        self.bike = None
        # We store frogs separately so that they can be released when Eric gets
        # the storeroom key
        self.frogs = []
        # We store the desk lid separately so that it can be opened and closed
        self.desk_lid = None
        # We store the conker separately so that we can check whether it is hit
        # by a catapult pellet
        self.conker = None

    def add_sprite(self, group_id, sprite_id, sprite_index):
        if group_id not in self.sprite_groups:
            self.sprite_groups[group_id] = [{}, {}]
        sprite = self.get_animatory_state(sprite_index)
        self.sprite_groups[group_id][0][sprite_id] = sprite
        self.sprite_groups[group_id][1][sprite_id] = pygame.transform.flip(sprite, True, False)

    def add_eric(self, character_id, name, sprite_group_id, initial_as, flags):
        self.eric = Eric(character_id, name, flags)
        self.eric.set_animatory_states(*self.sprite_groups[sprite_group_id])
        self.eric.initialise_animatory_state(initial_as)
        self.everything.append(self.eric)
        self.characters[character_id] = self.eric
        self.character_list.append(self.eric)

    def add_character(self, character_id, name_and_title, sprite_group_id, initial_as, flags):
        names = name_and_title.partition('/')
        name, title = names[0], names[2]
        c = char.Character(character_id, name, flags)
        c.title = title
        c.set_animatory_states(*self.sprite_groups[sprite_group_id])
        c.initialise_animatory_state(initial_as)
        self.everything.append(c)
        self.characters[character_id] = c
        self.character_list.append(c)
        self.movables.append(c)

    def add_pellet(self, character_id, pellet_id, sprite_group_id, tap_id, pellet_range, hit_zone):
        pellet = Pellet(pellet_id, tap_id, pellet_range, hit_zone)
        pellet.set_animatory_states(*self.sprite_groups[sprite_group_id])
        self.everything.append(pellet)
        self.movables.append(pellet)
        self.get(character_id).pellet = pellet

    def add_water_drop(self, object_id, sprite_group_id, tap_id):
        self.water_drop = droppable.WaterDrop(object_id, tap_id)
        self.water_drop.set_animatory_states(*self.sprite_groups[sprite_group_id])
        self.everything.append(self.water_drop)
        self.movables.append(self.water_drop)

    def add_sherry_drop(self, object_id, sprite_group_id, tap_id):
        self.sherry_drop = droppable.SherryDrop(object_id, tap_id)
        self.sherry_drop.set_animatory_states(*self.sprite_groups[sprite_group_id])
        self.everything.append(self.sherry_drop)
        self.movables.append(self.sherry_drop)

    def add_conker(self, object_id, sprite_group_id, tap_id, min_x, max_x, min_y, max_y):
        self.conker = droppable.Conker(object_id, tap_id, min_x, max_x, min_y, max_y)
        self.conker.set_animatory_states(*self.sprite_groups[sprite_group_id])
        self.everything.append(self.conker)
        self.movables.append(self.conker)

    def add_water(self, character_id, water_id, sprite_group_id, tap_id):
        water = Water(water_id, tap_id)
        water.set_animatory_states(*self.sprite_groups[sprite_group_id])
        self.everything.append(water)
        self.movables.append(water)
        self.get(character_id).water = water

    def add_stinkbomb(self, character_id, stinkbomb_id, sprite_group_id, tap_id):
        stinkbomb = Stinkbomb(stinkbomb_id, tap_id)
        stinkbomb.set_animatory_states(*self.sprite_groups[sprite_group_id])
        self.everything.append(stinkbomb)
        self.movables.append(stinkbomb)
        self.get(character_id).stinkbomb = stinkbomb

    def add_desk_lid(self, desk_lid_id, sprite_group_id, tap_id):
        self.desk_lid = DeskLid(desk_lid_id, tap_id)
        self.desk_lid.set_animatory_states(*self.sprite_groups[sprite_group_id])
        self.everything.append(self.desk_lid)
        self.movables.append(self.desk_lid)

    def add_mouse(self, mouse_id, sprite_group_id, initial_as, location, tap_id, immortal=True):
        mouse = animal.Mouse(mouse_id, tap_id, initial_as, location, immortal)
        mouse.set_animatory_states(*self.sprite_groups[sprite_group_id])
        self.everything.append(mouse)
        self.animals.append(mouse)
        self.movables.append(mouse)
        return mouse

    def add_frog(self, frog_id, sprite_group_id, initial_as, location, tap_id):
        frog = animal.Frog(frog_id, tap_id, initial_as, location)
        frog.set_animatory_states(*self.sprite_groups[sprite_group_id])
        self.everything.append(frog)
        self.animals.append(frog)
        self.movables.append(frog)
        self.frogs.append(frog)

    def add_bike(self, bike_id, sprite_group_id, initial_as, location, tap_id):
        self.bike = Bike(bike_id, tap_id, initial_as, location)
        self.bike.set_animatory_states(*self.sprite_groups[sprite_group_id])
        self.everything.append(self.bike)
        self.movables.append(self.bike)

    def add_plant(self, plant_id, sprite_group_id, x, y, tap_id):
        plant = Plant(plant_id, tap_id, x, y)
        plant.set_animatory_states(*self.sprite_groups[sprite_group_id])
        self.everything.append(plant)
        self.plants.append(plant)
        self.movables.append(plant)
        return plant

    def get_animatory_state(self, state):
        width = self.sprites.get_width() // SPRITE_COLUMNS
        height = self.sprites.get_height() // SPRITE_ROWS
        surface = self.sprites.subsurface((width * (state % SPRITE_COLUMNS), height * (state // SPRITE_COLUMNS)), (width, height))
        surface.set_colorkey(SPRITE_COLORKEY)
        return surface

    def initialise(self, skool, screen, beeper):
        self.skool = skool
        self.screen = screen
        self.beeper = beeper
        for character in self.everything:
            self.initialise_character(character)

    def initialise_character(self, character):
        character.set_components(self, self.skool, self.screen, self.beeper)

    def reinitialise(self):
        for c in self.everything:
            c.reinitialise()

    def initialise_safe_combo(self):
        letters = ''
        for character in self.character_list:
            character.initialise_special_answer()
            if character.has_safe_secret():
                letters += character.initialise_safe_secret()
        return letters

    def initialise_bike_combo(self):
        digits = ''
        for character in self.character_list:
            if character.has_bike_secret():
                digits += character.initialise_bike_secret()
        return digits

    def initialise_storeroom_combo(self):
        letters = ''
        for character in self.character_list:
            if character.has_storeroom_secret():
                letters += character.initialise_storeroom_secret()
        return letters

    def add_command(self, tap_id, command_class, *params):
        if tap_id not in self.taps:
            self.taps[tap_id] = CommandListTemplate(tap_id)
        self.taps[tap_id].add_command(command_class, *params)

    def set_random_locations(self, character_id, locations):
        if character_id in self.characters:
            self.characters[character_id].set_random_locations(locations)

    def add_command_list(self, character_id, lesson_id, tap_id):
        if character_id in self.characters:
            self.characters[character_id].add_command_list(lesson_id, tap_id)

    def set_location(self, character_id, x, y):
        self.get(character_id).set_initial_location(x, y)

    def set_sit_down_message(self, character_id, message):
        self.characters[character_id].set_sit_down_message(message)

    def get(self, character_id):
        return self.characters.get(character_id, None)

    def add_blackboard_message(self, character_id, message):
        self.characters[character_id].add_blackboard_message(message)

    def add_lines_message(self, character_id, message_id, message):
        message_lines = message.split(LINES_MESSAGE_NEWLINE)
        if character_id == '*':
            for c in self.get_lines_givers():
                c.add_lines_message(message_id, message_lines)
        else:
            self.get(character_id).add_lines_message(message_id, message_lines)

    def add_lesson_message(self, character_id, message, condition):
        if character_id == '*':
            for c in self.character_list:
                if c.is_a_teacher():
                    c.add_lesson_message(message, condition)
        else:
            self.get(character_id).add_lesson_message(message, condition)

    def move(self, keyboard):
        for movable in self.movables:
            movable.move()
        return self.eric.move(keyboard)

    def get_images(self):
        return [thing.get_image() for thing in self.everything]

    def get_speech_bubbles(self):
        return [character.get_speech_bubble() for character in self.character_list]

    def set_lesson(self, lesson_id):
        for movable in self.movables:
            tap_id = movable.get_tap_id(lesson_id)
            movable.set_command_list_template(self.taps[tap_id])
            movable.remove_bubble()
        self.eric.unfreeze()

    def get_facing_characters(self, character, offset):
        target_x = character.x + offset * character.direction
        target_y = character.y
        target_direction = -1 * character.direction
        facing_characters = []
        for c in self.character_list:
            if (c.x, c.y, c.direction) == (target_x, target_y, target_direction):
                facing_characters.append(c)
        return facing_characters

    def has_kissees(self):
        """Return whether anyone in the cast can kiss Eric."""
        return any(c.can_kiss_eric() for c in self.character_list)

    def kissee(self):
        for c in self.get_facing_characters(self.eric, KISS_DISTANCE):
            if c.can_kiss_eric() and c.is_interruptible():
                return c

    def get_punchees(self, character, offset):
        punchees = []
        for c in self.get_facing_characters(character, offset):
            if c.is_punchable_now():
                punchees.append(c)
        return punchees

    def get_pelletables(self):
        """Returns a list of the pelletable characters."""
        return [c for c in self.character_list if c.is_pelletable()]

    def get_pelletable(self, x, y):
        """Returns the most suitable character to hit (if any) with a catapult
        pellet at the given coordinates."""
        pelletables = self.get_pelletables()
        for adult in [c for c in pelletables if c.is_adult()]:
            if (adult.x, adult.y) == (x, y):
                return adult
        for child in [c for c in pelletables if not c.is_adult()]:
            if (child.x, child.y) == (x, y):
                return child

    def get_waterable(self, x, y):
        for c in self.character_list:
            if c.has_bike_secret() and (c.x, c.y) == (x, y):
                return c

    def get_sherryable(self, x, y):
        for c in self.character_list:
            if c.has_storeroom_secret() and (c.x, c.y) == (x, y):
                return c

    def get_conkerable(self, x, y):
        for c in self.character_list:
            if (c.is_conkerable() or c.is_very_conkerable()) and (c.x, c.y) == (x, y):
                return c

    def get_adults(self):
        return [c for c in self.character_list if c.is_adult()]

    def get_lines_givers(self):
        return [c for c in self.character_list if c.can_give_lines()]

    def get_nearby_characters(self, character, candidates, witness):
        """Return a list of characters from the given list of candidates who are close enough to the given character to be visible to him (regardless of the direction he's facing)."""
        x0 = character.x - NEARBY_X_RANGE
        x1 = character.x + NEARBY_X_RANGE
        y0 = character.y - NEARBY_Y_RANGE
        y1 = character.y + NEARBY_Y_RANGE
        nearby_characters = []
        for c in candidates:
            if c is not character and x0 <= c.x <= x1 and y0 <= c.y <= y1 and c.has_line_of_sight_to(character):
                if not witness or c.direction * (character.x - c.x) >= 0:
                    nearby_characters.append(c)
        return nearby_characters

    def get_witnesses(self, character, candidates):
        """Returns a list of characters from the given list of candidates who
        are close enough to the given character (and are facing the right way)
        to be able to see him."""
        return self.get_nearby_characters(character, candidates, True)

    def get_nearby_adults(self, character):
        """Returns a list of adults who are close enough to the given character
        (and are facing the right way) to be able to see him."""
        return self.get_witnesses(character, self.get_adults())

    def get_nearby_lines_givers(self, character):
        """Returns a list of lines-givers who are close enough to the given
        character (and are facing the right way) to be able to see him."""
        return self.get_witnesses(character, self.get_lines_givers())

    def get_potential_lines_recipients(self, character):
        lines_recipients = [c for c in self.character_list if c.can_receive_lines()]
        return self.get_nearby_characters(character, lines_recipients, False)

    def get_nearest_lines_recipient(self, character):
        """Return the potential lines recipient nearest to the given
        character."""
        candidates = self.get_potential_lines_recipients(character)
        if len(candidates) > 0:
            nearest = candidates[0]
            for c in candidates[1:]:
                if abs(c.x - character.x) < abs(nearest.x - character.x):
                    nearest = c
            return nearest

    def can_get_lines(self, message_id):
        return any(c.can_give_lines_message(message_id) for c in self.get_lines_givers())

    def get_animal(self, character):
        x = character.x + character.direction
        for animal in self.animals:
            if animal.y == character.y and animal.x == x and animal.is_sitting():
                return animal

    def is_bike_visible(self):
        return self.bike and self.bike.is_visible()

    def is_beside_bike(self, character):
        return self.bike and (self.bike.x, self.bike.y) == (character.x, character.y)

    def move_bike_away(self, door):
        if self.bike and door.top_y <= self.bike.y <= door.bottom_y and door.x - 2 <= self.bike.x <= door.x:
            self.bike.x = door.x + 1

    def expand_names(self, message):
        """Replace occurrences of $BLAH with the name of the character whose unique ID is 'BLAH'."""
        index = 0
        marker = '$'
        while message.find(marker, index) >=0:
            start = message.index(marker, index)
            end = start + 1
            while end < len(message) and message[end].isalnum():
                end += 1
            character = self.get(message[start + 1:end])
            if character:
                message = message.replace(message[start:end], character.name, 1)
            index = end
        return message

    def is_home(self, x):
        """Return whether every character is on the 'home' side of the given x-coordinate."""
        return all(c.is_home(x) for c in self.character_list)

    def is_standing_on_kid(self, character):
        """Return whether the character is standing on a kid who's been knocked out."""
        for c in self.character_list:
            if c.is_knocked_out() and c.x - 1 <= character.x <= c.x + 1 and character.y == c.y - 1:
                return True

    def somebody_near_door(self, door):
        """Return True if somebody is standing near the door, False otherwise."""
        for c in self.character_list:
            if door.x - 2 <= c.x <= door.x + 1 and door.top_y <= c.y <= door.bottom_y:
                return True

    def is_beside_eric(self, character):
        """Return whether the character is beside Eric (and so need go no further to find him)."""
        eric_x, eric_y = self.get_location_of_eric()
        return abs(character.x - eric_x) <= BESIDE_ERIC_X_RANGE and character.y == eric_y

    def get_location_of_eric(self):
        """Return the non-staircase location closest to Eric."""
        return self.eric.get_location()

    def is_eric_expelled(self):
        return self.eric.expelled

    def expel_eric(self, command_list_id):
        for c in self.character_list:
            if c.can_expel_eric():
                self.eric.expelled = True
                self.change_command_list(c, command_list_id)
                return

    def get_eric_stopper(self):
        for c in self.character_list:
            if c.is_stopping_eric(self.eric):
                return c

    def shadow_eric(self, command_list_id):
        for c in self.character_list:
            if c.can_shadow_eric():
                self.change_command_list(c, command_list_id)
                return

    def freeze_eric(self):
        return self.eric.freeze()

    def unfreeze_eric(self):
        self.eric.unfreeze()

    def eric_understood(self):
        return self.eric.understood_message()

    def is_touching_eric(self, character):
        return (character.x, character.y) == (self.eric.x, self.eric.y)

    def is_eric_absent(self):
        return self.eric.is_absent()

    def open_desk(self, character, desk):
        self.desk_lid.raise_lid(desk, character)

    def trip_people_up_at(self, character, x, y):
        # Assume trippability == pelletability
        for trippable in self.get_pelletables():
            if trippable is not character and (trippable.x, trippable.y) == (x, y) and trippable.is_deckable():
                trippable.deck()

    def water_plant(self, plant_pot, liquid):
        if liquid == char.WATER:
            plant_pot.plant.grow()

    def lift_anyone_at(self, x, y):
        for c in self.character_list:
            if (c.x, c.y) == (x, y):
                c.y -= 1

    def plant(self, character):
        for plant in self.plants:
            if plant.supports(character):
                return plant

    def drop_anyone_at(self, x, y):
        for c in self.character_list:
            if (c.x, c.y) == (x, y):
                c.fall_off_plant()

    def knock_cup(self, cup):
        if cup.frogs:
            for frog in cup.frogs:
                frog.fall_from_cup(cup)
            return
        x, y = cup.x - droppable.DROP_X, cup.y - droppable.DROP_Y
        if cup.contents == char.WATER and not self.water_drop.is_visible():
            self.water_drop.fall(x, y)
        elif cup.contents == char.SHERRY and not self.sherry_drop.is_visible():
            self.sherry_drop.fall(x, y)

    def hit_conker(self, pellet):
        if self.conker and self.conker.hit_by(pellet):
            self.conker.fall(pellet.x, pellet.y - 2)
            return True
        return False

    def conker_falling(self):
        return self.conker and self.conker.is_visible()

    def unchain_bike(self):
        self.bike.unchain()

    def insert_frog(self, cup):
        for frog in self.frogs:
            if not frog.is_visible():
                cup.insert_frog(frog)
                break

    def check_heads(self, frog):
        for c in self.character_list:
            if c.has_safe_key():
                hit_head = False
                x_delta = animal.FROG_X - 1
                y_delta = animal.FROG_Y + 1
                if c.is_standing():
                    hit_head = (frog.x, frog.y) == (c.x - x_delta, c.y - y_delta)
                elif c.is_knocked_over():
                    hit_head = (frog.x, frog.y) == (c.x - x_delta, c.y - y_delta + 1)
                if hit_head:
                    self.eric.take_safe_key()
                    return True
        return False

    def smeller(self, stinkbomb):
        for c in self.character_list:
            if c.can_smell_stinkbomb(stinkbomb):
                return c

    def caught_mouse(self, mouse):
        if mouse.immortal:
            self.relocate_mouse(mouse)
        else:
            self.kill_mouse(mouse)

    def relocate_mouse(self, mouse):
        locations = list(MOUSE_LOCATIONS)
        while locations and self.screen.contains(mouse):
            mouse.x, mouse.y = locations.pop(random.randrange(len(locations)))
        mouse.restart_table()

    def release_mice(self, num_mice, x, y):
        while num_mice > 0:
            mouse_id = 'MortalMouse%i' % num_mice
            tap_id = MOUSE_COMMAND_LIST
            mouse = self.add_mouse(mouse_id, MOUSE_SPRITE_GROUP, RUN, (x, y), tap_id, False)
            self.initialise_character(mouse)
            mouse.set_command_list_template(self.taps[tap_id])
            num_mice -= 1

    def kill_mouse(self, mouse):
        self.everything.remove(mouse)
        self.movables.remove(mouse)
        self.animals.remove(mouse)

    def scare_musophobes(self, mouse):
        for c in self.character_list:
            c.check_mouse(mouse)

    def change_command_list(self, character, command_list_id):
        character.set_command_list_template(self.taps[command_list_id])

    ############################################################################
    # Grass config
    ############################################################################
    def set_hitters(self, hitters):
        self.hitters = hitters

    def set_writers(self, writers):
        self.writers = writers

    def set_hit_tale(self, hit_tale):
        self.hit_tale = hit_tale

    def set_write_tale(self, write_tale):
        self.write_tale = write_tale

    def set_absent_tale(self, absent_tale):
        self.absent_tale = absent_tale

    def expand_title(self, message, character):
        return message.replace(TITLE_MARKER, character.get_title())

    def get_hit_tale(self, teacher):
        hitter_id = random.choice(self.hitters)
        message = self.expand_names(self.hit_tale.replace(GRASSEE_MARKER, '$%s' % hitter_id))
        return hitter_id, self.expand_title(message, teacher)

    def get_write_tale(self, writer_id, teacher):
        if writer_id in self.writers:
            culprit = random.choice(self.writers)
            message = self.expand_names(self.write_tale.replace(GRASSEE_MARKER, '$%s' % writer_id))
            return writer_id, self.expand_title(message, teacher)
        return None, None

    def get_absent_tale(self, teacher):
        return self.expand_title(self.expand_names(self.absent_tale), teacher)
