# -*- coding: utf-8 -*-
# Copyright 2008, 2010 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import character
from animatorystates import *
from location import Location

# Coordinates of the water/sherry drop within its sprite (used for placement
# when spilled from a cup, and collision detection)
DROP_X, DROP_Y = 1, 3

class Droppable(character.Character):
    def __init__(self, object_id, tap_id, animatory_state, direction):
        character.Character.__init__(self, object_id, object_id)
        self.tap_id = tap_id
        self.animatory_state = self.initial_as = animatory_state
        self.direction = self.initial_direction = direction
        self.x = -3
        self.y = 0
        self.initial_location = Location((self.x, self.y))

    def get_tap_id(self, lesson_id):
        return self.tap_id

    def fall(self, x, y):
        self.x = x
        self.y = y

class Drop(Droppable):
    def __init__(self, object_id, tap_id, animatory_state, direction):
        Droppable.__init__(self, object_id, tap_id, animatory_state, direction)

    def hit_victim(self):
        hit_x = self.x - DROP_X + 1
        hit_y = self.y + character.SPRITE_HEIGHT - DROP_Y + 3
        victim = self.get_victim(hit_x, hit_y)
        if victim and victim.is_standing():
            self.reveal_secret(victim)
            victim.deck()
            return True
        victim = self.get_victim(hit_x, hit_y - 1)
        if victim and victim.is_knocked_over():
            self.reveal_secret(victim)
            return True
        return False

class WaterDrop(Drop):
    def __init__(self, object_id, tap_id):
        Drop.__init__(self, object_id, tap_id, WATER_DROP, -1)

    def get_victim(self, x, y):
        return self.cast.get_waterable(x, y)

    def reveal_secret(self, victim):
        victim.reveal_bike_secret()

class SherryDrop(Drop):
    def __init__(self, object_id, tap_id):
        Drop.__init__(self, object_id, tap_id, SHERRY_DROP, 1)

    def get_victim(self, x, y):
        return self.cast.get_sherryable(x, y)

    def reveal_secret(self, victim):
        victim.reveal_storeroom_secret()

class Conker(Droppable):
    def __init__(self, object_id, tap_id, min_x, max_x, min_y, max_y):
        Droppable.__init__(self, object_id, tap_id, CONKER, -1)
        self.min_x = min_x
        self.max_x = max_x
        self.min_y = min_y
        self.max_y = max_y

    def hit_by(self, pellet):
        return self.min_x <= pellet.x <= self.max_x and self.min_y <= pellet.y <= self.max_y

    def hit_victim(self):
        victim = self.cast.get_conkerable(self.x, self.y + 4)
        if victim:
            victim.deck(victim.is_very_conkerable())
            return True
        return False
