# -*- coding: utf-8 -*-
# Copyright 2008, 2010 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import character
from animatorystates import *
from ai import Write, Jump, Hit, FireCatapult, Freeze, Catch, FireWaterPistol, DumpWaterPistol, DropStinkbomb, RideBike, FallToFloor, JumpOutOfWindow, ClimbOverSkoolGate, Kiss, ReleaseMice, SKOOL_GATE
import lines
import items
import bike
import keys

# Minimum delay between two non-immediate lines-givings by the same teacher
LINES_DELAY = 60

# The command list Mr Wacker should use to expel Eric after he's jumped out of
# the top-floor window
NOT_A_BIRD = 'ExpelEric:NotABird'

# ID of the science lab storeroom door
STOREROOM_DOOR = 'ScienceLabDoor'

# ID of the assembly hall
ASSEMBLY_HALL = 'AssemblyHall'

# Delay before Eric returns to the floor after jumping
JUMP_DELAY = 4
# Delay before Eric stands up after bending over
BEND_OVER_DELAY = 4

# Maximum number of mice to release per attempt
MAX_MICE_RELEASE = 5

# Points scored when bike is unchained
BIKE_COMBO_SCORE = 1000
# Points scored when storeroom combination is obtained
STOREROOM_COMBO_SCORE = 1000
# Points scored when safe key is obtained
SAFE_KEY_SCORE = 2000

# Number of lines kissee does for Eric on each kiss
KISS_LINES = 1000

class Eric(character.Character):
    def __init__(self, character_id, name, flags):
        character.Character.__init__(self, character_id, name, flags)
        self._reinitialise()

    def reinitialise(self):
        character.Character.reinitialise(self)
        self._reinitialise()

    def _reinitialise(self):
        self.writing = False
        self.frozen = False
        self.understood = False
        self.controller = None
        self.lines_delay = LINES_DELAY
        self.last_lines_giver = None
        self.inventory = set()
        self.mice = 0
        self.last_bike_key = None
        self.bike_key = None
        self.on_bike = False
        self.started_pedalling = False
        self.sitting_on_saddle = False
        self.expelled = False
        self.hide_coords = None

    def move(self, keyboard):
        self.keyboard = keyboard
        self.get_lines()
        self.walk_delay -= 1
        if self.walk_delay > 0:
            return 0
        self.barrier = self.skool.barrier(self) or self.cast.get_eric_stopper()
        self.floor = self.get_floor()
        self.keyboard.pump()
        self.walk_delay = 2
        if self.controller:
            next_controller = self.controller.command(self)
            if next_controller is self.controller:
                self.controller = None
            elif next_controller is not None:
                self.controller = next_controller
            return self.screen.get_scroll_increment(self.x)
        if self.midstride():
            self.make_walking_sound(self.get_walk_state_index())
            return self.walk()
        if self.is_knocked_out() or self.is_sitting():
            if self.keyboard.was_pressed(*keys.SIT_STAND):
                if self.is_sitting_on_chair():
                    self.chair().vacate()
                self.animatory_state = WALK0
                self.beeper.make_sitting_sound()
            elif self.is_sitting_on_chair() and self.keyboard.was_pressed(*keys.OPEN_DESK):
                self.open_desk(self.skool.desk(self))
            return 0
        if self.keyboard.was_pressed(*keys.SIT_STAND) and self.sit():
            self.beeper.make_sitting_sound()
            return 0
        if self.keyboard.was_pressed(*keys.FIRE_CATAPULT) and self.can_fire_catapult():
            self.controller = FireCatapult()
            return 0
        if self.keyboard.was_pressed(*keys.FIRE_WATER_PISTOL) and self.can_fire_water_pistol():
            self.controller = FireWaterPistol()
            return 0
        if self.keyboard.was_pressed(*keys.DUMP_WATER_PISTOL) and self.can_dump_water_pistol():
            self.bend_over()
            self.controller = DumpWaterPistol()
            return 0
        if self.keyboard.was_pressed(*keys.DROP_STINKBOMB) and self.can_drop_stinkbomb():
            self.controller = DropStinkbomb()
            return 0
        if self.keyboard.was_pressed(*keys.HIT):
            self.controller = Hit()
            return 0
        if self.keyboard.was_pressed(*keys.JUMP):
            self.controller = Jump()
            return 0
        if self.keyboard.was_pressed(*keys.WRITE) and self.skool.beside_blackboard(self):
            self.raise_arm()
            self.controller = Write()
            self.keyboard.start_writing()
            self.writing = True
            return 0
        if self.keyboard.was_pressed(*keys.CATCH) and self.can_bend_over():
            self.bend_over()
            self.controller = Catch()
            return 0
        if self.keyboard.was_pressed(*keys.MOUNT_BIKE) and self.can_mount_bike():
            self.mount_bike()
            self.controller = RideBike(self.cast.bike)
            return 0
        if self.keyboard.was_pressed(*keys.KISS) and self.can_kiss():
            self.previous_as = self.animatory_state
            self.controller = Kiss()
            return 0
        if self.keyboard.was_pressed(*keys.RELEASE_MICE) and self.can_release_mice():
            self.bend_over()
            self.controller = ReleaseMice()
            return 0
        self.staircase = self.skool.staircase(self)
        if not self.is_supported():
            self.y += 1
            return 0
        old_ws, old_as, old_direction = self.get_walk_state_index(), self.animatory_state, self.direction
        plant = self.cast.plant(self)
        if self.keyboard.pressed(*keys.LEFT):
            if plant and plant.is_fully_grown():
                barrier = self.skool.barrier(self, 1)
                if barrier and barrier.barrier_id == SKOOL_GATE and barrier.is_shut():
                    self.controller = ClimbOverSkoolGate()
                    return 0
            self.left()
        elif self.keyboard.pressed(*keys.RIGHT):
            if plant and plant.is_fully_grown():
                window = self.skool.window(self)
                if window and not window.is_shut():
                    self.controller = JumpOutOfWindow(window.barrier_id)
                    return 0
                elif self.barrier and self.barrier.barrier_id == SKOOL_GATE and self.barrier.is_shut():
                    self.controller = ClimbOverSkoolGate()
                    return 0
            self.right()
        elif self.keyboard.pressed(*keys.UP):
            self.up()
        elif self.keyboard.pressed(*keys.DOWN):
            self.down()
        if (self.animatory_state, self.direction) != (old_as, old_direction):
            self.make_walking_sound(old_ws)
        return 0

    def can_fire_catapult(self):
        return character.Character.can_fire_catapult(self) and not self.cast.conker_falling()

    def can_sit_on_stairs(self):
        """Return whether Eric can sit on the stairs."""
        return self.cast.can_get_lines(lines.NO_SITTING_ON_STAIRS)

    def is_supported(self):
        """Return whether Eric is standing on something that prevents him from falling."""
        return (self.staircase
            or self.cast.is_standing_on_kid(self)
            or self.skool.on_floor(self)
            or self.skool.plant_pot(self.x, self.y)
            or self.cast.plant(self))

    def can_open_door(self, door):
        if door.barrier_id == STOREROOM_DOOR and items.STOREROOM_KEY in self.inventory:
            return True
        return character.Character.can_open_door(self, door)

    def open_door(self):
        self.skool.move_door(self.barrier.barrier_id, False)
        return 0

    def get_lines(self):
        self.lines_delay = max(0, self.lines_delay - 1)
        if self.lines_delay > LINES_DELAY / 2:
            return
        if self.lines_delay == 0:
            self.last_lines_giver = None
        room = self.skool.room(self)
        home_room = self.skool.get_home_room()
        if self.skool.in_no_go_zone(self):
            self.alert_lines_givers(lines.GET_OUT, True)
        elif self.should_get_along():
            self.alert_lines_givers(lines.GET_ALONG, True)
        elif self.is_sitting_on_stairs():
            self.alert_lines_givers(lines.NO_SITTING_ON_STAIRS, True)
        elif self.is_knocked_out():
            self.alert_lines_givers(lines.GET_UP, True)
        elif self.on_bike and not self.skool.in_playground(self):
            self.alert_lines_givers(lines.NO_BIKES, True)
        elif self.skool.plant_pot(self.x, self.y) or self.cast.plant(self):
            self.alert_lines_givers(lines.GET_OFF_PLANT, True)
        else:
            is_assembly = self.skool.is_assembly()
            if self.is_sitting_on_floor():
                # Eric is sitting on the floor...
                if room and room.room_id == ASSEMBLY_HALL:
                    if self.direction < 0:
                        # ...facing left in the assembly hall...
                        if is_assembly:
                            # ...and it's assembly - tell him to sit facing the
                            # stage
                            self.alert_lines_givers(lines.SIT_FACING_STAGE, True)
                        else:
                            # ...and it's not assembly - tell him to get off
                            # the floor
                            self.alert_lines_givers(lines.GET_UP, True)
                else:
                    # ...somewhere other than the assembly hall - tell him to
                    # get off the floor
                    self.alert_lines_givers(lines.GET_UP, True)
            elif self.is_standing() and home_room and home_room.contains(self) and (home_room.chairs or home_room.room_id == ASSEMBLY_HALL):
                # Eric is standing up either in a classroom after the lesson
                # has started, or in the assembly hall after assembly has
                # started - tell him to sit down
                self.alert_lines_givers(lines.SIT_DOWN, True)

    def is_absent(self):
        """Return whether Eric is playing truant."""
        return self.skool.get_home_room() not in (None, self.skool.room(self))

    def should_get_along(self):
        """Return whether Eric is somewhere other than he should be."""
        return self.skool.should_get_along(self)

    def make_walking_sound(self, index):
        self.beeper.make_walking_sound(index)

    def walk(self, on_stairs=False):
        character.Character.walk(self, on_stairs)
        if not self.midstride():
            return self.screen.get_scroll_increment(self.x)
        return 0

    def dethrone(self):
        self.sit_on_floor()
        self.beeper.make_knocked_out_sound()

    def deck(self):
        if self.is_sitting_on_chair():
            self.chair().vacate()
        self.knock_over()
        if self.writing:
            self.writing = False
            self.keyboard.finish_writing()
            self.controller = None
        self.beeper.make_knocked_out_sound()

    def alert_lines_givers(self, message_id, reset_delay=False):
        nearby_lines_givers = self.cast.get_nearby_lines_givers(self)
        if nearby_lines_givers:
            lines_giver = nearby_lines_givers[0]
            if message_id == lines.GET_ALONG:
                teacher = self.skool.get_teacher()
                if teacher in nearby_lines_givers:
                    lines_giver = teacher
                    message_id = teacher.get_come_along_message_id()
                elif self.skool.in_playground(self):
                    message_id = lines.BACK_TO_SKOOL
            if reset_delay:
                if lines_giver is self.last_lines_giver:
                    return
                self.lines_delay = LINES_DELAY
            lines_giver.give_lines(self.character_id, message_id)
            self.last_lines_giver = lines_giver

    def aim_catapult(self):
        character.Character.aim_catapult(self)
        self.alert_lines_givers(lines.NO_CATAPULTS)
        self.beeper.make_catapult_sound()

    def has_water_pistol(self):
        return items.WATER_PISTOL in self.inventory or items.SHERRY_PISTOL in self.inventory

    def can_fire_water_pistol(self):
        if self.has_water_pistol():
            return character.Character.can_fire_water_pistol(self)

    def can_dump_water_pistol(self):
        return self.has_water_pistol() and self.can_bend_over()

    def dump_water_pistol(self):
        pistol = None
        if items.WATER_PISTOL in self.inventory:
            pistol = items.WATER_PISTOL
        elif items.SHERRY_PISTOL in self.inventory:
            pistol = items.SHERRY_PISTOL
        if pistol:
            self.inventory.remove(pistol)
            self.print_inventory()
            self.skool.hide_in_desk(items.WATER_PISTOL)
        self.walk_delay = BEND_OVER_DELAY

    def fire_water_pistol(self):
        liquid = character.SHERRY if items.SHERRY_PISTOL in self.inventory else character.WATER
        character.Character.fire_water_pistol(self, liquid)
        self.alert_lines_givers(lines.NO_WATERPISTOLS)
        self.beeper.make_catapult_sound()

    def can_drop_stinkbomb(self):
        return self.floor and self.has_stinkbomb() and character.Character.can_drop_stinkbomb(self)

    def has_stinkbomb(self):
        return any([i in self.inventory for i in (items.STINKBOMBS1, items.STINKBOMBS2, items.STINKBOMBS3)])

    def drop_stinkbomb(self):
        character.Character.drop_stinkbomb(self)
        if items.STINKBOMBS3 in self.inventory:
            self.inventory.remove(items.STINKBOMBS3)
            self.inventory.add(items.STINKBOMBS2)
        elif items.STINKBOMBS2 in self.inventory:
            self.inventory.remove(items.STINKBOMBS2)
            self.inventory.add(items.STINKBOMBS1)
        elif items.STINKBOMBS1 in self.inventory:
            self.inventory.remove(items.STINKBOMBS1)
        self.print_inventory()
        self.alert_lines_givers(lines.NO_STINKBOMBS)

    def open_desk(self, desk):
        if desk:
            self.cast.open_desk(self, desk)

    def collect_desk_contents(self, desk):
        if desk.contents:
            if desk.contents == items.STINKBOMBS3:
                for item in (items.STINKBOMBS2, items.STINKBOMBS1):
                    if item in self.inventory:
                        self.inventory.remove(item)
            self.inventory.add(desk.contents)
            if desk.contents == items.WATER_PISTOL:
                desk.empty()
            self.beeper.make_bingo_sound()
            self.print_inventory()

    def punch(self):
        character.Character.punch(self)
        self.alert_lines_givers(lines.NO_HITTING)

    def jump(self):
        self.previous_as = self.animatory_state
        self.y -= 1
        self.walk_delay = JUMP_DELAY
        self.animatory_state = ARM_UP
        self.alert_lines_givers(lines.NO_JUMPING)

    def can_bend_over(self):
        return BENDING_OVER in self.as_dict_L

    def bend_over(self):
        self.previous_as = self.animatory_state
        self.walk_delay = 1
        self.animatory_state = BENDING_OVER

    def catch_animal(self):
        animal = self.cast.get_animal(self)
        if animal:
            if animal.is_mouse():
                self.skool.add_to_score(10)
                self.beeper.make_mouse_sound()
                self.mice += 1
                self.print_mouse_inventory()
                self.cast.caught_mouse(animal)
            elif animal.is_frog():
                animal.hide()
                self.beeper.make_bingo_sound()
                self.inventory.add(items.FROG)
                self.print_inventory()
        self.walk_delay = BEND_OVER_DELAY

    def print_mouse_inventory(self):
        self.screen.print_mice(self.mice, self.skool.get_mouse_image())

    def can_release_mice(self):
        return self.floor and self.mice > 0 and self.can_bend_over()

    def release_mice(self):
        if self.mice > 0:
            num_mice = min(self.mice, MAX_MICE_RELEASE)
            self.mice -= num_mice
            self.print_mouse_inventory()
            self.cast.release_mice(num_mice, self.x + self.direction, self.y)
        self.walk_delay = BEND_OVER_DELAY

    def print_inventory(self):
        self.screen.print_inventory(self.skool.get_inventory_images(self.inventory))

    def stand_up(self):
        self.animatory_state = self.previous_as
        self.walk_delay = 1

    def get_location(self):
        """Return the on-floor location that is closest to Eric."""
        if self.hide_coords:
            # If Eric is hiding (e.g. while kissing), temporarily restore his
            # pre-hide coordinates for the location check
            self.x, self.y = self.hide_coords
        location = character.Character.get_location(self)
        if self.hide_coords:
            self.hide()
        return location

    def get_location_above_hand(self):
        return (self.x + self.direction + 1, self.y)

    def descend(self):
        """Make Eric finish his jump."""
        x, y = self.get_location_above_hand()
        self.skool.check_safe(x, y, items.SAFE_KEY in self.inventory)
        self.check_shields_at(x, y)
        if self.skool.check_drinks_cabinet(x, y) and items.WATER_PISTOL in self.inventory:
            self.inventory.remove(items.WATER_PISTOL)
            self.inventory.add(items.SHERRY_PISTOL)
            self.print_inventory()
            self.beeper.make_sherry_sound()
        if not self.cast.is_standing_on_kid(self) and not self.skool.plant_pot(self.x, self.y):
            self.y += 1
        self.animatory_state = self.previous_as

    def write(self):
        """Control Eric when he's writing on a board. Return True if Eric has finished writing."""
        blackboard = self.skool.room(self).blackboard
        if self.has_arm_raised():
            self.lower_arm()
            self.writing = self.keyboard.writing
            if self.writing:
                self.alert_lines_givers(lines.NO_WRITING)
            else:
                self.check_combinations(blackboard)
            return not self.writing
        key_down_events = self.keyboard.key_down_events
        if key_down_events:
            first_event = key_down_events[0]
            if first_event.key in keys.ENTER:
                self.raise_arm()
                self.keyboard.finish_writing()
                return
            char = first_event.unicode
            if self.skool.font.has_char(char):
                self.raise_arm()
                self.skool.write_on_board(self, blackboard, char)
        else:
             # Maximise responsiveness
            self.walk_delay = 1

    def check_combinations(self, blackboard):
        if not self.cast.is_bike_visible() and self.skool.got_bike_combination(blackboard):
            self.skool.add_to_score(BIKE_COMBO_SCORE // 10)
            self.skool.unchain_bike()
            self.beeper.make_bingo_sound()
        elif items.STOREROOM_KEY not in self.inventory and self.skool.got_storeroom_combination(blackboard):
            self.skool.add_to_score(STOREROOM_COMBO_SCORE // 10)
            self.inventory.add(items.STOREROOM_KEY)
            self.print_inventory()
            self.beeper.make_bingo_sound()

    def is_eric(self):
        return True

    def freeze(self):
        if not self.writing:
            self.controller = Freeze()
            self.frozen = True
        return self.frozen

    def unfreeze(self):
        if self.frozen:
            self.controller = None
            self.frozen = False
            self.understood = False

    def check_understanding(self):
        if self.keyboard.was_pressed(*keys.UNDERSTOOD):
            self.understood = True

    def understood_message(self):
        return self.understood

    def paralyse(self):
        self.cast.expel_eric(NOT_A_BIRD)

    def take_safe_key(self):
        if items.SAFE_KEY not in self.inventory:
            self.skool.add_to_score(SAFE_KEY_SCORE // 10)
            self.beeper.make_safe_key_sound()
            self.inventory.add(items.SAFE_KEY)
            self.print_inventory()

    def can_kiss(self):
        return self.cast.has_kissees()

    def kissee(self):
        return self.cast.kissee()

    def kiss(self):
        self.skool.add_lines(-KISS_LINES // 10)
        self.beeper.make_bingo_sound()

    def hide(self):
        self.hide_coords = (self.x, self.y)
        character.Character.hide(self)

    def unhide(self):
        if self.hide_coords:
            self.x, self.y = self.hide_coords
            self.hide_coords = None

    def can_mount_bike(self):
        return RIDING_BIKE0 in self.as_dict_L and self.cast.is_beside_bike(self)

    def mount_bike(self):
        self.last_bike_key = None
        self.on_bike = True
        self.started_pedalling = False
        self.sitting_on_saddle = True
        self.previous_as = self.animatory_state
        self.animatory_state = RIDING_BIKE0

    def check_bike_keys(self):
        self.walk_delay = bike.SPEED
        self.bike_key = None
        for key in keys.LEFT + keys.RIGHT + keys.UP + keys.DOWN + keys.JUMP:
            if self.keyboard.was_pressed(key):
                self.bike_key = key
                break

    def rode_into_barrier(self, bike, barrier_id=None):
        if bike.is_visible():
            return bike.hit_barrier(barrier_id)
        # Check from a distance of 1 if travelling leftwards so that the bike
        # does not appear too close to the barrier if it hits it
        distance = 0 if self.direction > 0 else 1
        barrier = self.skool.barrier(self, distance)
        if barrier_id:
            return barrier and barrier.barrier_id == barrier_id
        return barrier is not None

    def pedalled(self):
        if self.sitting_on_saddle:
            pedalled = self.bike_key in keys.LEFT + keys.RIGHT and self.bike_key != self.last_bike_key
            if pedalled:
                self.started_pedalling = True
                self.pedal()
                self.last_bike_key = self.bike_key
            if self.started_pedalling:
                self.move_bike()
            return pedalled
        return False

    def pedal(self):
        if self.animatory_state == RIDING_BIKE0:
            self.animatory_state = RIDING_BIKE1
        else:
            self.animatory_state = RIDING_BIKE0

    def move_bike(self):
        self.x += self.direction

    def dismounted(self):
        return self.sitting_on_saddle and self.bike_key in keys.DOWN

    def dismount(self):
        self.on_bike = False

    def stood_on_saddle(self):
        if self.sitting_on_saddle and self.bike_key in keys.UP:
            self.last_bike_key = self.bike_key
            self.sitting_on_saddle = False
        return not self.sitting_on_saddle

    def stand_on_saddle(self):
        self.animatory_state = WALK0
        self.x -= self.direction
        self.y -= 1

    def got_back_on_saddle(self):
        if not self.sitting_on_saddle and self.bike_key in keys.DOWN:
            self.last_bike_key = self.bike_key
            self.sitting_on_saddle = True
        return self.sitting_on_saddle

    def get_back_on_saddle(self):
        self.animatory_state = RIDING_BIKE0
        self.x += self.direction
        self.y += 1

    def jumped(self):
        return not self.sitting_on_saddle and self.bike_key in keys.JUMP + keys.UP

    def check_cup(self):
        if items.FROG in self.inventory:
            cup = self.skool.cup(*self.get_location_above_hand())
            if cup:
                self.cast.insert_frog(cup)
                self.inventory.remove(items.FROG)
                self.print_inventory()
                self.beeper.make_bingo_sound()

    def fall_to_floor(self):
        self.sit_on_floor()
        self.beeper.make_knocked_out_sound()

    def fall_off_plant(self):
        self.controller = FallToFloor()

    def fly(self, x_inc, y_inc):
        self.x += x_inc * self.direction
        self.y += y_inc
        self.make_walking_sound(self.x % 4)

    def end_flight(self):
        if not self.is_standing():
            self.beeper.make_knocked_out_sound()
