# -*- coding: utf-8 -*-
# Copyright 2008, 2010 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import os
import pygame

# Width of the screen (in tiles)
WIDTH = 32
# Height of the play area (in tiles)
SKOOL_HEIGHT = 21
# Height of the screen (in tiles)
HEIGHT = SKOOL_HEIGHT + 3

# x-coordinate of Eric at which screen will be scrolled right
SCROLL_MIN_X = 9
# x-coordinate of Eric at which screen will be scrolled left
SCROLL_MAX_X = WIDTH - 10
# Number of columns to scroll
SCROLL_COLS = 8

# Size of the lesson box (width and height in tiles)
LESSON_BOX_SIZE = (8, 3)
# Size of the inventory (width and height in tiles)
INVENTORY_SIZE = (6, 1)
# Size of a lines bubble (width and height in tiles)
LINES_BUBBLE_SIZE = (8, 3)

# Offsets used to position printing of score, lines total and hi-score
SCORE_Y_OFFSET = 1
LINES_Y_OFFSET = 9
HISCORE_Y_OFFSET = 17

# Colours
BLACK = (0, 0, 0)
BLUE = (0, 0, 197)
RED = (197, 0, 0)
MAGENTA = (197, 0, 197)
GREEN = (0, 198, 0)
CYAN = (0, 198, 197)
YELLOW = (197, 198, 0)
WHITE = (205, 198, 205)
BRIGHT_BLACK = (0, 0, 0)
BRIGHT_BLUE = (0, 0, 255)
BRIGHT_RED = (255, 0, 0)
BRIGHT_MAGENTA = (255, 0, 255)
BRIGHT_GREEN = (0, 255, 0)
BRIGHT_CYAN = (0, 255, 255)
BRIGHT_YELLOW = (255, 255, 0)
BRIGHT_WHITE = (255, 255, 255)

# Size of the bounding rectangle of a speech bubble (width and height in tiles)
SPEECH_BUBBLE_SIZE = (8, 3)
# Transparent colour used in the speech bubble
SPEECH_BUBBLE_COLORKEY = (0, 255, 0)
# Bit pattern used to open the lip of the speech bubble
OPEN_LIP_BYTE = 66 # 01000010

# Transparent colour used in the skool ink image
SKOOL_COLORKEY = (255, 255, 255)

# Ink and paper colours in font.png (used to create transparency)
FONT_INK = (0, 1, 2)
FONT_PAPER = (255, 254, 253)

class Screen:
    def __init__(self, mode, scale, background, gallery, scroll_fps, title, icon_fname):
        self.initialise_colours()
        self.mode = mode
        self.scale = scale
        self.background = background
        self.scroll_fps = scroll_fps
        self.screen = pygame.display.set_mode(self.scale_coords((WIDTH, HEIGHT)))
        if os.path.isfile(icon_fname):
            pygame.display.set_icon(pygame.image.load(icon_fname).convert())
        pygame.display.set_caption(title)
        self.screen.fill(self.colours[background])
        self.bubble = gallery.get_speech_bubble()
        self.score_box = gallery.get_score_box()
        self.logo = gallery.get_logo()
        self.font = Font(gallery.get_font())
        self.inventory_coords = None
        self.mouse_box_coords = None

    def scale_coords(self, coords):
        return (8 * self.scale * coords[0], 8 * self.scale * coords[1])

    def initialise_colours(self):
        self.colours = []
        self.colours.append(BLACK)
        self.colours.append(BLUE)
        self.colours.append(RED)
        self.colours.append(MAGENTA)
        self.colours.append(GREEN)
        self.colours.append(CYAN)
        self.colours.append(YELLOW)
        self.colours.append(WHITE)
        self.colours.append(BRIGHT_BLACK)
        self.colours.append(BRIGHT_BLUE)
        self.colours.append(BRIGHT_RED)
        self.colours.append(BRIGHT_MAGENTA)
        self.colours.append(BRIGHT_GREEN)
        self.colours.append(BRIGHT_CYAN)
        self.colours.append(BRIGHT_YELLOW)
        self.colours.append(BRIGHT_WHITE)

    def initialise_column(self, initial_column, skool):
        self.max_column = skool.get_width() - WIDTH
        eric_x = skool.get_eric().x
        default_initial_column = min(self.max_column, max((eric_x - WIDTH // 2) // 8 * 8, 0))
        column = min(self.max_column, default_initial_column if initial_column is None else initial_column)
        self.column = self.initial_column = max(0, 8 * (column // 8))

    def reinitialise(self):
        self.column = self.initial_column
        self.print_lesson('', '')
        self.print_inventory()
        self.print_mice()

    def add_score_box(self, coords, x_offset, attribute):
        self.score_box_coords = coords
        self.score_box_x_offset = x_offset
        self.score_box_attribute = attribute
        self.screen.blit(self.score_box, self.scale_coords(self.score_box_coords))

    def add_logo(self, coords):
        self.screen.blit(self.logo, self.scale_coords(coords))

    def set_lesson_box_properties(self, attr, coords):
        self.lesson_box_attr = attr
        self.lesson_box_pos = self.scale_coords(coords)

    def add_mouse_inventory(self, coords):
        self.mouse_box_coords = coords

    def add_inventory(self, coords):
        self.inventory_coords = coords

    def get_scroll_increment(self, x):
        offset = x - self.column
        if SCROLL_MAX_X <= offset < WIDTH:
            return 1
        if -3 < offset <= SCROLL_MIN_X:
            return -1
        return 0

    def scroll_skool(self, skool, clock):
        self.column -= WIDTH
        background = pygame.Surface((self.screen.get_width(), 8 * self.scale * skool.get_height()))
        for n in range(WIDTH):
            self.column += 1
            skool.draw()
            self.screen.blit(background, (-8 * (n + 1) * self.scale, 0))
            self.update()
            clock.tick(self.scroll_fps)

    def scroll(self, inc, skool, clock):
        if self.column == self.max_column and inc > 0:
            return
        if self.column == 0 and inc < 0:
            return
        for i in range(SCROLL_COLS):
            self.column += inc
            skool.draw()
            self.update()
            clock.tick(self.scroll_fps)

    def get_text(self, words, ink, paper):
        return self.font.render(words, ink, paper)

    def get_ink_and_paper(self, attribute):
        ink = self.colours[(attribute & 64) // 8 + (attribute & 7)]
        paper = self.colours[(attribute & 120) // 8]
        return ink, paper

    def print_lesson(self, teacher, room):
        lesson_box = pygame.Surface(self.scale_coords(LESSON_BOX_SIZE))
        ink, paper = self.get_ink_and_paper(self.lesson_box_attr)
        lesson_box.fill(paper)
        teacher_text = self.get_text(teacher, ink, paper)
        room_text = self.get_text(room, ink, paper)
        font_height = self.scale * 8
        teacher_x = (lesson_box.get_width() - teacher_text.get_width()) // 2
        teacher_y = (lesson_box.get_height() - 2 * font_height) // 2
        lesson_box.blit(teacher_text, (teacher_x, teacher_y))
        room_x = (lesson_box.get_width() - room_text.get_width()) // 2
        room_y = teacher_y + font_height
        lesson_box.blit(room_text, (room_x, room_y))
        self.screen.blit(lesson_box, self.lesson_box_pos)
        self.update()

    def print_number(self, number, y_offset):
        if number < 0:
            return
        ink, paper = self.get_ink_and_paper(self.score_box_attribute)
        number_text = self.get_text(str(number) if number > 0 else "", ink, paper)
        width = 24 * self.scale
        background = pygame.Surface((width, 7 * self.scale))
        background.fill(paper)
        background.blit(number_text, (width - number_text.get_width(), 0))
        coords = self.scale_coords(self.score_box_coords)
        top_left_x = coords[0] + self.score_box_x_offset * self.scale - width
        top_left_y = coords[1] + y_offset * self.scale
        self.screen.blit(background, (top_left_x, top_left_y))
        self.update()

    def print_score(self, score):
        self.print_number(score, SCORE_Y_OFFSET)

    def print_lines(self, lines):
        self.print_number(lines, LINES_Y_OFFSET)

    def print_hi_score(self, hiscore):
        self.print_number(hiscore, HISCORE_Y_OFFSET)

    def print_inventory(self, item_images=()):
        if not self.inventory_coords:
            return
        inventory_box = pygame.Surface(self.scale_coords(INVENTORY_SIZE))
        ink, paper = self.get_ink_and_paper(self.background)
        inventory_box.fill(paper)
        x = 0
        for image in item_images:
            if image:
                inventory_box.blit(image, (x, 0))
                x += image.get_width()
        self.screen.blit(inventory_box, self.scale_coords(self.inventory_coords))
        self.update()

    def print_mice(self, count=0, mouse_image=None):
        if not self.mouse_box_coords:
            return
        max_mice = 8
        mouse_box = pygame.Surface(self.scale_coords((max_mice, 1)))
        ink, paper = self.get_ink_and_paper(self.background)
        mouse_box.fill(paper)
        for x in range(min((count, max_mice))):
            mouse_box.blit(mouse_image, (x * mouse_image.get_width(), 0))
        self.screen.blit(mouse_box, self.scale_coords(self.mouse_box_coords))
        self.update()

    def contains(self, character, full=True):
        """Return whether a character is on-screen."""
        return self.column <= character.x <= self.column + WIDTH - (3 if full else 1)

    def print_lines_bubble(self, x, y, message, ink_colour, paper_colour):
        bubble_x = 8 * (x // 8) - self.column
        bubble_coords = self.scale_coords((bubble_x, y))

        lines_bubble = pygame.Surface(self.scale_coords(LINES_BUBBLE_SIZE))
        ink = self.colours[ink_colour]
        paper = self.colours[paper_colour]
        lines_bubble.fill(paper)
        text1 = self.get_text(message[0], ink, paper)
        text2 = self.get_text(message[1], ink, paper)
        text1_x = (lines_bubble.get_width() - text1.get_width()) // 2
        text1_y = self.scale * 4
        text2_x = (lines_bubble.get_width() - text2.get_width()) // 2
        text2_y = text1_y + 8 * self.scale
        lines_bubble.blit(text1, (text1_x, text1_y))
        lines_bubble.blit(text2, (text2_x, text2_y))
        self.screen.blit(lines_bubble, bubble_coords)
        self.update()

    def get_bubble(self, words, lip_pos, shift):
        ink = self.colours[0]
        paper = self.colours[7]
        bubble = pygame.Surface(self.scale_coords(SPEECH_BUBBLE_SIZE))
        bubble.fill(SPEECH_BUBBLE_COLORKEY)
        bubble.set_colorkey(SPEECH_BUBBLE_COLORKEY)
        bubble.blit(self.bubble, (0, 0))
        lip = self.bubble.subsurface(self.scale_coords((8, 0)), self.scale_coords((1, 1)))
        lip_coords = self.scale_coords((lip_pos, 2))
        bubble.blit(lip, lip_coords)
        open_lip_byte = OPEN_LIP_BYTE
        for bit in range(0, 8 * self.scale, self.scale):
            colour = ink if open_lip_byte & 128 else paper
            for r in range(0, self.scale + 1):
                for c in range(0, self.scale + 1):
                    bubble.set_at((lip_coords[0] + bit + c, lip_coords[1] - 1 - r), colour)
            open_lip_byte *= 2
        text = self.get_text(words, ink, paper)
        inset_x = (4 - 8 * min(shift, 0)) * self.scale
        inset_y = 4 * self.scale
        text_x = max(shift, 0) * 8 * self.scale
        max_width = self.scale * 8 * (SPEECH_BUBBLE_SIZE[0] - 1)
        width = min(max_width, text.get_width() - text_x)
        if inset_x + width > max_width:
            width = max_width - inset_x
        text_window = text.subsurface((text_x, 0), (width, 8 * self.scale))
        bubble.blit(text_window, (inset_x, inset_y))
        return (bubble, text_x > text.get_width())

    def update(self):
        pygame.display.update()

    def draw(self, skool_images, blackboards, cast, others):
        if self.mode == 0:
            self.draw_skool(self.screen, skool_images[0])
            for x, y, text in blackboards:
                self.draw_blackboard(self.screen, x, y, text)
            for x, y, sprite in cast:
                self.draw_image(self.screen, x, y, sprite)
        else:
            scratch = pygame.Surface(self.scale_coords((WIDTH, SKOOL_HEIGHT)))
            self.draw_skool(scratch, skool_images[1])
            for x, y, text in blackboards:
                self.draw_blackboard(scratch, x, y, text)
            for x, y, sprite in cast:
                self.draw_image(scratch, x, y, sprite)
            self.draw_skool(self.screen, skool_images[2])
            scratch.set_colorkey(SKOOL_COLORKEY)
            self.screen.blit(scratch, (0, 0))
        for x, y, image in others:
            self.draw_image(self.screen, x, y, image)

    def draw_skool(self, surface, skool):
        surface.blit(skool, self.scale_coords((-self.column, 0)))

    def draw_blackboard(self, surface, x, y, text):
        for line_no, image in enumerate(text):
            coords = (self.scale * 8 * (x - self.column), self.scale * (8 * (y + line_no) + 1))
            surface.blit(image, coords)

    def draw_image(self, surface, x, y, image):
        if image:
            surface.blit(image, self.scale_coords((x - self.column, y)))

    def take_screenshot(self, filename):
        pygame.image.save(self.screen, filename)

class Gallery:
    def __init__(self, images_dir, image_set, scale, images):
        self.images_dir = images_dir
        self.image_set = image_set
        self.scale = scale
        self.images = images

    def get_image(self, image_id):
        if image_id not in self.images:
            return None
        scale_up = True
        image_set_dir = os.path.join(self.images_dir, '%sx%i' % (self.image_set, self.scale))
        if os.path.isdir(image_set_dir):
            scale_up = False
        else:
            image_set_dir = os.path.join(self.images_dir, '%sx1' % self.image_set)
        fname = os.path.join(*self.images[image_id].split('/'))
        image_file = os.path.join(image_set_dir, fname)
        if not os.path.exists(image_file):
            return None
        img = pygame.image.load(image_file).convert()
        if scale_up:
            img = pygame.transform.scale(img, (self.scale * img.get_width(), self.scale * img.get_height()))
        return img

    def get_speech_bubble(self):
        return self.get_image('SPEECH_BUBBLE')

    def get_font(self):
        return self.get_image('FONT')

    def get_inventory(self):
        return self.get_image('INVENTORY')

    def get_logo(self):
        return self.get_image('LOGO')

    def get_mutables(self):
        return (self.get_image('MUTABLES'), self.get_image('MUTABLES_INK'), self.get_image('MUTABLES_PAPER'))

    def get_score_box(self):
        return self.get_image('SCOREBOX')

    def get_skool(self, mode):
        if mode == 1:
            return (None, self.get_image('SKOOL_INK'), self.get_image('SKOOL_PAPER'))
        return (self.get_image('SKOOL'), None, None)

    def get_sprites(self):
        return self.get_image('SPRITES')

class Font:
    def __init__(self, image):
        self.image = image
        self.character_offsets = {}

    def add_character_offset(self, char, offset, width):
        self.character_offsets[char] = (offset, width)

    def render(self, words, ink, paper):
        character_images = []
        total_width = 0
        height = self.image.get_height()
        scale = height // 8
        for c in words:
            offset, width = self.character_offsets[c]
            image = self.image.subsurface((scale * offset, 0), (scale * width, height))
            character_images.append(image)
            total_width += width
        text = pygame.Surface((scale * total_width, height))
        offset = 0
        for image in character_images:
            text.blit(image, (offset, 0))
            offset += image.get_width()
        paper_surface = pygame.Surface((text.get_width(), text.get_height()))
        paper_surface.fill(paper)
        ink_surface = pygame.Surface((text.get_width(), text.get_height()))
        ink_surface.fill(ink)
        text.set_colorkey(FONT_INK)
        ink_surface.blit(text, (0, 0))
        ink_surface.set_colorkey(FONT_PAPER)
        paper_surface.blit(ink_surface, (0, 0))
        return paper_surface

    def has_char(self, char):
        return char in self.character_offsets
