# -*- coding: utf-8 -*-
# Copyright 2008, 2010 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

from ai import *

class SkoolBuilder:
    def __init__(self, ini_file):
        self.sections = {}
        f = open(ini_file, 'r')
        section = None
        for line in f:
            if line[0] == '[':
                if section:
                    self.sections[section_name] = section
                section_name = line[1:line.index(']')].strip()
                section = []
            elif line.isspace():
                continue
            elif line.startswith(';'):
                continue
            else:
                section.append(line.strip())
        self.sections[section_name] = section
        f.close()

    def build_skool(self, skool):
        self.parse_sounds(skool.beeper)
        self.parse_sprite_groups(skool.cast)
        self.parse_characters(skool.cast)
        self.parse_eric(skool.cast)
        self.parse_catapult_pellets(skool.cast)
        self.parse_water_drop(skool.cast)
        self.parse_sherry_drop(skool.cast)
        self.parse_conker(skool.cast)
        self.parse_water(skool.cast)
        self.parse_stinkbombs(skool.cast)
        self.parse_mice(skool.cast)
        self.parse_frogs(skool.cast)
        self.parse_bike(skool)
        self.parse_timetable(skool.timetable)
        self.parse_special_playtimes(skool.timetable)
        self.parse_lessons(skool.timetable, skool.cast)
        self.parse_random_locations(skool.cast)
        self.parse_skool_locations(skool)
        self.parse_inventory(skool)
        self.parse_command_lists(skool.cast)
        self.parse_rooms(skool)
        self.parse_chairs(skool)
        self.parse_desks(skool)
        self.parse_desk_lid(skool.cast)
        self.parse_doors(skool)
        self.parse_windows(skool)
        self.parse_walls(skool)
        self.parse_staircases(skool)
        self.parse_floors(skool)
        self.parse_routes(skool)
        self.parse_no_go_zones(skool)
        self.parse_sit_down_messages(skool.cast)
        self.parse_character_widths(skool.font)
        self.parse_assembly_messages(skool)
        self.parse_blackboards(skool)
        self.parse_blackboard_messages(skool.cast)
        self.parse_questions_and_answers(skool.cast)
        self.parse_lines_messages(skool.cast)
        self.parse_lesson_messages(skool.cast)
        self.parse_shields(skool)
        self.parse_safe(skool)
        self.parse_cups(skool)
        self.parse_plants(skool)
        self.parse_grass_config(skool.cast)
        self.parse_skool_messages(skool)

    def get_elements(self, details, parse_ints=True):
        elements = []
        index = 0
        while index < len(details):
            if details[index].isspace():
                index += 1
            elif details[index] in '"\'':
                quote = details[index]
                end = details.index(quote, index + 1)
                elements.append(details[index + 1:end])
                index = details.find(',', end + 1)
                if index < 0:
                    index = len(details)
                index += 1
            elif details[index] == '(':
                end = details.index(')', index)
                coords = details[index + 1:end].split(',')
                elements.append((int(coords[0]), int(coords[1])))
                index = details.find(',', end + 1)
                if index < 0:
                    index = len(details)
                index += 1
            else:
                end = details.find(',', index)
                if end < 0:
                    end = len(details)
                element = details[index:end].strip()
                if element.isdigit() and parse_ints:
                    elements.append(int(element))
                else:
                    elements.append(element)
                index = end + 1
        return elements

    def parse_section(self, name, parse_ints=True):
        if name in self.sections:
            return [self.get_elements(line, parse_ints) for line in self.sections[name]]
        return []

    def get_sections(self, prefix, parse_ints=True):
        sections = {}
        for name in self.sections:
            if name.startswith(prefix):
                sections[name[len(prefix):].strip()] = self.parse_section(name, parse_ints)
        return sections

    def get_config(self, name):
        config = {}
        for key, value in self.parse_section(name):
            config[key] = value
        return config

    def get_game_config(self):
        return self.get_config('Game')

    def get_gallery_config(self):
        return self.get_config('Images')

    def parse_sounds(self, beeper):
        for sound_id, sound_file in self.parse_section('Sounds'):
            beeper.add_sound(sound_id, sound_file)

    def parse_sprite_groups(self, cast):
        for group_id, section in self.get_sections('SpriteGroup ').items():
            for sprite_id, sprite_index in section:
                cast.add_sprite(group_id, sprite_id, sprite_index)

    def parse_eric(self, cast):
        for character_id, name, sprite_group, initial_as, location, flags in self.parse_section('Eric'):
            cast.add_eric(character_id, name, sprite_group, initial_as, flags.upper())
            cast.set_location(character_id, *location)

    def parse_characters(self, cast):
        for character_id, name, sprite_group, initial_as, location, flags in self.parse_section('Characters'):
            cast.add_character(character_id, name, sprite_group, initial_as, flags.upper())
            cast.set_location(character_id, *location)

    def parse_catapult_pellets(self, cast):
        for character_id, pellet_id, sprite_group, tap_id, pellet_range, hit_zone in self.parse_section('CatapultPellets'):
            cast.add_pellet(character_id, pellet_id, sprite_group, tap_id, int(pellet_range), int(hit_zone))

    def parse_water_drop(self, cast):
        for object_id, sprite_group_id, tap_id in self.parse_section('WaterDrop'):
            cast.add_water_drop(object_id, sprite_group_id, tap_id)

    def parse_sherry_drop(self, cast):
        for object_id, sprite_group_id, tap_id in self.parse_section('SherryDrop'):
            cast.add_sherry_drop(object_id, sprite_group_id, tap_id)

    def parse_conker(self, cast):
        for object_id, sprite_group_id, tap_id, min_x, max_x, min_y, max_y in self.parse_section('Conker'):
            cast.add_conker(object_id, sprite_group_id, tap_id, min_x, max_x, min_y, max_y)

    def parse_water(self, cast):
        for character_id, water_id, sprite_group, tap_id in self.parse_section('Water'):
            cast.add_water(character_id, water_id, sprite_group, tap_id)

    def parse_stinkbombs(self, cast):
        for character_id, stinkbomb_id, sprite_group, tap_id in self.parse_section('Stinkbombs'):
            cast.add_stinkbomb(character_id, stinkbomb_id, sprite_group, tap_id)

    def parse_mice(self, cast):
        for mouse_id, sprite_group_id, initial_as, location, tap_id in self.parse_section('Mice'):
            cast.add_mouse(mouse_id, sprite_group_id, initial_as, location, tap_id)

    def parse_frogs(self, cast):
        for frog_id, sprite_group_id, initial_as, location, tap_id in self.parse_section('Frogs'):
            cast.add_frog(frog_id, sprite_group_id, initial_as, location, tap_id)

    def parse_bike(self, skool):
        for bike_id, sprite_group_id, initial_as, location, tap_id, top_left, size, coords in self.parse_section('Bike'):
            skool.add_bike(bike_id, sprite_group_id, initial_as, location, tap_id, top_left, size, coords)

    def parse_timetable(self, timetable):
        for lesson_type in self.parse_section('Timetable'):
            timetable.add_lesson(*lesson_type)

    def parse_special_playtimes(self, timetable):
        for lesson_type in self.parse_section('SpecialPlaytimes'):
            timetable.add_special_playtime(*lesson_type)

    def parse_lessons(self, timetable, cast):
        for details, section in self.get_sections('Lesson ').items():
            index = details.index(' ')
            lesson_id = details[:index]
            lesson_details = [e.strip() for e in details[index + 1:].split(',')]
            room_id = lesson_details[-1]
            teacher_id = lesson_details[0] if len(lesson_details) > 1 else ''
            hide_teacher = teacher_id.startswith('*')
            if hide_teacher:
                teacher_id = teacher_id[1:]
            timetable.add_lesson_details(lesson_id, hide_teacher, teacher_id, room_id)
            for character_id, tap_id in section:
                cast.add_command_list(character_id, lesson_id, tap_id)

    def parse_random_locations(self, cast):
        for elements in self.parse_section('RandomLocations'):
            character_id = elements[0]
            locations = elements[1:]
            cast.set_random_locations(character_id, locations)

    def parse_skool_locations(self, skool):
        for location_id, x, y in self.parse_section('SkoolLocations'):
            skool.add_location(location_id, (int(x), int(y)))

    def parse_inventory(self, skool):
        for item_id, top_left, size in self.parse_section('Inventory'):
            skool.add_inventory_item(item_id, top_left, size)

    def parse_command_lists(self, cast):
        for tap_id, section in self.get_sections('CommandList ').items():
            for elements in section:
                command_class = eval(elements[0])
                params = []
                for e in elements[1:]:
                    try:
                        params.append(int(e))
                    except ValueError:
                        params.append(e)
                cast.add_command(tap_id, command_class, *params)

    def parse_rooms(self, skool):
        for room_id, name, y, min_x, max_x, get_along in self.parse_section('Rooms'):
            skool.add_room(room_id, name, y, min_x, max_x, get_along.upper() == 'Y')

    def parse_chairs(self, skool):
        for elements in self.parse_section('Chairs'):
            room_id = elements[0]
            for x in elements[1:]:
                skool.add_chair(room_id, x)

    def parse_desks(self, skool):
        for elements in self.parse_section('Desks'):
            room_id = elements[0]
            for x in elements[1:]:
                skool.add_desk(room_id, x)
        skool.fill_desks()

    def parse_desk_lid(self, cast):
        for desk_lid_id, sprite_group_id, tap_id in self.parse_section('DeskLid'):
            cast.add_desk_lid(desk_lid_id, sprite_group_id, tap_id)

    def parse_doors(self, skool):
        for door_id, x, bottom_y, top_y, initially_shut, auto_shuts, shut_top_left, size, coords in self.parse_section('Doors'):
            initially_shut = initially_shut.upper() == 'Y'
            auto_shuts = auto_shuts.upper() == 'Y'
            skool.add_door(door_id, x, bottom_y, top_y, initially_shut, auto_shuts, shut_top_left, size, coords)

    def parse_windows(self, skool):
        for window_id, x, bottom_y, top_y, initially_shut, opener_coords, shut_top_left, size, coords in self.parse_section('Windows'):
            initially_shut = initially_shut.upper() == 'Y'
            skool.add_window(window_id, x, bottom_y, top_y, initially_shut, opener_coords, shut_top_left, size, coords)

    def parse_walls(self, skool):
        for wall_id, x, bottom_y, top_y in self.parse_section('Walls'):
            skool.add_wall(wall_id, x, bottom_y, top_y)

    def parse_staircases(self, skool):
        for elements in self.parse_section('Staircases'):
            staircase_ids = elements[0].partition(':')
            bottom = elements[1]
            top = elements[2]
            force = len(elements) == 4
            skool.add_staircase(staircase_ids[0], bottom, top, force, staircase_ids[2])

    def parse_floors(self, skool):
        for floor_id, min_x, max_x, y in self.parse_section('Floors'):
            skool.add_floor(floor_id, int(min_x), int(max_x), int(y))

    def parse_routes(self, skool):
        for elements in self.parse_section('Routes'):
            home_floor_id = elements[0]
            staircase_id = elements[-1]
            dest_floor_ids = elements[1:-1]
            skool.add_routes(home_floor_id, dest_floor_ids, staircase_id)

    def parse_no_go_zones(self, skool):
        for zone_id, min_x, max_x, bottom_y, top_y in self.parse_section('NoGoZones'):
            skool.add_no_go_zone(zone_id, int(min_x), int(max_x), int(bottom_y), int(top_y))

    def parse_sit_down_messages(self, cast):
        for character_id, message in self.parse_section('SitDownMessages'):
            cast.set_sit_down_message(character_id, message)

    def parse_character_widths(self, font):
        for char, offset, width in self.parse_section('CharacterWidths'):
            font.add_character_offset(char, int(offset), int(width))

    def parse_assembly_messages(self, skool):
        generator = skool.assembly_message_generator
        for elements in self.parse_section('AssemblyMessages'):
            if elements[0] == 'MESSAGE':
                generator.set_text(elements[1])
            elif elements[0] == 'VERB':
                generator.add_verb(elements[1])
            elif elements[0] == 'NOUN':
                generator.add_noun(elements[1])

    def parse_blackboards(self, skool):
        for room_id, x, y in self.parse_section('Blackboards'):
            skool.add_blackboard(room_id, int(x), int(y))

    def parse_blackboard_messages(self, cast):
        for character_id, message in self.parse_section('BlackboardMessages'):
            cast.add_blackboard_message(character_id, message)

    def parse_questions_and_answers(self, cast):
        for teacher_id, section in self.get_sections('QuestionsAndAnswers ', False).items():
            qa_generator = cast.get(teacher_id).get_qa_generator()
            for elements in section:
                entry_type = elements[0]
                if entry_type == 'SpecialGroup':
                    qa_generator.set_special_group(*elements[1:])
                elif entry_type == 'SpecialQuestion':
                    qa_generator.set_special_question(elements[1])
                elif entry_type == 'SpecialAnswer':
                    qa_generator.set_special_answer(elements[1])
                elif entry_type == 'Question':
                    question_id = elements[1]
                    qa_group = elements[2]
                    text = elements[3]
                    qa_generator.add_question(question_id, qa_group, text)
                elif entry_type == 'Answer':
                    question_id = elements[1]
                    text = elements[2]
                    qa_generator.add_answer(question_id, text)
                elif entry_type == 'QAPair':
                    qa_group = elements[1]
                    qa_generator.add_qa_pair(qa_group, elements[2], elements[3])

    def parse_lines_messages(self, cast):
        for character_id, message_id, message in self.parse_section('LinesMessages'):
            cast.add_lines_message(character_id, message_id, message)

    def parse_lesson_messages(self, cast):
        for elements in self.parse_section('LessonMessages'):
            character_id = elements[0]
            message = elements[1]
            condition = '' if len(elements) < 3 else elements[2]
            cast.add_lesson_message(character_id, message, condition)

    def parse_shields(self, skool):
        for x, y, score, image_index in self.parse_section('Shields'):
            skool.add_shield(x, y, score, image_index)

    def parse_safe(self, skool):
        for elements in self.parse_section('Safe'):
            x = elements[0]
            y = elements[1]
            score = elements[2]
            image_index = None if len(elements) < 4 else elements[3]
            skool.add_safe(x, y, score, image_index)

    def parse_cups(self, skool):
        for cup_id, empty_top_left, size, coords in self.parse_section('Cups'):
            skool.add_cup(cup_id, empty_top_left, size, coords)

    def parse_plants(self, skool):
        for plant_id, sprite_group_id, x, y, tap_id in self.parse_section('Plants'):
            skool.add_plant(plant_id, sprite_group_id, x, y, tap_id)

    def parse_grass_config(self, cast):
        for elements in self.parse_section('Grass'):
            param = elements[0]
            values = elements[1:]
            if param == 'Hitters':
                cast.set_hitters(values)
            elif param == 'Writers':
                cast.set_writers(values)
            elif param == 'HitTale':
                cast.set_hit_tale(values[0])
            elif param == 'WriteTale':
                cast.set_write_tale(values[0])
            elif param == 'AbsentTale':
                cast.set_absent_tale(values[0])

    def parse_skool_messages(self, skool):
        for message_id, message in self.parse_section('SkoolMessages'):
            skool.add_message(message_id, message)
