# -*- coding: utf-8 -*-
# Copyright 2008, 2010 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import random
import signals
from location import Location
from animatorystates import *
from room import BB_WIDTH

# Distance teacher walks back after wiping a blackboard
BB_BACKTRACK = 3

# Distance teacher paces up and down during class without a Q-and-A session
UP_DOWN_DISTANCE = 3

# Range used by the WalkAround command
WALKABOUT_RANGE = (1, 7)

# Probability that a teacher will write on the blackboard
P_WRITE_ON_BOARD = 0.375

# Probability that a lesson with Eric will have a Q-and-A session
P_QASESSION = 0.90625

# Probability that the bully or tearaway will hit or fire if conditions are
# suitable
P_HIT_FIRE = 0.5

# Distance a pellet will travel upwards after hitting a teacher's head
PELLET_VERTICAL = 5

# Water animation phases
WATER_PHASES = (
    (WATER1, 2, -1, 0),
    (WATER2, 2,  0, 1),
    (WATER3, 1,  0, 0),
    (WATER4, 0,  1, 0),
    (WATER4, 0,  1, 2)
)

# Delay before an opened desk lid closes
DESK_LID_DELAY = 10

# Delay before a knocked-out kid gets up
KO_DELAY = 10

# Delay before a dethroned kid gets up
DETHRONED_DELAY = 3

# Delay before a knocked-over teacher gets up
KNOCKED_OVER_DELAY = 15
# Delay before a knocked-over teacher gives lines
REPRIMAND_DELAY = 8

# Time to wait for Eric to respond before repeating message
TELL_ERIC_DELAY = 20

# Min and max number of sprint sessions a mortal mouse engages in
MOUSE_LIFE = (10, 41)
# Min and max number of sprints per session
MOUSE_SPRINTS = (2, 5)
# Min and max distance of any sprint
MOUSE_SPRINT_DISTANCE = (2, 5)
# Min and max delay before a hidden mouse reappears
MOUSE_HIDE_DELAY = (5, 20)

# Time a musophobe will spend trying to evade a mouse
EVADE_MOUSE_COUNT = 21

# Probability that the frog will move if Eric is nearby
P_HOP = 0.75
# Probability that the frog will turn round
P_TURN_ROUND = 0.25
# Probability that the frog will attempt a short hop (instead of a long hop) if
# not turning round
P_SHORT_HOP = 0.5
# Animation phases for the frog when turning round
FROG_TURN_ROUND = ((HOP1, 0, 1), (SIT, 0, -1))
# Animation phases for the frog when doing a short hop
FROG_SHORT_HOP = ((HOP1, 1, 1), (SIT, 0, 1))
# Animation phases for the frog when doing a long hop
FROG_LONG_HOP = ((HOP1, 1, 1), (HOP2, 1, 1), (HOP1, 1, 1), (SIT, 0, 1))

# Time a plant takes to grow before dying
PLANT_LIFE = 26
# Delay before a watered plant reaches half height
PLANT_DELAY1 = 12
# Delay before a watered plant reaches full height
PLANT_DELAY2 = 19

# ID of the barrier Eric must hit to invoke the FlyOverSkoolGate command
SKOOL_GATE = 'SkoolGate'
# Animation phases for the flight off the bike over the closed skool gate
FLIGHT_SKOOL_GATE = (
    (1, -1, WALK0),
    (1, -1, WALK0),
    (1, -1, WALK0),
    (1,  1, WALK0),
    (1,  1, SITTING_ON_CHAIR),
    (1,  0, SITTING_ON_FLOOR),
    (1,  1, SITTING_ON_FLOOR),
    (1,  1, SITTING_ON_FLOOR)
)
# Animation phases for climb over skool gate from watered plant
CLIMB_SKOOL_GATE = (
    (0, 0, WALK1),
    (1, 0, WALK2),
    (0, 0, WALK3),
    (1, 0, WALK0),
    (0, 0, WALK1),
    (1, 0, WALK2),
    (0, 1, WALK2),
    (0, 1, WALK2),
    (0, 1, WALK2)
)

# ID of the middle-floor window (to decide which descent animation to use)
MIDDLE_WINDOW = 'MiddleWindow'
# ID of the top-floor window (to decide which descent animation to use)
TOP_WINDOW = 'UpperWindow'
# Animation phases for descent from middle-floor window
DESCENT_MIDDLE_WINDOW = (
    (0, 0, WALK1),
    (1, 0, WALK2),
    (0, 0, WALK3),
    (1, 0, WALK0),
    (0, 1, WALK0),
    (0, 1, WALK0),
    (0, 1, WALK0),
    (0, 1, WALK0),
    (0, 1, WALK0),
    (0, 1, WALK0),
    (0, 1, WALK0),
    (0, 1, WALK0),
    (0, 1, WALK0),
    (0, 1, WALK0)
)
# Animation phases for descent from top-floor window
DESCENT_TOP_WINDOW = (
    (0, 0, WALK1),
    (1, 0, WALK2),
    (0, 0, WALK3),
    (1, 0, WALK0),
    (1, 1, WALK0),
    (1, 1, WALK0),
    (1, 1, WALK0),
    (1, 1, WALK0),
    (0, 1, WALK0),
    (0, 1, WALK0),
    (0, 1, WALK0),
    (0, 1, WALK0),
    (0, 1, WALK0),
    (0, 1, WALK0),
    (0, 1, SITTING_ON_CHAIR),
    (0, 1, SITTING_ON_CHAIR),
    (0, 1, SITTING_ON_CHAIR),
    (0, 1, SITTING_ON_FLOOR),
    (0, 1, SITTING_ON_FLOOR),
    (0, 1, SITTING_ON_FLOOR),
    (0, 1, KNOCKED_OUT)
)

# Minimum x-coordinate for character under control of ChaseEricOut command
CHASE_ERIC_OUT_MIN_X = 137

# Min and max number of steps to walk to find a spot to sit during assembly
ASSEMBLY_SIT_RANGE = (1, 4)

class CommandListTemplate:
    def __init__(self, tap_id):
        self.tap_id = tap_id
        self.commands = []

    def add_command(self, command_class, *params):
        self.commands.append((command_class, params))

    def get_commands(self, start):
        return [cmd_class(*params) for cmd_class, params in self.commands][start:]

class CommandList:
    def __init__(self):
        self.stack = []
        self.index = None
        self.restart_index = None
        self.template = None
        self.controlling_command = None
        self.controlling_command_timer = None
        self.subcommand = None

    def command(self, character):
        # If the controlling command is not already in control of the
        # character, give it a chance (unless the current command is
        # uninterruptible)
        if self.stack and self.is_interruptible() and self.controlling_command and self.controlling_command not in self.stack:
            self.stack.append(self.controlling_command)
        # Now give the subcommand a chance to run
        if self.subcommand and self.subcommand not in self.stack and self.is_interruptible():
            self.stack.append(self.subcommand)
        while True:
            if self.stack:
                command = self.stack[-1]
                subcommand = command.command(character)
                if subcommand is None:
                    break
                elif subcommand is not command:
                    self.stack.append(subcommand)
                elif self.stack:
                    self.stack.pop()
            else:
                # Remove any controlling command before moving to the next
                # command
                if self.controlling_command:
                    self.controlling_command_timer -= 1
                    if self.controlling_command_timer < 0:
                        self.controlling_command = None
                        character.trigger_speed_change()
                self.stack.append(self.commands[self.index])
                self.index += 1

    def restart(self, index=None):
        self.index = index or 0
        self.commands = self.template.get_commands(self.restart_index)

    def add_command(self, command):
        self.stack.append(command)

    def set_template(self, template):
        self.template = template
        self.subcommand = None
        while self.stack and self.stack[0].is_interruptible():
            self.stack.pop(0)
        self.restart_index = 0
        self.restart()

    def set_controlling_command(self, command):
        self.controlling_command = command
        self.controlling_command_timer = 1

    def set_restart_point(self):
        """Discard the current and all previous commands."""
        self.restart_index = self.index
        self.restart()

    def jump(self, offset):
        """Jump forwards (or backwards) in the list of commands."""
        self.restart(self.index + offset)

    def is_GoToing(self):
        """Return whether the character is under the control of a GoTo* command."""
        return self.stack and self.stack[0].is_GoTo()

    def get_GoTo_destination(self):
        """Return the destination of the character, or None if he is not under the control of a GoTo* command."""
        if self.is_GoToing():
            return self.stack[0].get_destination()

    def set_GoTo_destination(self, destination):
        """Set the destination of the character if he is under the control of a GoTo* command."""
        if destination and self.is_GoToing():
            self.stack[0].set_destination(destination)

    def is_interruptible(self):
        for command in self.stack:
            if not command.is_interruptible():
                return False
        return True

    def set_subcommand(self, command_name, args):
        command_class = eval(command_name)
        self.subcommand = command_class(*args)

class Command:
    def is_interruptible(self):
        return True

    def is_GoTo(self):
        return False

class ComplexCommand(Command):
    def __init__(self):
        self.index = 0
        self.commands = self.get_commands()

    def command(self, character):
        val = self.commands[self.index](character)
        if val is not False:
            self.index = (self.index + 1) % len(self.commands)
        return val or None

    def restart(self):
        self.index = -1

    def done(self, character):
        return self

class SitForAssembly(ComplexCommand):
    def get_commands(self):
        return (self.find_spot_to_sit, self.sit_down, self.get_up, self.done)

    def find_spot_to_sit(self, character):
        if character.got_signal(signals.ASSEMBLY_FINISHED):
            return self
        return GoToXY(character.x - random.randint(*ASSEMBLY_SIT_RANGE), character.y)

    def sit_down(self, character):
        if character.got_signal(signals.ASSEMBLY_FINISHED):
            return self
        if character.direction < 0:
            character.turn()
            return False
        character.sit()

    def get_up(self, character):
        if character.got_signal(signals.ASSEMBLY_FINISHED):
            character.get_up()
        else:
            return False

class MoveDoor(ComplexCommand):
    def __init__(self, barrier_id, shut):
        ComplexCommand.__init__(self)
        self.barrier_id = barrier_id
        self.shut = shut

    def get_commands(self):
        return (self.raise_arm, self.move_door, self.lower_arm, self.done)

    def raise_arm(self, character):
        if character.check_door_status(self.barrier_id, self.shut):
            # Move on now if the door is already in the desired state
            return self
        character.raise_arm()

    def move_door(self, character):
        character.move_door(self.barrier_id, self.shut)

    def lower_arm(self, character):
        character.lower_arm()

class OpenDoor(MoveDoor):
    def __init__(self, barrier_id):
        MoveDoor.__init__(self, barrier_id, False)

class ShutDoor(MoveDoor):
    def __init__(self, barrier_id):
        MoveDoor.__init__(self, barrier_id, True)

class WipeBoard(ComplexCommand):
    def __init__(self):
        ComplexCommand.__init__(self)
        self.column = 0

    def get_commands(self):
        return (self.walk, self.wipe, self.lower_arm, self.done)

    def walk(self, character):
        character.wiping_board = True
        x = character.get_blackboard_edge() if self.column == 0 else character.x + character.direction
        return GoToXY(x, character.y)

    def wipe(self, character):
        character.raise_arm()
        wipe_col = self.column if character.direction > 0 else BB_WIDTH - 1 - self.column
        character.wipe_board(wipe_col, self.column == BB_WIDTH - 1)

    def lower_arm(self, character):
        self.column += 1
        character.lower_arm()
        if self.column < BB_WIDTH:
            # Repeat until the board is clean
            self.restart()
        else:
            character.wiping_board = False

    def is_interruptible(self):
        # Teacher should finish the job even if the bell rings
        return False

class GoTo(Command):
    def __init__(self, location_id, destination=None, go_one_step=False):
        self.location_id = location_id
        self.destination = destination
        self.go_one_step = go_one_step
        self.done = False

    def command(self, character):
        if self.destination is None:
            self.destination = character.resolve_location_id(self.location_id)
        if (character.x, character.y) == (self.destination.x, self.destination.y) or self.done:
            return self
        if character.on_stairs():
            character.walk(True)
            return
        if character.is_sitting():
            character.get_up()
            return
        next_staircase = character.get_next_staircase(self.destination)
        if next_staircase:
            if character.y == next_staircase.bottom.y:
                next_x = next_staircase.bottom.x
            else:
                next_x = next_staircase.top.x
        else:
            next_x = self.destination.x
        if character.x < next_x:
            self.done = self.go_one_step
            return character.right()
        elif character.x > next_x:
            self.done = self.go_one_step
            return character.left()
        else:
            if next_staircase.contains(character):
                character.walk(True)
            else:
                character.turn()
        self.done = self.go_one_step

    def get_destination(self):
        return self.destination

    def set_destination(self, destination):
        self.destination = destination

    def is_GoTo(self):
        return True

class GoToRandomLocation(GoTo):
    def __init__(self):
        GoTo.__init__(self, None)

    def command(self, character):
        if self.destination is None:
            self.destination = Location(character.get_random_destination())
        return GoTo.command(self, character)

class GoToXY(GoTo):
    def __init__(self, x, y):
        GoTo.__init__(self, None, Location((x, y)))

class GoTowardsXY(GoTo):
    def __init__(self, x, y):
        GoTo.__init__(self, None, Location((x, y)), True)

class Restart(Command):
    def command(self, character):
        character.restart_table()
        return self

class FindSeat(Command):
    def __init__(self, go_to_back=True, move_along=True):
        self.go_to_back = go_to_back
        self.move_along = move_along

    def command(self, character):
        next_chair, direction = character.get_next_chair(self.move_along, self.go_to_back)
        self.move_along = False
        if character.x == next_chair.x:
            if character.direction == direction:
                character.sit()
                return self
            character.turn()
        else:
            return GoToXY(next_chair.x, next_chair.y)

class MoveAboutUntil(Command):
    def __init__(self, signal):
        self.signal = signal

    def command(self, character):
        if character.got_signal(self.signal):
            return self
        return WalkAround(1)

class WalkAround(Command):
    def __init__(self, walkabouts):
        self.count = walkabouts
        self.origin = None

    def command(self, character):
        if self.count == 0:
            return self
        # Cannot move about an x-coordinate on a staircase, so get off the
        # stairs first
        if character.on_stairs():
            character.walk(True)
            return
        if self.origin is None:
            self.origin = character.x
        if character.x == self.origin:
            return GoToXY(character.x - random.randint(*WALKABOUT_RANGE), character.y)
        self.count -= 1
        return GoToXY(self.origin, character.y)

class SitStill(Command):
    def command(self, character):
        return

class GrassAndAnswerQuestions(Command):
    def __init__(self):
        self.ready = False

    def command(self, character):
        if not character.is_sitting_on_chair():
            return FindSeat(False, False)
        character.keep_seat()
        if not self.ready:
            self.ready = True
            character.signal(signals.SWOT_READY)
            character.start_lesson()
        return character.next_swot_action()

class WaitUntil(Command):
    def __init__(self, signal):
        self.signal = signal

    def command(self, character):
        if character.got_signal(self.signal):
            return self

class DoAssemblyDuty(Command):
    def __init__(self):
        self.fetched_eric = False

    def command(self, character):
        if self.fetched_eric:
            character.restart_table()
            return self
        if character.got_signal(signals.ASSEMBLY_FINISHED):
            return self
        if not character.is_time_to_start_lesson():
            return
        character.signal(signals.TIME_FOR_ASSEMBLY)
        character.set_home_room()
        if character.is_eric_absent():
            self.fetched_eric = True
            return FetchEric()

class AwaitAssemblyTime(Command):
    def command(self, character):
        if not character.got_signal(signals.TIME_FOR_ASSEMBLY):
            character.restart_table()
        return self

class ConductAssembly(Command):
    def __init__(self):
        self.spoken = False

    def command(self, character):
        if self.spoken:
            character.signal(signals.ASSEMBLY_FINISHED)
            character.unset_home_room()
            return self
        self.spoken = True
        return Say(character.get_assembly_message())

class StartLessonIfReady(Command):
    def __init__(self, signal):
        self.signal = signal
        self.ready = False

    def command(self, character):
        if not self.ready:
            if character.is_time_to_start_lesson():
                self.ready = True
                character.signal(self.signal)
                character.reveal_safe_secret(False)
                return TellKidsToSitDown()
            else:
                character.restart_table()
        return self

class StartDinnerIfReady(Command):
    def command(self, character):
        if character.is_time_to_start_lesson():
            character.set_home_room()
        else:
            character.restart_table()
        return self

class TellKidsToSitDown(Command):
    def __init__(self):
        self.spoken = False

    def command(self, character):
        if self.spoken:
            return self
        self.spoken = True
        return Say(character.sit_down_message)

class ConductClass(Command):
    def command(self, character):
        if character.is_teaching_eric():
            if character.got_signal(signals.SWOT_READY):
                return ConductClassWithEric()
        else:
            return ConductClassWithoutEric()

class ConductClassWithoutEric(Command):
    def __init__(self):
        self.wiped_board = False
        self.walked_to_board = False
        self.wrote_on_board = False
        self.told_class = False

    def command(self, character):
        if character.room.has_blackboard():
            if not self.wiped_board:
                self.wiped_board = True
                return WipeBoard()
            if not self.walked_to_board:
                self.walked_to_board = True
                return GoToXY(character.x - BB_BACKTRACK * character.direction, character.y)
            if not self.wrote_on_board:
                self.wrote_on_board = True
                if random.random() < P_WRITE_ON_BOARD:
                    return WriteOnBoard(character.get_blackboard_message())
        if not self.told_class:
            self.told_class = True
            return TellClassWhatToDo()
        return WalkUpOrDown()

class TellClassWhatToDo(Command):
    def __init__(self):
        self.done = False

    def command(self, character):
        if self.done:
            return self
        self.done = True
        return Say(character.get_lesson_message())

class WalkUpOrDown(Command):
    def __init__(self):
        self.done = False

    def command(self, character):
        if self.done:
            return self
        self.done = True
        return GoToXY(character.x - UP_DOWN_DISTANCE * character.direction, character.y)

class WriteOnBoard(Command):
    def __init__(self, message):
        self.message = message
        self.index = 1
        self.arm_down = True

    def command(self, character):
        if character.room.blackboard:
            if character.write_on_board(self.message, self.index):
                character.lower_arm()
                return self
            if self.arm_down:
                character.raise_arm()
            else:
                character.lower_arm()
            self.arm_down = not self.arm_down
        self.index += 1

    def is_interruptible(self):
        return False

class ConductClassWithEric(Command):
    def __init__(self):
        self.joined = False

    def command(self, character):
        if not self.joined:
            self.joined = True
            character.join_lesson(random.random() < P_QASESSION)
        return character.next_teacher_action()

class Say(Command):
    def __init__(self, words, notify=False):
        self.words = words
        self.notify = notify
        self.shift = -4

    def command(self, character):
        if character.say(self.words, self.shift):
            character.remove_bubble()
            if self.notify:
                character.finished_speaking()
            return self
        self.shift += 1

class Signal(Command):
    def __init__(self, signal):
        self.signal = signal

    def command(self, character):
        character.signal(self.signal)
        return self

class Unsignal(Command):
    def __init__(self, signal):
        self.signal = signal

    def command(self, character):
        character.unsignal(self.signal)
        return self

class SetControllingCommand(Command):
    def __init__(self, command_name, *params):
        self.command_name = command_name
        self.params = params

    def command(self, character):
        command_class = eval(self.command_name)
        character.set_controlling_command(command_class(*self.params))
        return self

class HitOrFireNowAndThen(Command):
    def ready(self, character):
        if not character.is_standing():
            return False
        if character.on_stairs():
            return False
        if character.get_nearby_adults():
            return False
        return random.random() < P_HIT_FIRE

    def command(self, character):
        if not self.ready(character):
            return self
        return self.get_command()

class FireNowAndThen(HitOrFireNowAndThen):
    def ready(self, character):
        if not HitOrFireNowAndThen.ready(self, character):
            return False
        if character.x % 4 != 0:
            return False
        return not character.pellet.is_visible()

    def get_command(self):
        return FireCatapult()

class FireCatapult(ComplexCommand):
    def get_commands(self):
        return (self.aim, self.fire, self.lower, self.done)

    def aim(self, character):
        if character.is_raising_catapult():
            character.aim_catapult()
        else:
            character.raise_catapult()
            return False

    def fire(self, character):
        character.fire_catapult()

    def lower(self, character):
        if character.is_lowering_catapult():
            character.complete_action()
        else:
            character.lower_catapult()
            return False

    def is_interruptible(self):
        return False

class MovePellet(Command):
    def __init__(self):
        self.count = 0
        self.x_inc = None
        self.y_inc = 0

    def command(self, pellet):
        if not pellet.is_visible():
            return
        if self.count <= 0:
            self.count = pellet.pellet_range
            self.x_inc = pellet.direction
            self.y_inc = 0
        pellet.run(True)
        self.count -= 1
        if self.count == 0 or pellet.barrier:
            self.end_flight(pellet)
            return
        if self.y_inc < 0 or self.count <= pellet.hit_zone:
            if pellet.hit_shield() or pellet.hit_cup() or pellet.hit_conker():
                self.end_flight(pellet)
                return
        if self.y_inc == 0 and self.count <= pellet.hit_zone:
            victim = pellet.get_victim()
            if victim:
                if victim.is_deckable():
                    victim.deck()
                    self.end_flight(pellet)
                    return
                if victim.is_adult() and victim.is_knocked_over():
                    self.x_inc = 0
                    self.y_inc = -1
                    self.count = PELLET_VERTICAL
        pellet.x += self.x_inc
        pellet.y += self.y_inc

    def end_flight(self, pellet):
        self.count = 0
        pellet.hide()

    def is_interruptible(self):
        return False

class FireWaterPistol(ComplexCommand):
    def get_commands(self):
        return (self.aim, self.fire, self.lower, self.done)

    def aim(self, character):
        character.aim_water_pistol()

    def fire(self, character):
        character.fire_water_pistol()

    def lower(self, character):
        character.complete_action()

    def is_interruptible(self):
        return False

class DumpWaterPistol(ComplexCommand):
    def get_commands(self):
        return (self.dump_water_pistol, self.stand_up)

    def dump_water_pistol(self, eric):
        eric.dump_water_pistol()

    def stand_up(self, eric):
        eric.stand_up()
        return self

class MoveWater(Command):
    def __init__(self):
        self.phase = 0
        self.phases = WATER_PHASES

    def command(self, water):
        if not water.is_visible():
            return
        water.run(True)
        state, x_inc, y_inc, hit = self.phases[self.phase]
        if hit == 1 and water.hit_cup():
            return self.end_flight(water)
        if hit == 2 and water.hit_plant():
            return self.end_flight(water)
        if water.hit_floor():
            return self.end_flight(water)
        water.x += water.direction * x_inc
        water.y += y_inc
        water.animatory_state = state
        self.phase = min(self.phase + 1, len(self.phases) - 1)

    def end_flight(self, water):
        self.phase = 0
        water.hide()

    def is_interruptible(self):
        return False

class DropStinkbomb(ComplexCommand):
    def get_commands(self):
        return (self.raise_arm, self.drop, self.lower, self.done)

    def raise_arm(self, character):
        character.raise_arm()

    def drop(self, character):
        character.drop_stinkbomb()

    def lower(self, character):
        character.complete_action()

    def is_interruptible(self):
        return False

class Stink(Command):
    def __init__(self):
        self.reinitialise()

    def reinitialise(self):
        self.count = 29

    def command(self, bomb):
        if not bomb.is_visible():
            return
        self.count -= 1
        if self.count <= 0:
            self.reinitialise()
            bomb.hide()
            return
        bomb.run(True)
        bomb.move_cloud()

class MoveDeskLid(ComplexCommand):
    def __init__(self):
        self.reinitialise()

    def reinitialise(self):
        self.count = DESK_LID_DELAY

    def command(self, desk_lid):
        if not desk_lid.is_visible():
            return
        desk_lid.run(True)
        if self.count >= DESK_LID_DELAY:
            desk_lid.deliver_contents()
        self.count -= 1
        if self.count <= 0:
            self.reinitialise()
            desk_lid.hide()

class HitNowAndThen(HitOrFireNowAndThen):
    def ready(self, character):
        return HitOrFireNowAndThen.ready(self, character) and self.in_front_of_victim(character)

    def in_front_of_victim(self, character):
        return len(character.get_punchees(3)) > 0

    def get_command(self):
        return Hit()

class Hit(ComplexCommand):
    def get_commands(self):
        return (self.aim, self.hit, self.lower, self.done)

    def aim(self, character):
        if character.is_raising_fist():
            character.punch()
        else:
            character.raise_fist()
            return False

    def hit(self, character):
        facing_characters = character.get_punchees(2)
        if facing_characters:
            facing_characters[0].deck()

    def lower(self, character):
        if character.is_lowering_fist():
            character.complete_action()
        else:
            character.lower_fist()
            return False

    def is_interruptible(self):
        return False

class Floored(Command):
    def __init__(self, count):
        self.count = count

    def command(self, character):
        if character.is_standing():
            return self
        self.count -= 1
        if self.count < 0:
            character.stand_up()

    def is_interruptible(self):
        return False

class KnockedOut(Floored):
    def __init__(self):
        Floored.__init__(self, KO_DELAY)

class Dethroned(Floored):
    def __init__(self):
        Floored.__init__(self, DETHRONED_DELAY)

class KnockedOver(Floored):
    def __init__(self, sleep):
        Floored.__init__(self, KNOCKED_OVER_DELAY)
        self.sleep = sleep

    def command(self, character):
        if self.sleep:
            if character.is_time_to_wake():
                return self
            return
        if Floored.command(self, character) is self:
            return self
        if self.count == KNOCKED_OVER_DELAY - 1:
            character.reveal_safe_secret(True)
        if self.count == KNOCKED_OVER_DELAY - REPRIMAND_DELAY:
            character.reprimand()

class WriteOnBoardUnless(Command):
    def __init__(self, signal):
        self.signal = signal
        self.wrote_on_board = False

    def command(self, character):
        if character.got_signal(self.signal) or character.room.blackboard_dirty():
            return self
        if self.wrote_on_board:
            return self
        self.wrote_on_board = True
        return WriteOnBoard(character.get_blackboard_message())

class SetRestartPoint(Command):
    def command(self, character):
        character.set_restart_point()
        return self

class JumpIfShut(Command):
    def __init__(self, door_id, offset):
        self.door_id = door_id
        self.offset = offset

    def command(self, character):
        character.jump_if_shut(self.door_id, self.offset)
        return self

class JumpIfOpen(Command):
    def __init__(self, door_id, offset):
        self.door_id = door_id
        self.offset = offset

    def command(self, character):
        character.jump_if_open(self.door_id, self.offset)
        return self

class WalkFast(Command):
    def command(self, character):
        character.run(True)
        # Ensure that no speed change is triggered while this command is in
        # effect
        character.speed_change_delay = 10
        return self

class StalkAndHit(HitNowAndThen):
    def __init__(self, character_id):
        self.character_id = character_id

    def command(self, character):
        character.stalk(self.character_id)
        return HitNowAndThen.command(self, character)

class WaitAtDoor(Command):
    def __init__(self, door_id):
        self.door_id = door_id

    def command(self, character):
        if character.wait_at_door(self.door_id):
            return self

class Jump(ComplexCommand):
    def get_commands(self):
        return (self.up, self.down, self.done)

    def up(self, character):
        character.jump()

    def down(self, character):
        character.descend()

class Write:
    def __init__(self):
        self.finished = False

    def command(self, character):
        if self.finished:
            character.lower_arm()
            return self
        self.finished = character.write()

class FindEric(Command):
    def __init__(self):
        self.running = False

    def command(self, character):
        character.stop_clock()
        if character.is_beside_eric():
            if character.freeze_eric():
                return self
            return
        if not self.running:
            self.running = True
            character.set_controlling_command(WalkFast())
        return GoTowardsXY(*character.get_location_of_eric())

class Freeze:
    def command(self, eric):
        eric.check_understanding()

class TellEric(Command):
    def __init__(self, message):
        self.told_eric = False
        self.message = message

    def command(self, character):
        if self.told_eric:
            character.unfreeze_eric()
            return self
        self.told_eric = True
        return Say(character.expand_names(self.message))

class TellEricAndWait(Command):
    def __init__(self, message):
        self.told_eric = 0
        self.delay = TELL_ERIC_DELAY
        self.message = message

    def command(self, character):
        if self.told_eric == 0:
            self.told_eric = 1
            return Say(character.expand_names(self.message))
        if character.eric_understood():
            character.unfreeze_eric()
            return self
        self.delay -= 1
        if self.delay < 0:
            self.delay = TELL_ERIC_DELAY
            self.told_eric = 0

class SetClock(Command):
    def __init__(self, ticks):
        self.ticks = ticks

    def command(self, character):
        character.start_clock(self.ticks)
        return self

class CheckIfTouchingEric(Command):
    def __init__(self, eric_knows_signal, eric_has_mumps_signal):
        self.eric_knows_signal = eric_knows_signal
        self.eric_has_mumps_signal = eric_has_mumps_signal

    def command(self, character):
        if character.got_signal(self.eric_knows_signal) and character.is_touching_eric():
            character.signal(self.eric_has_mumps_signal)
        return self

class EndGame(Command):
    def command(self, character):
        character.end_game()

class AddLines(Command):
    def __init__(self, lines):
        self.lines = lines

    def command(self, character):
        character.add_lines(self.lines)
        return self

    def is_interruptible(self):
        return False

class FetchEric(Command):
    def __init__(self):
        self.running = False

    def command(self, character):
        if not character.is_eric_absent() or character.is_eric_expelled():
            character.set_controlling_command(None)
            return self
        if not self.running:
            self.running = True
            character.set_controlling_command(WalkFast())
        if not character.is_beside_eric():
            return GoTowardsXY(*character.get_location_of_eric())
        if not character.is_facing_eric():
            character.turn()

class FindEricIfMissing(Command):
    def command(self, character):
        if character.is_eric_absent():
            return FetchEric()
        return self

class TripPeopleUp(Command):
    def command(self, character):
        character.run(True)
        # Ensure that no speed change is triggered while this command is in
        # effect
        character.speed_change_delay = 10
        character.trip_people_up()
        return self

class Follow(Command):
    def __init__(self, character_id):
        self.target_id = character_id
        self.done = False

    def command(self, character):
        if self.done:
            return self
        self.done = True
        return GoTo(None, character.get_destination(self.target_id))

class MoveMouse(Command):
    def __init__(self):
        self.reset_sprints()
        self.hide_x = -1
        self.hide_delay = None
        self.life = random.randint(*MOUSE_LIFE)

    def reset_sprints(self):
        self.sprints = random.randint(*MOUSE_SPRINTS)
        self.reset_sprint_distance()

    def reset_sprint_distance(self):
        self.sprint_distance = random.randint(*MOUSE_SPRINT_DISTANCE)

    def command(self, mouse):
        mouse.run(True)
        if self.hide_x >= 0:
            if not mouse.immortal:
                self.life -= 1
                if self.life <= 0:
                    mouse.die()
                    return
            if self.hide_delay > 0:
                self.hide_delay -= 1
                return
            mouse.x = self.hide_x
            self.hide_x = -1
            self.reset_sprints()
        elif self.sprint_distance > 0:
            self.sprint_distance -= 1
            self.move(mouse)
        elif self.sprints > 0:
            self.sprints -= 1
            self.reset_sprint_distance()
            mouse.direction = random.choice((-1, 1))
            self.move(mouse)
        else:
            self.hide_x = mouse.x
            self.hide_delay = random.randint(*MOUSE_HIDE_DELAY)
            mouse.hide()

    def move(self, mouse):
        if mouse.is_blocked():
            mouse.turn()
        mouse.x += mouse.direction
        mouse.scare_people()

    def is_interruptible(self):
        return False

class EvadeMouse(Command):
    def __init__(self):
        self.count = EVADE_MOUSE_COUNT
        self.in_the_air = False
        self.on_a_chair = False
        self.old_as = None

    def command(self, character):
        if self.old_as is None:
            self.old_as = character.animatory_state
            if character.is_sitting():
                character.get_up()
                return
        self.count -= 1
        if self.count <= 0:
            if self.in_the_air or self.on_a_chair:
                character.y += 1
                self.in_the_air = self.on_a_chair = False
                return
            character.animatory_state = self.old_as
            return self
        character.run(False)
        if self.in_the_air:
            character.y += 1
            self.in_the_air = False
        elif not self.on_a_chair:
            if character.chair(False):
                self.on_a_chair = True
            character.y -= 1
            self.in_the_air = not self.on_a_chair

    def is_interruptible(self):
        return False

class MoveFrog(Command):
    def command(self, frog):
        if frog.trapped or not frog.is_visible():
            return
        if frog.falling:
            if frog.check_heads():
                # Make the frog bounce off the character's head
                frog.animatory_state = HOP1
                frog.direction = -1
                frog.x -= 1
            else:
                frog.y += 1
                if frog.get_floor():
                    frog.falling = False
                    frog.animatory_state = SIT
            return
        if not frog.is_eric_nearby() and random.random() < P_HOP:
            return
        frog.run(True)
        if random.random() < P_TURN_ROUND:
            phases = FROG_TURN_ROUND
        elif random.random() < P_SHORT_HOP:
            phases = FROG_TURN_ROUND if frog.is_blocked(1) else FROG_SHORT_HOP
        else:
            phases = FROG_TURN_ROUND if frog.is_blocked(3) else FROG_LONG_HOP
        return Hop(phases)

    def is_interruptible(self):
        return False

class Hop(Command):
    def __init__(self, phases):
        self.phases = phases
        self.index = 0

    def command(self, frog):
        if self.index >= len(self.phases):
            return self
        state, x_inc, dir_change = self.phases[self.index]
        frog.animatory_state = state
        frog.x += x_inc * frog.direction
        frog.direction *= dir_change
        self.index += 1

    def is_interruptible(self):
        return False

class Catch(ComplexCommand):
    def get_commands(self):
        return (self.catch_animal, self.stand_up)

    def catch_animal(self, eric):
        eric.catch_animal()

    def stand_up(self, eric):
        eric.stand_up()
        return self

class ReleaseMice(ComplexCommand):
    def get_commands(self):
        return (self.release_mice, self.stand_up)

    def release_mice(self, eric):
        eric.release_mice()

    def stand_up(self, eric):
        eric.stand_up()
        return self

class Grow(Command):
    def __init__(self):
        self.count = PLANT_LIFE

    def command(self, plant):
        if not plant.growing:
            return
        plant.run(True)
        self.count -= 1
        if self.count == PLANT_LIFE - PLANT_DELAY1:
            plant.appear()
        elif self.count == PLANT_LIFE - PLANT_DELAY2:
            plant.finish_growing()
        elif self.count <= 0:
            plant.die()
            self.count = PLANT_LIFE

    def is_interruptible(self):
        return False

class Fall(Command):
    def command(self, thing):
        if not thing.is_visible():
            return
        thing.run(True)
        if thing.floor or thing.hit_victim():
            thing.hide()
        thing.y += 1

    def is_interruptible(self):
        return False

class MoveBike(Command):
    def command(self, bike):
        if bike.momentum <= 0 or not bike.is_visible():
            return
        bike.wheel()
        if bike.momentum <= 0:
            bike.fall()

class RideBike(Command):
    def __init__(self, bike):
        self.bike = bike
        bike.prepare()

    def command(self, character):
        character.check_bike_keys()
        if character.rode_into_barrier(self.bike):
            self.bike.momentum = 0
        elif character.pedalled():
            self.bike.pedal()
            return
        elif character.dismounted():
            character.stand_up()
            self.bike.start_wheeling(character)
            character.dismount()
            return self
        elif character.sitting_on_saddle:
            if character.stood_on_saddle():
                self.bike.start_wheeling(character)
                character.stand_on_saddle()
        elif character.got_back_on_saddle():
            character.get_back_on_saddle()
            self.bike.hide()
        elif character.jumped():
            character.dismount()
            return JumpOffSaddle()
        else:
            # Eric is standing on the saddle of the bike; move him with it
            character.x += character.direction
        self.bike.momentum -= 1
        if self.bike.momentum <= 0:
            character.dismount()
            if character.sitting_on_saddle:
                self.bike.fall(character)
            else:
                self.bike.fall()
                if character.rode_into_barrier(self.bike, SKOOL_GATE):
                    return FlyOverSkoolGate()
            return FallToFloor()

class FallToFloor(Command):
    def __init__(self, land_on_feet=False):
        self.land_on_feet = land_on_feet

    def command(self, character):
        if character.floor:
            if not self.land_on_feet:
                character.fall_to_floor()
            return self
        character.y += 1

class Flight(Command):
    def __init__(self, phases, paralyse=False):
        self.index = 0
        self.phases = phases
        self.paralyse = paralyse

    def command(self, eric):
        if self.index < len(self.phases):
            x_inc, y_inc, state = self.phases[self.index]
            self.index += 1
            eric.fly(x_inc, y_inc)
            eric.animatory_state = state
            return
        eric.end_flight()
        if self.paralyse:
            eric.paralyse()
            return Freeze()
        return self

class FlyOverSkoolGate(Flight):
    def __init__(self):
        Flight.__init__(self, FLIGHT_SKOOL_GATE)

class JumpOutOfWindow(Flight):
    def __init__(self, window_id):
        paralyse = False
        if window_id == MIDDLE_WINDOW:
            phases = DESCENT_MIDDLE_WINDOW
        elif window_id == TOP_WINDOW:
            phases = DESCENT_TOP_WINDOW
            paralyse = True
        Flight.__init__(self, phases, paralyse)

class ClimbOverSkoolGate(Flight):
    def __init__(self):
        Flight.__init__(self, CLIMB_SKOOL_GATE)

class JumpOffSaddle(ComplexCommand):
    def get_commands(self):
        return (self.rise, self.reach, self.check_cup, self.fall, self.done)

    def rise(self, eric):
        eric.y -= 1

    def reach(self, eric):
        eric.y -= 1
        eric.animatory_state = ARM_UP

    def check_cup(self, eric):
        eric.check_cup()
        eric.animatory_state = SITTING_ON_FLOOR

    def fall(self, eric):
        if eric.floor:
            eric.fall_to_floor()
            return
        eric.y += 1
        return False

class Kiss(ComplexCommand):
    def __init__(self):
        ComplexCommand.__init__(self)
        self.kissee = None

    def get_commands(self):
        return (self.start_kiss, self.finish_kiss, self.done)

    def start_kiss(self, eric):
        self.kissee = eric.kissee()
        if self.kissee:
            self.kissee.pause()
            if self.kissee.will_kiss_eric():
                eric.x += eric.direction
                eric.hide()
                self.kissee.kiss_eric()
            else:
                self.kissee.raise_arm()
        else:
            eric.walk()
            eric.walk_delay = 4

    def finish_kiss(self, eric):
        if self.kissee:
            if self.kissee.is_kissing_eric():
                eric.kiss()
                eric.unhide()
                self.kissee.finish_kiss()
            else:
                self.kissee.lower_arm()
                eric.deck()
            self.kissee.resume()
        else:
            eric.stand_up()

class Pause(Command):
    def command(self, character):
        if not character.paused:
            return self

class WatchForEric(Command):
    def __init__(self, command_list_id, alert_message):
        self.command_list_id = command_list_id
        self.alert_message = alert_message

    def command(self, character):
        if character.should_stop_eric():
            character.raise_arm()
            character.stop_eric = True
            return StopEric(self.command_list_id, self.alert_message)
        return self

class StopEric(Command):
    def __init__(self, command_list_id, alert_message):
        self.raised_alarm = False
        self.command_list_id = command_list_id
        self.alert_message = alert_message

    def command(self, character):
        if not self.raised_alarm:
            self.raised_alarm = True
            character.raise_alarm(self.alert_message, self.command_list_id)
        if not character.should_stop_eric():
            character.lower_arm()
            character.stop_eric = False
            return self

class ShadowEric(Command):
    def __init__(self):
        self.running = False

    def command(self, character):
        if not self.running:
            self.running = True
            character.set_controlling_command(WalkFast())
        if not character.is_beside_eric():
            return GoTowardsXY(*character.get_location_of_eric())
        if not character.is_facing_eric():
            character.turn()

class SetSubcommand(Command):
    def __init__(self, command_name, *args):
        self.command_name = command_name
        self.args = args

    def command(self, character):
        character.set_subcommand(self.command_name, self.args)
        return self

class MonitorEric(Command):
    def __init__(self, command_list_id):
        self.command_list_id = command_list_id

    def command(self, character):
        if character.should_chase_eric():
            character.change_command_list(self.command_list_id)
        return self

class ChaseEricOut(Command):
    def __init__(self):
        self.running = False

    def command(self, character):
        if not self.running:
            self.running = True
            character.set_controlling_command(WalkFast())
        if character.x >= CHASE_ERIC_OUT_MIN_X:
            if not character.is_beside_eric():
                return GoTowardsXY(*character.get_location_of_eric())
            if not character.is_facing_eric():
                character.turn()
