# -*- coding: utf-8 -*-
# Copyright 2008, 2010 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import sys
import random
from location import Location
from ai import *
from animatorystates import *
import lines
import items
from lesson import Lesson, QAGenerator
import debug

# Valid safe combination characters
SAFE_SECRETS = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ'
# Valid bike combination characters
BIKE_SECRETS = '0123456789'
# Valid storeroom combination characters
STOREROOM_SECRETS = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ'
# Ink colour of the secret revelation message box
SECRET_INK = 7
# Paper colour of the secret revelation message box
SECRET_PAPER = 0

# Lines message template
LINES_MSG = '%i0 LINES'
# Min and max number of lines (divided by 100) that may be given
LINES_RANGE = (1, 8)
# Paper colour of the lines message box when Eric is the recipient
LINES_PAPER_ERIC = 2
# Paper colour of the lines message box when anyone else is the recipient
LINES_PAPER_OTHER = 4
# Ink colour of the lines message box
LINES_INK = 7

# Condition indicating a dirty blackboard
BOARD_DIRTY = 'BoardDirty'

# Initial value of a character's kiss counter
KISSES = 40
# How much the kiss counter is decremented by each kiss
KISS_DECREMENT = 7

# Maximum distance at which a stinkbomb can be smelt
STINK_RANGE = 3

# Character speeds
SLOW = 5
FAST = 3
FASTER = 2

# Maximum distance at which a musophobe can detect a mouse
MOUSE_PROXIMITY = 5

# Min and max values of the delay between a character's speed changes
SPEED_CHANGE_DELAY_RANGE = (16, 31)

# The height of a character's sprite
SPRITE_HEIGHT = 4

# What the skool clock is set to when a character is paralysed by a falling
# conker
CONKER_CLOCK_TICKS = 1200
# The time at which a paralysed character remobilises
CONKER_WAKE_TIME = 200

# Liquid IDs
WATER = 'WATER'
SHERRY = 'SHERRY'

# Min and max page numbers to substitute into a lesson message
PAGE_NUMBER_RANGE = (100, 999)

# The x-coordinate at or beyond which Eric is considered to be trying to escape
# from skool (96 is the x-coordinate of the door to the boys' skool)
ESCAPE_MIN_X = 96
# The minimum and maximum distance from the watcher that Eric must be to make
# the watcher raise the alarm
DANGER_ZONE = (2, 6)
# Ink colour of the alarm message box
ALARM_INK = 2
# Paper colour of the alarm message box
ALARM_PAPER = 6

# The x-coordinate at or beyond which Eric must be to make a monitor chase him
# away (160 is the x-coordinate of the entrance to the girls' skool)
CHASE_MIN_X = 160

class Character:
    def __init__(self, character_id, name, flags=''):
        self.character_id = character_id
        self.name = name
        self.flags = flags
        self.command_list = CommandList()
        self.x, self.y = -3, 0
        self.random_locations = ()
        self.direction = 1 # 1 = facing right, -1 = facing left
        self.vertical_direction = 0 # 1 = down, 0 = neither up nor down
        self.action = 0 # 0 = walking, 1 = still, 2 = writing, 3 = other
        self.walk_states = [WALK0, WALK1, WALK2, WALK3]
        self.speed = 1 # 1 = walk, 2 = run
        self.speed_change_delay = 1
        self.walk_delay = random.randrange(3, 6)
        self.action_delay = 1
        self.staircase = None
        self.barrier = None
        self.room = None
        self.floor = None
        self.command_lists = {}
        self.sit_down_message = None
        self.bubble = None
        self.blackboard_messages = []
        self.qa_generator = None
        self.wiping_board = False
        self.pellet = None
        self.water = None
        self.stinkbomb = None
        self.lines_message = None
        self.lines_messages = {}
        self.come_along_messages = []
        self.come_along_index = 0
        self.lesson_messages = []
        self.changes_seats = True
        self.special_answer = None
        self.safe_secret = None
        self.bike_secret = None
        self.storeroom_secret = None
        self.kisses = KISSES
        self.barrier_id = character_id
        self.stop_eric = False

    def set_subcommand(self, command_name, args):
        self.command_list.set_subcommand(command_name, args)

    def set_components(self, cast, skool, screen, beeper):
        self.cast = cast
        self.skool = skool
        self.screen = screen
        self.beeper = beeper

    def get_title(self):
        return self.title or self.name

    def set_initial_location(self, x, y):
        self.initial_location = Location((x, y))
        self.x = x
        self.y = y

    def initialise_special_answer(self):
        if self.qa_generator:
            self.special_answer = self.qa_generator.initialise_special_answer()

    def set_sit_down_message(self, message):
        self.sit_down_message = message

    def set_animatory_states(self, as_dict_L, as_dict_R):
        self.as_dict_L = as_dict_L
        self.as_dict_R = as_dict_R

    def initialise_animatory_state(self, initial_as):
        self.direction = self.initial_direction = -1 if initial_as[0].upper() == 'L' else 1
        self.animatory_state = self.initial_as = self.walk_states[int(initial_as[1:])]

    def reinitialise(self):
        self.x, self.y = self.initial_location.x, self.initial_location.y
        self.direction = self.initial_direction
        self.vertical_direction = 0
        self.animatory_state = self.initial_as
        self.bubble = None
        self.wiping_board = False
        self.action = 0
        self.run(False)
        self.speed_change_delay = 1
        self.command_list = CommandList()
        self.kisses = KISSES
        self.stop_eric = False

    def set_controlling_command(self, command):
        self.command_list.set_controlling_command(command)

    def is_adult(self):
        return 'A' in self.flags

    def is_conkerable(self):
        return 'C' in self.flags

    def can_open_doors(self):
        return 'D' in self.flags

    def can_expel_eric(self):
        return 'E' in self.flags

    def is_punchable(self):
        return 'F' in self.flags

    def can_shadow_eric(self):
        return 'H' in self.flags

    def has_safe_key(self):
        return 'K' in self.flags

    def can_give_lines(self):
        return 'L' in self.flags

    def is_scared_of_mice(self):
        return 'M' in self.flags

    def can_smell_stinkbombs(self):
        return 'N' in self.flags

    def is_pelletable(self):
        return 'P' in self.flags

    def can_receive_lines(self):
        return 'R' in self.flags

    def has_safe_secret(self):
        return 'S' in self.flags

    def is_a_teacher(self):
        return 'T' in self.flags

    def sometimes_runs(self):
        return 'W' not in self.flags

    def has_bike_secret(self):
        return 'X' in self.flags

    def has_storeroom_secret(self):
        return 'Y' in self.flags

    def is_very_conkerable(self):
        return 'Z' in self.flags

    def is_interruptible(self):
        """Return whether this character's current command can be interrupted."""
        return self.command_list.is_interruptible()

    def is_home(self, x):
        if 'G' in self.flags:
            return self.x > x
        elif 'B' in self.flags:
            return self.x < x - 1
        return True

    def is_punchable_now(self):
        return self.is_punchable() and (self.is_standing() or self.is_sitting_on_chair())

    def get_location(self):
        """Return the on-floor location that is closest to this character."""
        staircase = self.skool.staircase(self)
        if staircase and staircase.supports(self):
            if staircase.bottom.y - self.y < self.y - staircase.top.y:
                return staircase.bottom.coords()
            return staircase.top.coords()
        return self.x, self.skool.floor_below(self).y

    def check_mouse(self, mouse):
        if not (self.is_scared_of_mice() and self.is_interruptible() and self.bubble is None):
            return
        # The character must be on the same floor as and facing the mouse in
        # order to be scared of it
        if self.y == mouse.y and 0 <= (mouse.x - self.x) * self.direction <= MOUSE_PROXIMITY:
            self.add_command(EvadeMouse())

    def can_kiss_eric(self):
        return KISSING_ERIC in self.as_dict_L

    def will_kiss_eric(self):
        return self.kisses > 0

    def pause(self):
        self.paused = True
        self.add_command(Pause())

    def resume(self):
        self.paused = False

    def kiss_eric(self):
        self.previous_as = self.animatory_state
        self.animatory_state = KISSING_ERIC
        self.x += self.direction
        self.direction *= -1
        self.kisses = max(self.kisses - KISS_DECREMENT, 0)

    def is_kissing_eric(self):
        return self.animatory_state == KISSING_ERIC

    def finish_kiss(self):
        self.animatory_state = self.previous_as
        self.direction *= -1

    def get_image(self):
        as_dict = self.as_dict_R if self.direction > 0 else self.as_dict_L
        return (self.x, self.y, as_dict[self.animatory_state])

    def get_speech_bubble(self):
        return self.bubble or (0, 0, None)

    def set_random_locations(self, locations):
        self.random_locations = locations

    def get_random_destination(self):
        dest = (self.x, self.y)
        choice = len(self.random_locations)
        if choice == 0:
            return dest
        if choice == 1:
            return self.random_locations[0]
        while dest == (self.x, self.y):
            dest = random.choice(self.random_locations)
        return dest

    def add_command_list(self, lesson_id, tap_id):
        self.command_lists[lesson_id] = tap_id

    def get_tap_id(self, lesson_id):
        return self.command_lists[lesson_id]

    def set_command_list_template(self, template):
        self.command_list.set_template(template)
        self.lesson = None
        self.trigger_speed_change()

    def change_command_list(self, command_list_id):
        self.cast.change_command_list(self, command_list_id)

    def restart_table(self):
        self.command_list.restart()

    def trigger_speed_change(self):
        self.speed_change_delay = 1

    def reset_walk_delay(self):
        self.walk_delay = FAST if self.speed == 2 else SLOW

    def is_time_to_move(self):
        if self.bubble:
            self.action_delay = (self.action_delay + 1) % FAST
            return self.action_delay == 0
        if self.action in (1, 2):
            # Writing or not moving
            self.action_delay = (self.action_delay + 1) % SLOW
            return self.action_delay == 0
        if self.action == 3:
            # Perform hitting, firing etc. quickly
            self.action_delay = (self.action_delay + 1) % FASTER
            return self.action_delay == 0
        self.walk_delay -= 1
        if self.walk_delay > 0:
            return False
        self.speed_change_delay -= 1
        if self.speed_change_delay == 0:
            self.speed_change_delay = random.randint(*SPEED_CHANGE_DELAY_RANGE)
            if self.sometimes_runs():
                self.run(self.speed_change_delay & 1)
            else:
                self.run(False)
        self.reset_walk_delay()
        return True

    def move(self):
        self.give_lines_now()
        if not self.is_time_to_move():
            return
        if self.midstride():
            return self.walk()
        self.staircase = self.skool.staircase(self)
        self.barrier = self.skool.barrier(self)
        self.room = self.skool.room(self)
        self.floor = self.get_floor()
        self.command_list.command(self)

    def impeded(self, bottom_y, top_y):
        return top_y - SPRITE_HEIGHT < self.y <= bottom_y

    def can_open_door(self, door):
        return self.can_open_doors()

    def left(self):
        if self.direction > 0:
            self.turn()
        else:
            if self.barrier and self.barrier.x <= self.x:
                if self.barrier.is_door() and self.can_open_door(self.barrier):
                    return self.open_door()
                return GoToXY(self.x + 1, self.y)
            on_stairs = False
            if self.staircase:
                if self.staircase.direction < 0:
                    on_stairs = self.x != self.staircase.bottom.x or self.staircase.force
                else:
                    on_stairs = self.x != self.staircase.top.x or self.staircase.force
            self.walk(on_stairs)

    def right(self):
        if self.direction > 0:
            if self.barrier and self.x < self.barrier.x:
                if self.barrier.is_door() and self.can_open_door(self.barrier):
                    return self.open_door()
                return GoToXY(self.x - 1, self.y)
            on_stairs = False
            if self.staircase:
                if self.staircase.direction > 0:
                    on_stairs = self.x != self.staircase.bottom.x or self.staircase.force
                else:
                    on_stairs = self.x != self.staircase.top.x or self.staircase.force
            self.walk(on_stairs)
        else:
            self.turn()

    def up(self):
        if self.staircase:
            if self.direction == self.staircase.direction or (self.x == self.staircase.top.x and self.staircase.force):
                return self.walk(True)
            elif self.x != self.staircase.top.x:
                return self.turn()
        if self.direction > 0:
            self.right()
        else:
            self.left()

    def down(self):
        if self.staircase:
            if self.direction != self.staircase.direction or (self.x == self.staircase.bottom.x and self.staircase.force):
                return self.walk(True)
            elif self.x != self.staircase.bottom.x:
                return self.turn()
        if self.direction > 0:
            self.right()
        else:
            self.left()

    def turn(self):
        self.direction *= -1

    def get_walk_state_index(self):
        if self.animatory_state in self.walk_states:
            return self.walk_states.index(self.animatory_state)
        return -1

    def walk(self, on_stairs=False):
        if on_stairs:
            if self.direction == self.staircase.direction:
                if not self.midstride():
                    self.y -= 1
            else:
                self.vertical_direction = 1
        walk_state = (self.get_walk_state_index() + 1) % len(self.walk_states)
        if walk_state % 2 == 0:
            self.x += self.direction
            self.y += self.vertical_direction
            if self.wiping_board:
                # Animatory state sequence is 0, 1, 0, 1... when wiping board
                walk_state = 0
            self.vertical_direction = 0
        self.animatory_state = self.walk_states[walk_state]
        self.action = 0

    def chair(self, check_dir=True):
        return self.skool.chair(self, check_dir)

    def can_sit_on_stairs(self):
        return True

    def sit(self):
        if self.is_sitting():
            self.get_up()
            return True
        self.previous_as = self.animatory_state
        chair = self.chair()
        if chair:
            self.animatory_state = SITTING_ON_CHAIR
            occupant = chair.occupant
            chair.seat(self)
            if occupant:
                occupant.dethrone()
        else:
            staircase = self.skool.staircase(self)
            if staircase and staircase.supports(self):
                if self.can_sit_on_stairs() and self.direction * staircase.direction < 0:
                    self.animatory_state = SITTING_ON_CHAIR
            else:
                self.sit_on_floor()
        self.action = 1
        return self.animatory_state != self.previous_as

    def stand_up(self):
        self.animatory_state = WALK0
        self.reset_walk_delay()

    def dethrone(self):
        self.sit_on_floor()
        self.add_command(FindSeat(False))
        self.add_command(Dethroned())

    def keep_seat(self):
        self.changes_seats = False

    def sit_on_floor(self):
        self.animatory_state = SITTING_ON_FLOOR

    def raise_arm(self):
        self.animatory_state = ARM_UP

    def lower_arm(self):
        self.animatory_state = WALK0

    def knock_over(self):
        self.previous_as = self.animatory_state
        if self.is_adult():
            self.animatory_state = KNOCKED_OVER
        else:
            self.animatory_state = KNOCKED_OUT
        self.action = 1

    def raise_fist(self):
        self.animatory_state = HITTING0
        self.action = 3

    def punch(self):
        self.animatory_state = HITTING1

    def lower_fist(self):
        self.animatory_state = HITTING0

    def raise_catapult(self):
        self.animatory_state = CATAPULT0
        self.action = 3

    def aim_catapult(self):
        self.animatory_state = CATAPULT1

    def lower_catapult(self):
        self.animatory_state = CATAPULT0

    def aim_water_pistol(self):
        self.animatory_state = WATERPISTOL

    def complete_action(self):
        self.animatory_state = WALK0
        self.action = 0

    def is_standing(self):
        return self.animatory_state in (WALK0, WALK1, WALK2, WALK3)

    def is_sitting(self):
        return self.animatory_state in (SITTING_ON_FLOOR, SITTING_ON_CHAIR)

    def is_sitting_on_floor(self):
        return self.animatory_state == SITTING_ON_FLOOR and self.get_floor()

    def is_sitting_on_chair(self):
        return self.animatory_state == SITTING_ON_CHAIR and self.chair()

    def is_sitting_on_stairs(self):
        return self.animatory_state == SITTING_ON_CHAIR and not self.chair()

    def is_knocked_over(self):
        return self.animatory_state == KNOCKED_OVER

    def has_arm_raised(self):
        return self.animatory_state == ARM_UP

    def is_knocked_out(self):
        return self.animatory_state == KNOCKED_OUT

    def is_raising_catapult(self):
        return self.animatory_state == CATAPULT0

    def is_firing_catapult(self):
        return self.animatory_state == CATAPULT1

    def is_lowering_catapult(self):
        return self.animatory_state == CATAPULT0

    def is_raising_fist(self):
        return self.animatory_state == HITTING0

    def is_punching(self):
        return self.animatory_state == HITTING1

    def is_lowering_fist(self):
        return self.animatory_state == HITTING0

    def is_riding_bike(self):
        return self.animatory_state in (RIDING_BIKE0, RIDING_BIKE1)

    def is_deckable(self):
        """Return whether this character is in an animatory state and location amenable to being knocked over."""
        if self.get_floor() is None:
            return
        if self.is_adult():
            return self.is_standing() or self.has_arm_raised()
        return self.is_standing() or self.is_sitting_on_chair()

    def is_door(self):
        return False

    def get_up(self):
        if self.is_sitting_on_chair():
            self.chair().vacate()
        self.animatory_state = self.previous_as

    def midstride(self):
        return self.animatory_state in (WALK1, WALK3)

    def get_floor(self, thing=None):
        return self.skool.floor(thing or self)

    def on_stairs(self):
        if self.staircase:
            return self.x not in (self.staircase.bottom.x, self.staircase.top.x)
        return False

    def get_next_staircase(self, destination):
        if self.get_floor() is None:
            debug.log('%s at %i, %i has no home floor' % (self.name, self.x, self.y))
        if self.get_floor(destination) is None:
            debug.log('%s at %i, %i going to %i, %i has no destination floor' % (self.name, self.x, self.y, destination.x, destination.y))
        return self.skool.next_staircase(self.get_floor(), self.get_floor(destination))

    def open_door(self):
        return OpenDoor(self.barrier.barrier_id)

    def move_door(self, barrier_id, shut):
        if shut:
            self.skool.move_bike_away(barrier_id)
        self.skool.move_door(barrier_id, shut)

    def get_next_chair(self, move_along, go_to_back):
        return self.room.get_next_chair(self, move_along and self.changes_seats, go_to_back)

    def signal(self, signal):
        self.skool.signal(signal)

    def unsignal(self, signal):
        self.skool.unsignal(signal)

    def got_signal(self, signal):
        return self.skool.got_signal(signal)

    def is_time_to_start_lesson(self):
        return self.skool.is_time_to_start_lesson()

    def say(self, words, shift):
        if not self.bubble:
            bubble_x = 8 * ((self.x + 1) // 8)
            bubble_y = self.y - (3 if self.is_adult() or self.is_standing() else 2)
            self.bubble = [bubble_x, bubble_y, None]
        lip_pos = (self.x + 1) % 8
        bubble_img, done = self.screen.get_bubble(words, lip_pos, shift)
        self.bubble[2] = bubble_img
        return done

    def remove_bubble(self):
        self.bubble = None

    def get_assembly_message(self):
        return self.skool.get_assembly_message()

    def get_blackboard_edge(self):
        """Return the x-coordinate to which this character should proceed to reach the edge of the blackboard that is closer to him."""
        if self.room and self.room.blackboard:
            if self.x < self.room.blackboard.x - 2:
                return self.room.blackboard.x - 2
            return self.room.blackboard.right_x

    def wipe_board(self, column, clear):
        self.skool.wipe_board(self.room.blackboard, column, clear)

    def write_on_board(self, message, index):
        self.action = 2
        return self.skool.write_on_board(self, self.room.blackboard, message, index)

    def resolve_location_id(self, location_id):
        return self.skool.resolve_location_id(location_id)

    def is_teaching_eric(self):
        return self.skool.is_teaching_eric(self)

    def get_qa_generator(self):
        if self.qa_generator is None:
            self.qa_generator = QAGenerator()
        return self.qa_generator

    def add_blackboard_message(self, message):
        self.blackboard_messages.append(message)

    def get_blackboard_message(self):
        message = random.choice(self.blackboard_messages)
        return self.cast.expand_names(message)

    def run(self, run):
        self.speed = 2 if run else 1

    def is_facing(self, thing):
        return self.direction * (thing.x - self.x) >= 0

    def get_punchees(self, offset):
        return self.cast.get_punchees(self, offset)

    def add_command(self, command):
        self.command_list.add_command(command)

    def deck(self, paralyse=False):
        self.knock_over()
        self.kisses = max(0, self.kisses - 1)
        if self.is_adult():
            self.add_command(KnockedOver(paralyse))
            if paralyse:
                self.beeper.make_conker_sound()
                self.skool.rewind_clock(CONKER_CLOCK_TICKS)
        else:
            if self.previous_as == SITTING_ON_CHAIR:
                self.chair().vacate()
                self.add_command(FindSeat(False, False))
            self.add_command(KnockedOut())

    def is_time_to_wake(self):
        return self.skool.is_time_remaining(CONKER_WAKE_TIME)

    def can_fire_catapult(self):
        if CATAPULT0 in self.as_dict_L and CATAPULT1 in self.as_dict_L:
            return self.pellet and not self.pellet.is_visible()

    def fire_catapult(self):
        """Launch a catapult pellet."""
        self.pellet.launch(self.x + self.direction, self.y, self.direction)

    def can_fire_water_pistol(self):
        return WATERPISTOL in self.as_dict_L and self.water and not self.water.is_visible()

    def fire_water_pistol(self, liquid=WATER):
        self.water.launch(self.x + 2 * self.direction, self.y - 2, self.direction, liquid)

    def can_drop_stinkbomb(self):
        return ARM_UP in self.as_dict_L and self.stinkbomb and not self.stinkbomb.is_visible()

    def drop_stinkbomb(self):
        self.stinkbomb.drop(self.x + self.direction, self.y)

    def get_nearby_adults(self):
        """Return a list of adults who are close enough to this character (and are facing the right way) to be able to see him."""
        return self.cast.get_nearby_adults(self)

    def set_restart_point(self):
        """Discard the current and all previous commands in the command list (so any command that restarts the command list will see the next command as the first)."""
        self.command_list.set_restart_point()

    def jump_if_shut(self, door_id, offset):
        """Jump forwards (or backwards) in the command list if the given door is shut."""
        if self.skool.is_door_shut(door_id):
            self.command_list.jump(offset)

    def jump_if_open(self, door_id, offset):
        """Jump forwards (or backwards) in the command list if the given door is open."""
        if not self.skool.is_door_shut(door_id):
            self.command_list.jump(offset)

    def check_door_status(self, barrier_id, shut):
        return self.skool.is_door_shut(barrier_id) == shut

    def stalk(self, character_id):
        """Set this character's destination equal to that of another character's destination (if they are both GoTo-ing)."""
        self.command_list.set_GoTo_destination(self.get_destination(character_id))

    def get_GoTo_destination(self):
        """Return the destination of this character, or None if he is not under the control of the GoTo command."""
        return self.command_list.get_GoTo_destination()

    def get_destination(self, character_id):
        return self.cast.get(character_id).get_GoTo_destination()

    def wait_at_door(self, door_id):
        """Return whether the characters are on the correct side of the given door."""
        return self.cast.is_home(self.skool.get_door(door_id).x)

    def is_eric_expelled(self):
        """Return whether Eric is due to be or in the process of being expelled."""
        return self.cast.is_eric_expelled()

    def get_come_along_message_id(self):
        message_id = self.come_along_messages[self.come_along_index]
        self.come_along_index = (self.come_along_index + 1) % len(self.come_along_messages)
        if self.come_along_index == 0 and len(self.come_along_messages) > 2:
            self.come_along_index = 1
        return message_id

    def print_lines_message(self, recipient_id, message_id):
        if not self.screen.contains(self):
            return
        recipient = self.cast.get(recipient_id)
        admonition = self.lines_messages.get(message_id, None)
        if not (recipient and admonition):
            return
        if recipient.is_eric() and recipient.expelled:
            return
        num_lines = 10 * random.randint(*LINES_RANGE)
        message = (LINES_MSG % num_lines, recipient.name)
        if recipient.is_eric():
            paper = LINES_PAPER_ERIC
            self.skool.add_lines(num_lines)
        else:
            paper = LINES_PAPER_OTHER
            self.skool.add_to_score(num_lines)
        self.print_lines_bubble(message, LINES_INK, paper, 0)
        self.print_lines_bubble(admonition, LINES_INK, paper, 1)

    def print_lines_bubble(self, message, ink, paper, sound):
        y = self.y - 2 if self.is_knocked_over() else self.y - 3
        self.screen.print_lines_bubble(self.x, y, message, ink, paper)
        self.beeper.make_lines_sound(sound)

    def give_lines(self, recipient_id, message_id, now=False):
        self.lines_message = (recipient_id, message_id)
        if now:
            self.give_lines_now()

    def give_lines_now(self):
        if self.lines_message:
            self.print_lines_message(*self.lines_message)
            self.lines_message = None

    def can_see_special_answer(self):
        """Return whether this character can see his special answer written on a blackboard."""
        if self.special_answer:
            blackboard = self.skool.visible_blackboard(self)
            if blackboard:
                return blackboard.shows(self.special_answer)

    def reveal_safe_secret(self, decked):
        """Make this character reveal his safe combination letter (if he has one, and conditions are right)."""
        if not self.safe_secret or not self.skool.can_reveal_safe_secret():
            return
        if not self.screen.contains(self):
            return
        if decked:
            reveal = self.special_answer is None
        else:
            reveal = self.can_see_special_answer()
        if reveal:
            self.print_lines_bubble(('', self.safe_secret), SECRET_INK, SECRET_PAPER, 1)

    def reveal_bike_secret(self):
        if self.bike_secret and self.screen.contains(self):
            self.print_lines_bubble(('', self.bike_secret), SECRET_INK, SECRET_PAPER, 1)

    def reveal_storeroom_secret(self):
        if self.storeroom_secret and self.screen.contains(self):
            self.print_lines_bubble(('', self.storeroom_secret), SECRET_INK, SECRET_PAPER, 1)

    def initialise_safe_secret(self):
        if self.has_safe_secret():
            self.safe_secret = random.choice(SAFE_SECRETS)
        return self.safe_secret

    def initialise_bike_secret(self):
        if self.has_bike_secret():
            self.bike_secret = random.choice(BIKE_SECRETS)
        return self.bike_secret

    def initialise_storeroom_secret(self):
        if self.has_storeroom_secret():
            self.storeroom_secret = random.choice(STOREROOM_SECRETS)
        return self.storeroom_secret

    def is_eric(self):
        return False

    def add_lines_message(self, message_id, message_lines):
        if message_id not in self.lines_messages:
            self.lines_messages[message_id] = message_lines
        if message_id.startswith(lines.COME_ALONG_PREFIX):
            self.come_along_messages.append(message_id)

    def can_give_lines_message(self, message_id):
        return message_id in self.lines_messages

    def add_lesson_message(self, message, condition):
        self.lesson_messages.append((message, condition))

    def lesson_message_condition(self, condition):
        if condition == BOARD_DIRTY:
            return self.room.blackboard_dirty()
        return True

    def get_lesson_message(self):
        while True:
            message, condition = random.choice(self.lesson_messages)
            if self.lesson_message_condition(condition):
                break
        return message.replace('$n', str(random.randint(*PAGE_NUMBER_RANGE)))

    def reprimand(self):
        """Make this character give lines to the nearest lines recipient for knocking him over."""
        recipient = self.cast.get_nearest_lines_recipient(self)
        if recipient:
            self.give_lines(recipient.character_id, lines.NEVER_AGAIN)

    def check_shields_at(self, x, y):
        return self.skool.hit_shield(x, y)

    def has_line_of_sight_to(self, character):
        """Return whether this character has a line of sight to the given character, i.e. there are no walls between them."""
        return self.skool.line_of_sight_between(self, character)

    def get_location_of_eric(self):
        return self.cast.get_location_of_eric()

    def is_facing_eric(self):
        return self.is_facing(self.cast.eric)

    def is_beside_eric(self):
        """Return whether this character is beside Eric."""
        return self.cast.is_beside_eric(self)

    def is_eric_absent(self):
        return self.cast.is_eric_absent()

    def freeze_eric(self):
        return self.cast.freeze_eric()

    def unfreeze_eric(self):
        self.cast.unfreeze_eric()

    def eric_understood(self):
        return self.cast.eric_understood()

    def expand_names(self, message):
        return self.cast.expand_names(message)

    def stop_clock(self):
        self.skool.stop_clock()

    def start_clock(self, ticks):
        self.skool.start_clock(ticks)

    def is_touching_eric(self):
        return self.cast.is_touching_eric(self)

    def add_lines(self, lines):
        self.skool.add_lines(lines // 10)

    def end_game(self):
        self.skool.end_game()

    def trip_people_up(self):
        self.cast.trip_people_up_at(self, self.x, self.y)

    def hide(self):
        self.x = -3

    def is_visible(self):
        return self.x >= 0

    def fall_off_plant(self):
        self.add_command(FallToFloor())

    def fall_to_floor(self):
        self.sit_on_floor()
        self.walk_delay = 10

    def can_smell_stinkbomb(self, stinkbomb):
        if self.can_smell_stinkbombs() and self.is_standing() and self.y == stinkbomb.y:
            return abs(stinkbomb.x - self.x) <= STINK_RANGE
        return False

    def open_window(self, window):
        if not self.command_list.is_GoToing():
            # The character should return to the location from which he took
            # his window-opening detour so that he can resume his command list
            # correctly
            self.add_command(GoToXY(self.x, self.y))
        self.add_command(OpenDoor(window.barrier_id))
        self.add_command(GoToXY(*window.opener_coords))

    def should_stop_eric(self):
        """Return whether this character should prevent Eric from escaping."""
        eric_x, eric_y = self.cast.get_location_of_eric()
        if eric_y != self.y or self.direction > 0:
            return False
        return eric_x >= ESCAPE_MIN_X and self.x - DANGER_ZONE[1] <= eric_x <= self.x - DANGER_ZONE[0]

    def raise_alarm(self, message, command_list_id):
        """Make this character raise the alarm that Eric is escaping."""
        self.print_lines_bubble(self.skool.expand_message(message), ALARM_INK, ALARM_PAPER, 1)
        if not self.cast.is_eric_expelled():
            self.cast.shadow_eric(command_list_id)

    def is_stopping_eric(self, eric):
        return self.stop_eric and eric.impeded(self.y + 3, self.y) and self.direction != eric.direction and abs(self.x - eric.x) == 2

    def should_chase_eric(self):
        if not self.skool.is_playtime():
            eric_x, eric_y = self.cast.get_location_of_eric()
            return eric_x >= CHASE_MIN_X and eric_y == self.y
        return False

    ############################################################################
    # Lesson coordination methods
    ############################################################################
    def start_lesson(self):
        """Called by the Swot to start a lesson."""
        self.lesson = Lesson(self.cast, self, self.room)
        self.skool.set_lesson(self.lesson)

    def join_lesson(self, with_qa):
        """Called by the teacher to join the lesson started by the Swot."""
        self.lesson = self.skool.get_lesson()
        self.lesson.join(self, self.qa_generator if with_qa else None)

    def next_swot_action(self):
        """Return the next command to be executed by the swot (which may be None)."""
        return self.lesson.next_swot_action()

    def next_teacher_action(self):
        """Return the next command to be executed by the teacher (which may be None)."""
        return self.lesson.next_teacher_action()

    def finished_speaking(self):
        """Called by the swot or the teacher when he's finished speaking."""
        self.lesson.finished_speaking()

    def set_home_room(self):
        self.skool.set_home_room()

    def unset_home_room(self):
        self.skool.unset_home_room()

    def reset_come_along_index(self):
        self.come_along_index = 0
