# -*- coding: utf-8 -*-
# Copyright 2008, 2010 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import os
import pygame
import random
from cast import Cast
import character
from skool import Skool
from timetable import Timetable
from graphics import Screen, Gallery
from sound import Beeper
from input import Keyboard
from skoolbuilder import SkoolBuilder
import keys
import items
import debug

class Game:
    def __init__(self, pyskool_dir, scale, ini_file, quick_start, version, icon_fname):
        pygame.init()
        builder = SkoolBuilder(ini_file)
        config = builder.get_game_config()

        images_dir = os.path.join(pyskool_dir, config['ImagesDir'])
        image_set = config['ImageSet']
        scale = scale or config['Scale']
        gallery_config = builder.get_gallery_config()
        gallery = Gallery(images_dir, image_set, scale, gallery_config)

        title = 'Pyskool %s: %s' % (version, config['Name'])
        self.create_screen(config, scale, gallery, title, icon_fname)
        self.create_skool(pyskool_dir, config, gallery, builder, config['MaxLines'] // 10)
        self.screen.initialise_column(config['InitialColumn'], self.skool)

        self.quick_start = quick_start or config['QuickStart']
        self.speed = 1
        self.fps = config['GameFps']

    def create_screen(self, config, scale, gallery, title, icon_fname):
        self.screen = Screen(config['GraphicsMode'], scale, config['Background'], gallery, config['ScrollFps'], title, icon_fname)
        self.screen.set_lesson_box_properties(config['LessonBoxAttribute'], config['LessonBoxPos'])
        self.screen.add_logo(config['LogoPos'])
        self.screen.add_score_box(config['ScoreBoxPos'], config['ScoreBoxXOffset'], config['ScoreBoxAttribute'])
        self.screen.add_mouse_inventory(config.get('MouseInventoryPos'))
        self.screen.add_inventory(config.get('InventoryPos'))

    def create_skool(self, pyskool_dir, config, gallery, builder, max_lines):
        beeper = Beeper(os.path.join(pyskool_dir, config['SoundsDir']))
        cast = Cast(gallery.get_sprites())
        timetable = Timetable(config['LessonLength'], config['LessonStartTime'])
        self.skool = Skool(self.screen, beeper, cast, timetable, gallery, max_lines)
        builder.build_skool(self.skool)
        self.skool.initialise_cast()

    def no_lines(self, *args):
        return

    def check_cheat_keys(self, keyboard):
        if keyboard.was_pressed(keys.NEXT_LESSON):
            self.skool.next_lesson(False)
        if keyboard.is_pressed(keys.SLOW):
            self.speed = 0.5
        elif keyboard.is_pressed(keys.FAST):
            self.speed = 2
        else:
            self.speed = 1
        if self.skool.shields:
            if keyboard.was_pressed(keys.FLASH_MOST):
                for shield in self.skool.shields[:-1]:
                    shield.flash()
                self.skool.shields[-1].unflash()
                self.skool.safe.unflash()
                self.skool.shield_mode = 1
                debug.log('Flashed all but one shield; hit it and then open the safe')
            if keyboard.was_pressed(keys.UNFLASH_MOST):
                for shield in self.skool.shields[:-1]:
                    shield.unflash()
                self.skool.shields[-1].flash()
                self.skool.safe.flash()
                self.skool.shield_mode = 3
                debug.log('Unflashed all but one shield; hit it to go up a year')
        if keyboard.was_pressed(keys.NO_LINES):
            character.Character.give_lines = self.no_lines
            debug.log('Disabled lines-giving')
        if keyboard.was_pressed(keys.ADD_LINES):
            lines = 10 * random.randrange(1, 9)
            self.skool.add_lines(lines)
            debug.log('Added %i lines' % (lines * 10))
        if keyboard.was_pressed(keys.ZERO_LINES):
            self.skool.add_lines(-self.skool.scoreboard.lines)
            debug.log('Set lines total to zero')
        if keyboard.was_pressed(keys.REVEAL):
            for room in self.skool.rooms.values():
                for desk in room.desks:
                    if desk.contents:
                        debug.log('%s x=%i: %s' % (room.name, desk.x, desk.contents))
            for c in self.skool.cast.character_list:
                if c.special_answer:
                    debug.log('%s: %s' % (c.name, c.special_answer))
            if self.skool.safe_combination:
                debug.log('Safe: %s' % self.skool.safe_combination)
            if self.skool.bike_combination:
                debug.log('Bike: %s' % self.skool.bike_combination)
            if self.skool.storeroom_combination:
                debug.log('Storeroom: %s' % self.skool.storeroom_combination)
        if self.skool.inventory_item_ids:
            eric = self.skool.cast.eric
            inventory = eric.inventory
            if keyboard.was_pressed(keys.SWITCH_PISTOL):
                if items.WATER_PISTOL in inventory:
                    inventory.remove(items.WATER_PISTOL)
                    inventory.add(items.SHERRY_PISTOL)
                elif items.SHERRY_PISTOL in inventory:
                    inventory.remove(items.SHERRY_PISTOL)
                    inventory.add(items.WATER_PISTOL)
                eric.print_inventory()
            if keyboard.was_pressed(keys.GIVE_ALL):
                inventory.update((items.SAFE_KEY, items.STOREROOM_KEY, items.FROG, items.WATER_PISTOL, items.STINKBOMBS3))
                eric.mice = 8
                eric.print_inventory()
                eric.print_mouse_inventory()
                self.skool.unchain_bike()
        if self.skool.doors:
            if keyboard.was_pressed(keys.OPEN_DOORS):
                for door_id in self.skool.doors:
                    self.skool.move_door(door_id, False)
                for window_id in self.skool.windows:
                    self.skool.move_door(window_id, False)
                debug.log('Opened all doors and windows')
            if keyboard.was_pressed(keys.CLOSE_DOORS):
                for door_id in self.skool.doors:
                    self.skool.move_door(door_id, True)
                for window_id in self.skool.windows:
                    self.skool.move_door(window_id, True)
                debug.log('Closed all doors and windows')

    def play(self, cheat=False):
        clock = pygame.time.Clock()

        self.ring_bell = False

        self.screenshot = 0
        self.paused = False
        keyboard = Keyboard()
        while True:
            if not self.quick_start:
                self.ring_bell = True
                self.skool.scroll_on(clock)

            self.quick_start = False

            while not self.skool.game_over:
                if self.main_loop(keyboard, clock, cheat):
                    return

            self.skool.reinitialise()

    def main_loop(self, keyboard, clock, cheat):
        if keyboard.got_quit() or keyboard.was_pressed(keys.QUIT, force_check=True):
            return True

        self.paused ^= keyboard.was_pressed(keys.PAUSE, force_check=True)
        if self.paused:
            clock.tick(10)
            keyboard.pump()
            return False

        if self.skool.tick():
            self.skool.next_lesson(self.ring_bell)
            self.ring_bell = True

        if cheat:
            self.check_cheat_keys(keyboard)

        if keyboard.was_pressed(keys.SCREENSHOT):
            img_format = 'bmp' if pygame.version.vernum < (1, 8) else 'png'
            img_fname = 'pyskool%03i.%s' % (self.screenshot, img_format)
            self.screen.take_screenshot(img_fname)
            debug.log('Took screenshot: %s' % img_fname)
            self.screenshot += 1

        scroll = self.skool.move_characters(keyboard)
        self.skool.auto_shut_doors()

        clock.tick(self.fps * self.speed)

        self.skool.draw()
        self.screen.update()
        self.skool.scroll(scroll, clock)
