# -*- coding: utf-8 -*-
# Copyright 2008, 2010 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import random
import lines
import ai

# Probability that the swot will grass on someone for hitting him
P_GRASS_FOR_HITTING = 0.140625
# Probability that the teacher will give the swot lines for telling tales
P_LINES_FOR_TALES = 0.328125
# Probability that the teacher will write on the blackboard
P_WRITE_ON_BOARD = 0.28125

# Markers used in the assembly message template
VERB_MARKER = '$VERB'
NOUN_MARKER = '$NOUN'

class Lesson:
    def __init__(self, cast, swot, room):
        self.cast = cast
        self.swot = swot
        self.room = room
        self.hitter_id = None
        self.writer_id = None
        self.teacher = None
        self.qa_generator = None
        self.answer = None
        self.asked_special = False
        self.actor = None
        self.swot_action = self.check_eric_initial
        self.teacher_action = None
        self.base_action = self.tell_class_what_to_do
        self.base_location = None
        self.base_direction = None
        self.grassed = False
        self.absence_message_ids = (lines.BE_PUNCTUAL, lines.STAY_IN_CLASS)
        self.absence_index = 0

    def join(self, teacher, qa_generator):
        self.teacher = teacher
        self.qa_generator = qa_generator
        if qa_generator:
            self.base_action = self.ask_question
        self.actor = self.swot
        self.base_location = (teacher.x, teacher.y)
        self.base_direction = teacher.direction

    def next_swot_action(self):
        while self.actor is self.swot:
            next_action = self.swot_action()
            if next_action:
                return next_action

    def check_eric_initial(self):
        self.teacher.set_home_room()
        self.swot_action = self.grass_for_hitting
        if self.is_eric_absent():
            self.teacher_action = self.fetch_eric
            return ai.Say(self.cast.get_absent_tale(self.teacher), True)

    def grass_for_hitting(self):
        self.swot_action = self.grass_for_writing
        self.teacher_action = self.give_lines_for_hitting
        if random.random() < P_GRASS_FOR_HITTING:
            self.hitter_id, tale = self.cast.get_hit_tale(self.teacher)
            return ai.Say(tale, True)
        self.switch()

    def grass_for_writing(self):
        self.grassed = True
        self.teacher_action = self.give_lines_for_writing
        writer = self.room.get_blackboard_writer()
        if writer:
            self.writer_id, tale = self.cast.get_write_tale(writer.character_id, self.teacher)
            if tale:
                return ai.Say(tale, True)
        self.switch()

    def check_eric(self):
        if self.is_eric_absent():
            self.teacher_action = self.fetch_eric
            return ai.Say(self.cast.get_absent_tale(self.teacher), True)
        self.switch(self.base_action)

    def answer_question(self):
        self.swot_action = self.check_eric
        return ai.Say(self.answer)

    def next_teacher_action(self):
        while self.actor is self.teacher:
            next_action = self.teacher_action()
            if next_action:
                return next_action

    def fetch_eric(self):
        if random.random() < P_LINES_FOR_TALES:
            self.teacher.give_lines(self.swot.character_id, lines.NO_TALES, True)
        if self.is_eric_absent():
            self.teacher_action = self.return_to_base
            self.teacher.reset_come_along_index()
            self.absence_index = 1
            return ai.FetchEric()
        lines_message_id = self.absence_message_ids[self.absence_index]
        self.absence_index = 1
        self.teacher.give_lines(self.cast.eric.character_id, lines_message_id, True)
        self.switch()

    def return_to_base(self):
        if (self.teacher.x, self.teacher.y) != self.base_location:
            return ai.GoToXY(*self.base_location)
        if self.room.has_blackboard() and not self.grassed:
            if self.teacher.direction != self.base_direction:
                # Turn teacher round before continuing
                return ai.GoTowardsXY(self.teacher.x - self.teacher.direction, self.teacher.y)
            else:
                self.switch()
                return
        if self.qa_generator:
            if self.teacher.direction != self.base_direction:
                # Turn teacher round before continuing
                return ai.GoTowardsXY(self.teacher.x - self.teacher.direction, self.teacher.y)
            else:
                self.teacher_action = self.ask_question
        else:
            self.teacher_action = self.walk_up_or_down
            if self.teacher.direction != self.base_direction:
                # Instead of turning round to face in the base direction only
                # to turn around again immediately, start walking up and down
                # now
                return ai.GoToXY(self.teacher.x + ai.UP_DOWN_DISTANCE * self.teacher.direction, self.teacher.y)

    def give_lines(self, victim_id, message_id):
        if victim_id:
            victim_present = self.room.contains(self.cast.get(victim_id))
            punish_swot = random.random() < P_LINES_FOR_TALES
            if punish_swot or not victim_present:
                victim_id, message_id = self.swot.character_id, lines.NO_TALES
            self.teacher.give_lines(victim_id, message_id, True)

    def give_lines_for_hitting(self):
        self.give_lines(self.hitter_id, lines.NO_HITTING)
        self.switch()

    def give_lines_for_writing(self):
        self.give_lines(self.writer_id, lines.NO_WRITING)
        self.teacher_action = self.wipe_board

    def wipe_board(self):
        self.absence_index = 1
        if self.room.has_blackboard():
            self.teacher_action = self.walk_to_board
            return ai.WipeBoard()
        self.teacher_action = self.base_action

    def walk_to_board(self):
        self.teacher_action = self.write_on_board
        return ai.GoToXY(self.teacher.x - ai.BB_BACKTRACK * self.teacher.direction, self.teacher.y)

    def write_on_board(self):
        self.base_location = (self.teacher.x, self.teacher.y)
        self.base_direction = self.teacher.direction
        self.teacher_action = self.base_action
        if random.random() < P_WRITE_ON_BOARD:
            return ai.WriteOnBoard(self.teacher.get_blackboard_message())

    def ask_question(self):
        self.swot_action = self.answer_question
        return ai.Say(self.get_question(), True)

    def tell_class_what_to_do(self):
        self.base_action = self.walk_up_or_down
        self.teacher_action = self.walk_up_or_down
        return ai.TellClassWhatToDo()

    def walk_up_or_down(self):
        self.switch(self.check_eric)
        return ai.WalkUpOrDown()

    def get_question(self):
        generator = self.qa_generator.prepare_qa
        if not self.asked_special:
            self.asked_special = True
            if self.qa_generator.has_special_question():
                generator = self.qa_generator.prepare_special_qa
        question, self.answer = generator()
        return question

    def switch(self, action=None):
        if self.actor is self.swot:
            self.actor = self.teacher
            self.teacher_action = action or self.teacher_action
        else:
            self.actor = self.swot
            self.swot_action = action or self.swot_action

    def finished_speaking(self):
        self.switch()

    def is_eric_absent(self):
        return not self.room.contains(self.cast.eric)

class QAGenerator:
    def __init__(self):
        self.questions = []
        self.answers = {}
        self.qa_pairs = {}
        self.special_qa_group = None
        self.remaining = {}

    def set_special_group(self, qa_group, index):
        self.special_qa_group = qa_group
        self.special_qa_pair_index = int(index)

    def set_special_question(self, text):
        self.special_question = text

    def set_special_answer(self, text):
        self.special_answer = text

    def initialise_special_answer(self):
        if self.special_qa_group:
            self.special_answer_index = random.randrange(len(self.qa_pairs[self.special_qa_group]))
            return self.qa_pairs[self.special_qa_group][self.special_answer_index][self.special_qa_pair_index]

    def has_special_question(self):
        return self.special_qa_group is not None

    def add_question(self, question_id, qa_group, text):
        self.questions.append((question_id, qa_group, text))

    def add_answer(self, question_id, text):
        self.answers[question_id] = text

    def add_qa_pair(self, qa_group, word1, word2):
        if qa_group not in self.qa_pairs:
            self.qa_pairs[qa_group] = []
            self.remaining[qa_group] = []
        self.qa_pairs[qa_group].append((word1, word2))

    def expand(self, template, word1, word2):
        return template.replace('$1', word1).replace('$2', word2)

    def prepare_special_qa(self):
        word1, word2 = self.qa_pairs[self.special_qa_group][self.special_answer_index]
        return self.special_question, self.expand(self.special_answer, word1, word2)

    def prepare_qa(self):
        question_id, qa_group, question = random.choice(self.questions)
        answer = self.answers[question_id]
        if not self.remaining[qa_group]:
            self.remaining[qa_group] = list(range(len(self.qa_pairs[qa_group])))
        random_index = self.remaining[qa_group].pop(random.randrange(len(self.remaining[qa_group])))
        word1, word2 = self.qa_pairs[qa_group][random_index]
        return self.expand(question, word1, word2), self.expand(answer, word1, word2)

class AssemblyMessageGenerator:
    def __init__(self):
        self.text = None
        self.verbs = []
        self.nouns = []

    def set_text(self, text):
        self.text = text

    def add_verb(self, verb):
        self.verbs.append(verb)

    def add_noun(self, noun):
        self.nouns.append(noun)

    def generate_message(self):
        verb = random.choice(self.verbs)
        noun = random.choice(self.nouns)
        message = self.text.replace(VERB_MARKER, verb)
        return message.replace(NOUN_MARKER, noun)
