# -*- coding: utf-8 -*-
# Copyright 2008, 2010 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

"""
Build the skool and its cast of characters.
"""

class SkoolBuilder:
    """Builds a skool and its cast from the contents of an ini file.

    :param ini_file: The name of the ini file to use.
    """
    def __init__(self, ini_file):
        self.sections = {}
        f = open(ini_file, 'r')
        section = None
        for line in f:
            if line[0] == '[':
                if section:
                    self.sections[section_name] = section
                section_name = line[1:line.index(']')].strip()
                section = []
            elif line.isspace():
                continue
            elif line.startswith(';'):
                continue
            else:
                section.append(line.strip())
        self.sections[section_name] = section
        f.close()

    def build_skool(self, skool):
        """Build a skool from the contents of the ini file.

        :type skool: :class:`~skool.Skool`
        :param skool: The skool to build.
        """
        self.skool = skool
        self.cast = skool.cast
        self.beeper = skool.beeper
        self.timetable = skool.timetable
        self.font = skool.font
        self.assembly_message_generator = skool.assembly_message_generator

        self._parse_sounds()
        self._parse_sprite_groups()
        self._parse_characters()
        self._parse_eric()
        self._parse_catapult_pellets()
        self._parse_water_drop()
        self._parse_sherry_drop()
        self._parse_conker()
        self._parse_water()
        self._parse_stinkbombs()
        self._parse_mice()
        self._parse_frogs()
        self._parse_bike()
        self._parse_timetable()
        self._parse_special_playtimes()
        self._parse_lessons()
        self._parse_random_locations()
        self._parse_skool_locations()
        self._parse_inventory()
        self._parse_command_lists()
        self._parse_rooms()
        self._parse_chairs()
        self._parse_desks()
        self._parse_desk_lid()
        self._parse_doors()
        self._parse_windows()
        self._parse_walls()
        self._parse_staircases()
        self._parse_floors()
        self._parse_routes()
        self._parse_no_go_zones()
        self._parse_sit_down_messages()
        self._parse_character_widths()
        self._parse_assembly_messages()
        self._parse_blackboards()
        self._parse_blackboard_messages()
        self._parse_questions_and_answers()
        self._parse_lines_messages()
        self._parse_lesson_messages()
        self._parse_shields()
        self._parse_safe()
        self._parse_cups()
        self._parse_plants()
        self._parse_grass()

    def _get_elements(self, details, parse_ints=True):
        """Return a list of the elements extracted from a line in a section of
        the ini file.

        :param details: The line of the ini file.
        :param parse_ints: If `True`, any numeric element is converted to an
                           integer; otherwise it is left as a string.
        """
        elements = []
        index = 0
        while index < len(details):
            if details[index].isspace():
                index += 1
            elif details[index] in '"\'':
                quote = details[index]
                end = details.index(quote, index + 1)
                elements.append(details[index + 1:end])
                index = details.find(',', end + 1)
                if index < 0:
                    index = len(details)
                index += 1
            elif details[index] == '(':
                end = details.index(')', index)
                coords = details[index + 1:end].split(',')
                elements.append((int(coords[0]), int(coords[1])))
                index = details.find(',', end + 1)
                if index < 0:
                    index = len(details)
                index += 1
            else:
                end = details.find(',', index)
                if end < 0:
                    end = len(details)
                element = details[index:end].strip()
                if element.isdigit() and parse_ints:
                    elements.append(int(element))
                else:
                    elements.append(element)
                index = end + 1
        return elements

    def _parse_section(self, name, parse_ints=True):
        """Extract the elements from every line in a section of the ini file.
        The return value is a list of lists of elements from each line. If the
        named section does not exist, an empty list is returned.

        :param name: The name of the section.
        :param parse_ints: If `True`, any numeric element is converted to an
                           integer; otherwise it is left as a string.
        """
        if name in self.sections:
            return [self._get_elements(line, parse_ints) for line in self.sections[name]]
        return []

    def _get_sections(self, prefix, parse_ints=True):
        """Extract the elements from every line in the sections of the ini file
        whose names begin with a certain prefix. Return a dictionary whose keys
        are the section name suffixes, and where each value is a list of lists
        of elements from each line.

        :param prefix: The section name prefix.
        :param parse_ints: If `True`, any numeric element is converted to an
                           integer; otherwise it is left as a string.
        """
        sections = {}
        for name in self.sections:
            if name.startswith(prefix):
                sections[name[len(prefix):].strip()] = self._parse_section(name, parse_ints)
        return sections

    def get_config(self, name):
        """Return a dictionary of keys and values from a section of the ini
        file.

        :param name: The name of the section.
        """
        config = {}
        for key, value in self._parse_section(name):
            config[key] = value
        return config

    #//////////////////////////////////////////////////////////////////////////
    # Section parsing
    #//////////////////////////////////////////////////////////////////////////
    def _parse_sounds(self):
        """Parse the 'Sounds' section of the ini file."""
        for sound_id, sound_file in self._parse_section('Sounds'):
            self.beeper.add_sound(sound_id, sound_file)

    def _parse_sprite_groups(self):
        """Parse the 'SpriteGroup' sections of the ini file."""
        for group_id, section in self._get_sections('SpriteGroup ').items():
            for sprite_id, sprite_index in section:
                self.cast.add_sprite(group_id, sprite_id, sprite_index)

    def _parse_eric(self):
        """Parse the 'Eric' section of the ini file."""
        for character_id, name, sprite_group, initial_as, location, flags in self._parse_section('Eric'):
            self.cast.add_eric(character_id, name, sprite_group, initial_as, flags.upper())
            self.cast.set_location(character_id, *location)

    def _parse_characters(self):
        """Parse the 'Characters' section of the ini file."""
        for character_id, name_and_title, sprite_group, initial_as, location, flags in self._parse_section('Characters'):
            names = name_and_title.partition('/')
            name, title = names[0], names[2]
            self.cast.add_character(character_id, name, title, sprite_group, initial_as, flags.upper())
            self.cast.set_location(character_id, *location)

    def _parse_catapult_pellets(self):
        """Parse the 'CatapultPellets' section of the ini file."""
        for character_id, pellet_id, sprite_group, command_list_id, pellet_range, hit_zone in self._parse_section('CatapultPellets'):
            self.cast.add_pellet(character_id, pellet_id, sprite_group, command_list_id, pellet_range, hit_zone)

    def _parse_water_drop(self):
        """Parse the 'WaterDrop' section of the ini file."""
        for object_id, sprite_group_id, command_list_id in self._parse_section('WaterDrop'):
            self.cast.add_water_drop(object_id, sprite_group_id, command_list_id)

    def _parse_sherry_drop(self):
        """Parse the 'SherryDrop' section of the ini file."""
        for object_id, sprite_group_id, command_list_id in self._parse_section('SherryDrop'):
            self.cast.add_sherry_drop(object_id, sprite_group_id, command_list_id)

    def _parse_conker(self):
        """Parse the 'Conker' section of the ini file."""
        for object_id, sprite_group_id, command_list_id, min_x, max_x, min_y, max_y in self._parse_section('Conker'):
            self.cast.add_conker(object_id, sprite_group_id, command_list_id, min_x, max_x, min_y, max_y)

    def _parse_water(self):
        """Parse the 'Water' section of the ini file."""
        for character_id, water_id, sprite_group, command_list_id in self._parse_section('Water'):
            self.cast.add_water(character_id, water_id, sprite_group, command_list_id)

    def _parse_stinkbombs(self):
        """Parse the 'Stinkbombs' section of the ini file."""
        for character_id, stinkbomb_id, sprite_group, command_list_id in self._parse_section('Stinkbombs'):
            self.cast.add_stinkbomb(character_id, stinkbomb_id, sprite_group, command_list_id)

    def _parse_mice(self):
        """Parse the 'Mice' section of the ini file."""
        for mouse_id, sprite_group_id, initial_as, location, command_list_id in self._parse_section('Mice'):
            self.cast.add_mouse(mouse_id, sprite_group_id, initial_as, location, command_list_id)

    def _parse_frogs(self):
        """Parse the 'Frogs' section of the ini file."""
        for frog_id, sprite_group_id, initial_as, location, command_list_id in self._parse_section('Frogs'):
            self.cast.add_frog(frog_id, sprite_group_id, initial_as, location, command_list_id)

    def _parse_bike(self):
        """Parse the 'Bike' section of the ini file."""
        for bike_id, sprite_group_id, initial_as, location, command_list_id, top_left, size, coords in self._parse_section('Bike'):
            self.skool.add_bike(bike_id, sprite_group_id, initial_as, location, command_list_id, top_left, size, coords)

    def _parse_timetable(self):
        """Parse the 'Timetable' section of the ini file."""
        for lesson_type in self._parse_section('Timetable'):
            self.timetable.add_lesson(*lesson_type)

    def _parse_special_playtimes(self):
        """Parse the 'SpecialPlaytimes' section of the ini file."""
        for lesson_type in self._parse_section('SpecialPlaytimes'):
            self.timetable.add_special_playtime(*lesson_type)

    def _parse_lessons(self):
        """Parse the 'Lesson' sections of the ini file."""
        for details, section in self._get_sections('Lesson ').items():
            index = details.index(' ')
            lesson_id = details[:index]
            lesson_details = [e.strip() for e in details[index + 1:].split(',')]
            room_id = lesson_details[-1]
            teacher_id = lesson_details[0] if len(lesson_details) > 1 else ''
            hide_teacher = teacher_id.startswith('*')
            if hide_teacher:
                teacher_id = teacher_id[1:]
            self.timetable.add_lesson_details(lesson_id, hide_teacher, teacher_id, room_id)
            for character_id, command_list_id in section:
                self.cast.add_command_list(character_id, lesson_id, command_list_id)

    def _parse_random_locations(self):
        """Parse the 'RandomLocations' section of the ini file."""
        for elements in self._parse_section('RandomLocations'):
            character_id = elements[0]
            locations = elements[1:]
            self.cast.set_random_locations(character_id, locations)

    def _parse_skool_locations(self):
        """Parse the 'SkoolLocations' section of the ini file."""
        for location_id, x, y in self._parse_section('SkoolLocations'):
            self.skool.add_location(location_id, (x, y))

    def _parse_inventory(self):
        """Parse the 'Inventory' section of the ini file."""
        for item_id, top_left, size in self._parse_section('Inventory'):
            self.skool.add_inventory_item(item_id, top_left, size)

    def _parse_command_lists(self):
        """Parse the 'CommandList' sections of the ini file."""
        for command_list_id, section in self._get_sections('CommandList ').items():
            for elements in section:
                command_name = elements[0]
                params = []
                for e in elements[1:]:
                    try:
                        params.append(int(e))
                    except ValueError:
                        params.append(e)
                self.cast.add_command(command_list_id, command_name, *params)

    def _parse_rooms(self):
        """Parse the 'Rooms' section of the ini file."""
        for room_id, name, y, min_x, max_x, get_along in self._parse_section('Rooms'):
            self.skool.add_room(room_id, name, y, min_x, max_x, get_along.upper() == 'Y')

    def _parse_chairs(self):
        """Parse the 'Chairs' section of the ini file."""
        for elements in self._parse_section('Chairs'):
            room_id = elements[0]
            for x in elements[1:]:
                self.skool.add_chair(room_id, x)

    def _parse_desks(self):
        """Parse the 'Desks' section of the ini file."""
        for elements in self._parse_section('Desks'):
            room_id = elements[0]
            for x in elements[1:]:
                self.skool.add_desk(room_id, x)
        self.skool.fill_desks()

    def _parse_desk_lid(self):
        """Parse the 'DeskLid' section of the ini file."""
        for desk_lid_id, sprite_group_id, command_list_id in self._parse_section('DeskLid'):
            self.cast.add_desk_lid(desk_lid_id, sprite_group_id, command_list_id)

    def _parse_doors(self):
        """Parse the 'Doors' section of the ini file."""
        for door_id, x, bottom_y, top_y, initially_shut, auto_shuts, shut_top_left, size, coords in self._parse_section('Doors'):
            initially_shut = initially_shut.upper() == 'Y'
            auto_shuts = auto_shuts.upper() == 'Y'
            self.skool.add_door(door_id, x, bottom_y, top_y, initially_shut, auto_shuts, shut_top_left, size, coords)

    def _parse_windows(self):
        """Parse the 'Windows' section of the ini file."""
        for window_id, x, bottom_y, top_y, initially_shut, opener_coords, shut_top_left, size, coords in self._parse_section('Windows'):
            initially_shut = initially_shut.upper() == 'Y'
            self.skool.add_window(window_id, x, bottom_y, top_y, initially_shut, opener_coords, shut_top_left, size, coords)

    def _parse_walls(self):
        """Parse the 'Walls' section of the ini file."""
        for wall_id, x, bottom_y, top_y in self._parse_section('Walls'):
            self.skool.add_wall(wall_id, x, bottom_y, top_y)

    def _parse_staircases(self):
        """Parse the 'Staircases' section of the ini file."""
        for elements in self._parse_section('Staircases'):
            staircase_ids = elements[0].partition(':')
            bottom = elements[1]
            top = elements[2]
            force = len(elements) == 4
            self.skool.add_staircase(staircase_ids[0], bottom, top, force, staircase_ids[2])

    def _parse_floors(self):
        """Parse the 'Floors' section of the ini file."""
        for floor_id, min_x, max_x, y in self._parse_section('Floors'):
            self.skool.add_floor(floor_id, min_x, max_x, y)

    def _parse_routes(self):
        """Parse the 'Routes' section of the ini file."""
        for elements in self._parse_section('Routes'):
            home_floor_id = elements[0]
            staircase_id = elements[-1]
            dest_floor_ids = elements[1:-1]
            self.skool.add_routes(home_floor_id, dest_floor_ids, staircase_id)

    def _parse_no_go_zones(self):
        """Parse the 'NoGoZones' section of the ini file."""
        for zone_id, min_x, max_x, bottom_y, top_y in self._parse_section('NoGoZones'):
            self.skool.add_no_go_zone(zone_id, min_x, max_x, bottom_y, top_y)

    def _parse_sit_down_messages(self):
        """Parse the 'SitDownMessages' section of the ini file."""
        for character_id, message in self._parse_section('SitDownMessages'):
            self.cast.set_sit_down_message(character_id, message)

    def _parse_character_widths(self):
        """Parse the 'CharacterWidths' section of the ini file."""
        for char, offset, width in self._parse_section('CharacterWidths'):
            self.font.add_character_offset(char, offset, width)

    def _parse_assembly_messages(self):
        """Parse the 'AssemblyMessages' section of the ini file."""
        for elements in self._parse_section('AssemblyMessages'):
            if elements[0] == 'MESSAGE':
                self.assembly_message_generator.text = elements[1]
            elif elements[0] == 'VERB':
                self.assembly_message_generator.add_verb(elements[1])
            elif elements[0] == 'NOUN':
                self.assembly_message_generator.add_noun(elements[1])

    def _parse_blackboards(self):
        """Parse the 'Blackboards' section of the ini file."""
        for room_id, x, y in self._parse_section('Blackboards'):
            self.skool.add_blackboard(room_id, x, y)

    def _parse_blackboard_messages(self):
        """Parse the 'BlackboardMessages' section of the ini file."""
        for character_id, message in self._parse_section('BlackboardMessages'):
            self.cast.add_blackboard_message(character_id, message)

    def _parse_questions_and_answers(self):
        """Parse the 'QuestionsAndAnswers' sections of the ini file."""
        for teacher_id, section in self._get_sections('QuestionsAndAnswers ', False).items():
            qa_generator = self.cast.get(teacher_id).get_qa_generator()
            for elements in section:
                entry_type = elements[0]
                if entry_type == 'SpecialGroup':
                    qa_generator.set_special_group(elements[1], int(elements[2]))
                elif entry_type == 'SpecialQuestion':
                    qa_generator.special_question = elements[1]
                elif entry_type == 'SpecialAnswer':
                    qa_generator.special_answer = elements[1]
                elif entry_type == 'Question':
                    question_id = elements[1]
                    qa_group = elements[2]
                    text = elements[3]
                    qa_generator.add_question(question_id, qa_group, text)
                elif entry_type == 'Answer':
                    question_id = elements[1]
                    text = elements[2]
                    qa_generator.add_answer(question_id, text)
                elif entry_type == 'QAPair':
                    qa_group = elements[1]
                    qa_generator.add_qa_pair(qa_group, elements[2], elements[3])

    def _parse_lines_messages(self):
        """Parse the 'LinesMessages' section of the ini file."""
        for character_id, message_id, message in self._parse_section('LinesMessages'):
            self.cast.add_lines_message(character_id, message_id, message)

    def _parse_lesson_messages(self):
        """Parse the 'LessonMessages' section of the ini file."""
        for elements in self._parse_section('LessonMessages'):
            character_id = elements[0]
            message = elements[1]
            condition = '' if len(elements) < 3 else elements[2]
            self.cast.add_lesson_message(character_id, message, condition)

    def _parse_shields(self):
        """Parse the 'Shields' section of the ini file."""
        for score, top_left, size, coords in self._parse_section('Shields'):
            self.skool.add_shield(score, top_left, size, coords)

    def _parse_safe(self):
        """Parse the 'Safe' section of the ini file."""
        for top_left, size, coords in self._parse_section('Safe'):
            self.skool.add_safe(top_left, size, coords)

    def _parse_cups(self):
        """Parse the 'Cups' section of the ini file."""
        for cup_id, empty_top_left, size, coords in self._parse_section('Cups'):
            self.skool.add_cup(cup_id, empty_top_left, size, coords)

    def _parse_plants(self):
        """Parse the 'Plants' section of the ini file."""
        for plant_id, sprite_group_id, x, y, command_list_id in self._parse_section('Plants'):
            self.skool.add_plant(plant_id, sprite_group_id, x, y, command_list_id)

    def _parse_grass(self):
        """Parse the 'Grass' section of the ini file."""
        for elements in self._parse_section('Grass'):
            param = elements[0]
            values = elements[1:]
            if param == 'Hitters':
                self.cast.hitters = values
            elif param == 'Writers':
                self.cast.writers = values
            elif param == 'HitTale':
                self.cast.hit_tale = values[0]
            elif param == 'WriteTale':
                self.cast.write_tale = values[0]
            elif param == 'AbsentTale':
                self.cast.absent_tale = values[0]
