# -*- coding: utf-8 -*-
# Copyright 2008, 2010 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

"""
Defines the :class:`Game` class.
"""

import os
import pygame
import random
import time
from cast import Cast
import character
from skool import Skool
from graphics import Screen, Gallery, SPRITES
from sound import Beeper
from input import Keyboard
from skoolbuilder import SkoolBuilder
import keys
import items
import debug

class Game:
    """Builds the skool and the cast, and executes the main loop of the
    game.

    :param pyskool_dir: The absolute path of the top-level Pyskool directory.
    :type scale: number
    :param scale: The desired scale at which to run the game.
    :param ini_file: The name of the ini file to use.
    :param quick_start: `True` to force the game to start quickly, `None` to
                        use the `QuickStart` ini file parameter.
    :param cheat: `True` to force cheat keys to be enabled, `None` to use the
                   `Cheat` ini file parameter.
    :type version: string
    :param version: The version number of Pyskool.
    :param icon_fname: The name of the icon file to use.
    """
    def __init__(self, pyskool_dir, scale, ini_file, quick_start, cheat, version, icon_fname):
        pygame.init()
        builder = SkoolBuilder(ini_file)
        config = builder.get_config('Game')
        screen_config = builder.get_config('Screen')

        # Create the gallery
        images_dir = os.path.join(pyskool_dir, config['ImagesDir'])
        image_set = config['ImageSet']
        scale = scale or screen_config['Scale']
        gallery = Gallery(images_dir, image_set, scale, builder.get_config('Images'))

        title = 'Pyskool %s: %s' % (version, config['Name'])
        self.screen = Screen(screen_config, scale, gallery, title, icon_fname)
        self.beeper = Beeper(os.path.join(pyskool_dir, config['SoundsDir']))
        self.cast = Cast(gallery.get_image(SPRITES))
        self.skool = Skool(self.screen, self.beeper, self.cast, gallery, config)
        builder.build_skool(self.skool)
        self.keyboard = Keyboard()
        self.skool.initialise_cast(self.keyboard)
        self.screen.initialise_column(config['InitialColumn'], self.skool.get_width(), self.cast.eric.x)

        self.quick_start = quick_start or config['QuickStart']
        self.cheat = cheat or config['Cheat']
        self.speed = 1
        self.fps = config['GameFps']
        self.screenshot = 0

    def _no_lines(self, *args):
        """Method used to replace :meth:`character.Character.give_lines` when
        the :data:`~keys.NO_LINES` cheat key is pressed. The method does
        nothing.
        """
        return

    def _check_cheat_keys(self):
        """Check whether any cheat keys were pressed, and take appropriate
        action. This method is called from the main loop.
        """
        if self.keyboard.was_pressed(keys.NEXT_LESSON):
            self.skool.next_lesson(False)
        if self.keyboard.is_pressed(keys.SLOW):
            self.speed = 0.5
        elif self.keyboard.is_pressed(keys.FAST):
            self.speed = 2
        else:
            self.speed = 1
        if self.skool.shields:
            if self.keyboard.was_pressed(keys.FLASH_MOST):
                for shield in self.skool.shields[1:]:
                    shield.flash()
                self.skool.shields[0].unflash()
                self.skool.safe.unflash()
                self.skool.shield_mode = 1
                debug.log('Flashed all but one shield; hit it and then open the safe')
            if self.keyboard.was_pressed(keys.UNFLASH_MOST):
                for shield in self.skool.shields[1:]:
                    shield.unflash()
                self.skool.shields[0].flash()
                self.skool.safe.flash()
                self.skool.shield_mode = 3
                debug.log('Unflashed all but one shield; hit it to go up a year')
        if self.keyboard.was_pressed(keys.NO_LINES):
            character.Character.give_lines = self._no_lines
            debug.log('Disabled lines-giving')
        if self.keyboard.was_pressed(keys.ADD_LINES):
            lines = 10 * random.randrange(1, 9)
            self.skool.add_lines(lines)
            debug.log('Added %i lines' % (lines * 10))
        if self.keyboard.was_pressed(keys.ZERO_LINES):
            self.skool.add_lines(-self.skool.scoreboard.lines)
            debug.log('Set lines total to zero')
        if self.keyboard.was_pressed(keys.REVEAL):
            for room in self.skool.rooms.values():
                for desk in room.desks:
                    if desk.contents:
                        debug.log('%s x=%i: %s' % (room.name, desk.x, desk.contents))
            for c in self.cast.character_list:
                if c.special_answer:
                    debug.log('%s: %s' % (c.name, c.special_answer))
            if self.skool.safe_combination:
                debug.log('Safe: %s' % self.skool.safe_combination)
            if self.skool.bike_combination:
                debug.log('Bike: %s' % self.skool.bike_combination)
            if self.skool.storeroom_combination:
                debug.log('Storeroom: %s' % self.skool.storeroom_combination)
        if self.skool.inventory_item_ids:
            eric = self.cast.eric
            inventory = eric.inventory
            if self.keyboard.was_pressed(keys.SWITCH_PISTOL):
                if items.WATER_PISTOL in inventory:
                    inventory.remove(items.WATER_PISTOL)
                    inventory.add(items.SHERRY_PISTOL)
                elif items.SHERRY_PISTOL in inventory:
                    inventory.remove(items.SHERRY_PISTOL)
                    inventory.add(items.WATER_PISTOL)
                eric.print_inventory()
            if self.keyboard.was_pressed(keys.GIVE_ALL):
                inventory.update((items.SAFE_KEY, items.STOREROOM_KEY, items.FROG, items.WATER_PISTOL, items.STINKBOMBS3))
                if self.cast.frogs:
                    self.cast.frogs[0].hide()
                eric.mice = 8
                eric.print_inventory()
                eric.print_mouse_inventory()
                self.skool.unchain_bike()
        if self.skool.doors:
            if self.keyboard.was_pressed(keys.OPEN_DOORS):
                for door_id in self.skool.doors:
                    self.skool.move_door(door_id, False)
                for window_id in self.skool.windows:
                    self.skool.move_door(window_id, False)
                debug.log('Opened all doors and windows')
            if self.keyboard.was_pressed(keys.CLOSE_DOORS):
                for door_id in self.skool.doors:
                    self.skool.move_door(door_id, True)
                for window_id in self.skool.windows:
                    self.skool.move_door(window_id, True)
                debug.log('Closed all doors and windows')

    def play(self):
        """Start the game and enter the main loop."""
        clock = pygame.time.Clock()

        self.ring_bell = False

        self.paused = False
        while True:
            if not self.quick_start:
                self.ring_bell = True
                self.skool.scroll_on(clock)

            self.quick_start = False

            while not self.skool.game_over:
                if self._main_loop(clock):
                    return

            self.skool.reinitialise()

    def _main_loop(self, clock):
        """The main loop of the game. The following things are done in the main
        loop:

        * check the keyboard and act on keypresses
        * advance the skool clock
        * move the characters
        * shut any auto-shutting doors that need shutting
        * update the screen
        * scroll the screen if necessary

        :type clock: `pygame.time.Clock`
        :param clock: The clock to use for timing.
        :return: `True` if the quit key was pressed or the window close button
                 was clicked; `False` otherwise.
        """
        if self.keyboard.got_quit() or self.keyboard.was_pressed(keys.QUIT, force_check=True):
            return True

        if self.keyboard.was_pressed(keys.SCREENSHOT):
            img_format = 'bmp' if pygame.version.vernum < (1, 8) else 'png'
            timestamp = time.strftime('%Y%m%d-%H%M%S')
            img_fname = 'pyskool-%s-%03i.%s' % (timestamp, self.screenshot, img_format)
            self.screen.take_screenshot(img_fname)
            self.screenshot += 1
            debug.log('Took screenshot: %s' % img_fname)

        self.paused ^= self.keyboard.was_pressed(keys.PAUSE, force_check=True)
        if self.paused:
            self.beeper.pause()
            clock.tick(10)
            self.keyboard.pump()
            return False

        self.beeper.unpause()
        if self.beeper.is_busy():
            self.keyboard.pump()
            return False
        self.skool.resume()

        if self.skool.tick():
            self.skool.next_lesson(self.ring_bell)
            self.ring_bell = True

        if self.cheat:
            self._check_cheat_keys()

        scroll = self.skool.move_characters()
        self.skool.auto_shut_doors()

        clock.tick(self.fps * self.speed)

        self.skool.draw()
        self.screen.update()
        self.skool.scroll(scroll, clock)

        return False
