# -*- coding: utf-8 -*-
# Copyright 2008, 2010 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import random
from skool import Location, Signals
from animatorystates import *

class CommandListTemplate:
    def __init__(self, tap_id):
        self.tap_id = tap_id
        self.commands = []

    def add_command(self, command_class, *params):
        self.commands.append((command_class, params))

    def get_commands(self, start):
        return [cmd_class(*params) for cmd_class, params in self.commands][start:]

class CommandList:
    def __init__(self):
        self.stack = []
        self.index = None
        self.restart_index = None
        self.template = None
        self.controlling_command = None
        self.controlling_command_timer = None

    def command(self, character):
        # If the controlling command is not already in control of the character,
        # give it a chance
        if self.controlling_command and self.controlling_command not in self.stack:
            self.stack.append(self.controlling_command)
        while True:
            if self.stack:
                command = self.stack[-1]
                subcommand = command.command(character)
                if subcommand is None:
                    break
                elif subcommand is not command:
                    self.stack.append(subcommand)
                else:
                    self.stack.pop()
            else:
                # Remove any controlling command before moving to the next
                # command
                if self.controlling_command:
                    self.controlling_command_timer -= 1
                    if self.controlling_command_timer < 0:
                        self.controlling_command = None
                        character.trigger_speed_change()
                self.stack.append(self.commands[self.index])
                self.index += 1

    def restart(self, index=None):
        self.index = index or 0
        self.commands = self.template.get_commands(self.restart_index)

    def add_command(self, command):
        self.stack.append(command)

    def set_template(self, template):
        self.template = template
        while self.stack and self.stack[0].is_interruptible():
            self.stack.pop(0)
        self.restart_index = 0
        self.restart()

    def set_controlling_command(self, command):
        self.controlling_command = command
        self.controlling_command_timer = 1

    def set_restart_point(self):
        """Discard the current and all previous commands."""
        self.restart_index = self.index
        self.restart()

    def jump(self, offset):
        """Jump forwards (or backwards) in the list of commands."""
        self.restart(self.index + offset)

    def is_GoToing(self):
        """Returns whether the character is under the control of a GoTo*
        command."""
        return self.stack and self.stack[0].__class__.__name__.startswith('GoTo')

    def get_GoTo_destination(self):
        """Returns the destination of the character, or None if he is not under
        the control of the GoTo command."""
        if self.is_GoToing():
            return self.stack[0].get_destination()

    def set_GoTo_destination(self, destination):
        """Sets the destination of the character if he is under the control of
        the GoTo command."""
        if destination and self.is_GoToing():
            self.stack[0].set_destination(destination)

class Command:
    def is_interruptible(self):
        return True

class ComplexCommand(Command):
    def __init__(self):
        self.index = 0
        self.commands = self.get_commands()

    def command(self, character):
        val = self.commands[self.index](character)
        if val is not False:
            self.index = (self.index + 1) % len(self.commands)
        return val or None

    def restart(self):
        self.index = -1

    def done(self, character):
        return self

class SitForAssembly(ComplexCommand):
    def get_commands(self):
        return (self.find_spot_to_sit, self.sit_down, self.get_up, self.done)

    def find_spot_to_sit(self, character):
        if character.got_signal(Signals.ASSEMBLY_FINISHED):
            return self
        return GoToXY(character.x - random.randrange(1, 5), character.y)

    def sit_down(self, character):
        if character.got_signal(Signals.ASSEMBLY_FINISHED):
            return self
        if character.direction < 0:
            character.turn()
            return False
        character.sit()

    def get_up(self, character):
        if character.got_signal(Signals.ASSEMBLY_FINISHED):
            character.get_up()
        else:
            return False

class MoveDoor(ComplexCommand):
    def __init__(self, door_id, shut):
        ComplexCommand.__init__(self)
        self.door_id = door_id
        self.shut = shut

    def get_commands(self):
        return (self.raise_arm, self.move_door, self.lower_arm, self.done)

    def raise_arm(self, character):
        if character.check_door_status(self.door_id, self.shut):
            """Move on now if the door is already in the desired state."""
            return self
        character.raise_arm()

    def move_door(self, character):
        character.move_door(self.door_id, self.shut)

    def lower_arm(self, character):
        character.lower_arm()

class OpenDoor(MoveDoor):
    def __init__(self, door_id):
        MoveDoor.__init__(self, door_id, False)

class ShutDoor(MoveDoor):
    def __init__(self, door_id):
        MoveDoor.__init__(self, door_id, True)

class WipeBoard(ComplexCommand):
    def __init__(self):
        ComplexCommand.__init__(self)
        self.column = 7
        self.interrupted = False

    def get_commands(self):
        return (self.walk, self.wipe, self.lower_arm, self.walk_back, self.done)

    def walk(self, character):
        character.set_wiping_board(True)
        return GoToXY(character.x - 1, character.y)

    def wipe(self, character):
        character.raise_arm()
        character.wipe_board(self.column)

    def lower_arm(self, character):
        self.column -= 1
        character.lower_arm()
        if self.column >= 0:
            # Repeat until the board is clean
            self.restart()

    def walk_back(self, character):
        if self.interrupted:
            # No need to walk to the middle of the board if we were interrupted
            return self
        character.set_wiping_board(False)
        return GoToXY(character.x + 3, character.y)

    def is_interruptible(self):
        self.interrupted = True
        return False

class GoTo(Command):
    def __init__(self, location_id, destination=None, go_one_step=False):
        self.location_id = location_id
        self.destination = destination
        self.go_one_step = go_one_step
        self.done = False

    def command(self, character):
        if self.destination is None:
            self.destination = character.resolve_location_id(self.location_id)
        if (character.x, character.y) == (self.destination.x, self.destination.y) or self.done:
            return self
        if character.on_stairs():
            character.walk(True)
            return
        if character.is_sitting():
            character.get_up()
            return
        next_staircase = character.get_next_staircase(self.destination)
        if next_staircase:
            if character.y == next_staircase.bottom.y:
                next_x = next_staircase.bottom.x
            else:
                next_x = next_staircase.top.x
        else:
            next_x = self.destination.x
        if character.x < next_x:
            self.done = self.go_one_step
            return character.right()
        elif character.x > next_x:
            self.done = self.go_one_step
            return character.left()
        else:
            if next_staircase.contains(character):
                character.walk(True)
            else:
                character.turn()
        self.done = self.go_one_step

    def get_destination(self):
        return self.destination

    def set_destination(self, destination):
        self.destination = destination

class GoToRandomLocation(GoTo):
    def __init__(self):
        GoTo.__init__(self, None)

    def command(self, character):
        if self.destination is None:
            self.destination = Location(character.get_random_destination())
        return GoTo.command(self, character)

class GoToXY(GoTo):
    def __init__(self, x, y):
        GoTo.__init__(self, None, Location((x, y)))

class GoTowardsXY(GoTo):
    def __init__(self, x, y):
        GoTo.__init__(self, None, Location((x, y)), True)

class Restart(Command):
    def command(self, character):
        character.restart_table()
        return self

class FindSeat(Command):
    def __init__(self):
        self.move_along = True

    def command(self, character):
        next_chair = character.get_next_chair(self.move_along)
        self.move_along = False
        if next_chair == character.chair():
            character.sit()
            return self
        if character.x == next_chair.x and character.direction > 0:
            character.turn()
            return
        return GoToXY(next_chair.x, next_chair.y)

class MoveAboutUntil(Command):
    def __init__(self, signal):
        self.signal = signal

    def command(self, character):
        if character.got_signal(self.signal):
            return self
        return WalkAround(1)

class WalkAround(Command):
    def __init__(self, walkabouts):
        self.count = walkabouts
        self.origin = None

    def command(self, character):
        if self.count == 0:
            return self
        # Cannot move about an x-coordinate on a staircase, so get off the
        # stairs first
        if character.on_stairs():
            character.walk(True)
            return
        if self.origin is None:
            self.origin = character.x
        if character.x == self.origin:
            return GoToXY(character.x - random.randrange(0, 8), character.y)
        self.count -= 1
        return GoToXY(self.origin, character.y)

class SitStill(Command):
    def command(self, character):
        return

class GrassAndAnswerQuestions(Command):
    def __init__(self):
        self.ready = False

    def command(self, character):
        if not character.is_sitting_on_chair():
            return FindSeat()
        character.keep_seat()
        if not self.ready:
            self.ready = True
            character.signal(Signals.SWOT_READY)
            character.start_lesson()
        return character.next_swot_action()

class StartAssemblyIfReady(Command):
    def command(self, character):
        if character.got_signal(Signals.ASSEMBLY_FINISHED):
            return self
        elif character.got_signal(Signals.TIME_FOR_ASSEMBLY):
            return
        character.restart_table()
        return self

class AwaitAssemblyTime(Command):
    def command(self, character):
        if character.is_time_to_start_lesson():
            character.signal(Signals.TIME_FOR_ASSEMBLY)
        else:
            character.restart_table()
        return self

class ConductAssembly(Command):
    def __init__(self):
        self.spoken = False

    def command(self, character):
        if self.spoken:
            character.signal(Signals.ASSEMBLY_FINISHED)
            return self
        self.spoken = True
        return Say(character.get_assembly_message())

class StartLessonIfReady(Command):
    def __init__(self, signal):
        self.signal = signal
        self.ready = False

    def command(self, character):
        if not self.ready:
            if character.is_time_to_start_lesson():
                self.ready = True
                character.signal(self.signal)
                character.reveal_secret(False)
                return TellKidsToSitDown()
            else:
                character.restart_table()
        return self

class StartDinnerIfReady(Command):
    def command(self, character):
        if character.is_time_to_start_lesson():
            character.set_home_room()
        else:
            character.restart_table()
        return self

class TellKidsToSitDown(Command):
    def __init__(self):
        self.spoken = False

    def command(self, character):
        if self.spoken:
            return self
        self.spoken = True
        return Say(character.sit_down_message)

class ConductClass(Command):
    def __init__(self):
        self.wiped_board = False
        self.wrote_on_board = False
        self.told_class = False

    def command(self, character):
        if character.is_teaching_eric() and not character.got_signal(Signals.SWOT_READY):
            return
        if character.is_teaching_eric():
            return ConductClassWithEric()
        if character.room.has_blackboard():
            if not self.wiped_board:
                self.wiped_board = True
                return WipeBoard()
            if not self.wrote_on_board:
                self.wrote_on_board = True
                if random.randrange(256) >= 160:
                    return WriteOnBoard(character.get_blackboard_message())
        if not self.told_class:
            self.told_class = True
            return TellClassWhatToDo()
        return WalkUpOrDown()

class TellClassWhatToDo(Command):
    def __init__(self):
        self.done = False

    def command(self, character):
        if self.done:
            return self
        self.done = True
        return Say(character.get_lesson_message())

class WalkUpOrDown(Command):
    def __init__(self):
        self.done = False

    def command(self, character):
        if self.done:
            return self
        self.done = True
        return GoToXY(character.x - 3 * character.direction, character.y)

class WriteOnBoard(Command):
    def __init__(self, message):
        self.message = message
        self.index = 1
        self.arm_down = True

    def command(self, character):
        if character.room.blackboard:
            if character.write_on_board(self.message, self.index):
                character.lower_arm()
                return self
            if self.arm_down:
                character.raise_arm()
            else:
                character.lower_arm()
            self.arm_down = not self.arm_down
        self.index += 1

class ConductClassWithEric(Command):
    def __init__(self):
        self.joined = False

    def command(self, character):
        if not self.joined:
            self.joined = True
            character.join_lesson(random.randrange(256) < 232)
        return character.next_teacher_action()

class Say(Command):
    def __init__(self, words, notify=False):
        self.words = words
        self.notify = notify
        self.shift = -4

    def command(self, character):
        if character.say(self.words, self.shift):
            character.remove_bubble()
            if self.notify:
                character.finished_speaking()
            return self
        self.shift += 1

class Signal(Command):
    def __init__(self, signal):
        self.signal = signal

    def command(self, character):
        character.signal(self.signal)
        return self

class Unsignal(Command):
    def __init__(self, signal):
        self.signal = signal

    def command(self, character):
        character.unsignal(self.signal)
        return self

class SetControllingCommand(Command):
    def __init__(self, command_name, *params):
        self.command_name = command_name
        self.params = params

    def command(self, character):
        command_class = eval(self.command_name)
        character.set_controlling_command(command_class(*self.params))
        return self

class HitOrFireNowAndThen(Command):
    def ready(self, character):
        if not character.is_standing():
            return False
        if character.on_stairs():
            return False
        if character.get_nearby_adults():
            return False
        return random.randrange(100) > 50

    def command(self, character):
        if not self.ready(character):
            return self
        return self.get_command()

class FireNowAndThen(HitOrFireNowAndThen):
    def ready(self, character):
        if not HitOrFireNowAndThen.ready(self, character):
            return False
        if character.x % 4 != 0:
            return False
        return not character.pellet.is_airborne()

    def get_command(self):
        return FireCatapult()

class FireCatapult(ComplexCommand):
    def get_commands(self):
        return (self.aim, self.fire, self.lower, self.done)

    def aim(self, character):
        if character.is_raising_catapult():
            character.fire_catapult()
        else:
            character.raise_catapult()
            return False

    def fire(self, character):
        character.fire()

    def lower(self, character):
        if character.is_lowering_catapult():
            character.complete_action()
        else:
            character.lower_catapult()
            return False

    def is_interruptible(self):
        return False

class MovePellet(Command):
    def __init__(self):
        self.count = 0
        self.x_inc = None
        self.y_inc = 0

    def command(self, pellet):
        if not pellet.is_airborne():
            return
        if self.count <= 0:
            self.count = pellet.pellet_range
            self.x_inc = pellet.direction
            self.y_inc = 0
        pellet.run(True)
        self.count -= 1
        if self.count == 0 or pellet.barrier or pellet.check_shields():
            return self.end_flight(pellet)
        if self.y_inc == 0 and self.count <= pellet.hit_zone:
            victim = pellet.get_victim()
            if victim:
                if victim.is_deckable():
                    victim.deck()
                    return self.end_flight(pellet)
                if victim.is_adult() and victim.is_knocked_over():
                    self.x_inc = 0
                    self.y_inc = -1
                    self.count = 5
        pellet.x += self.x_inc
        pellet.y += self.y_inc

    def end_flight(self, pellet):
        self.count = 0
        pellet.hide()

    def is_interruptible(self):
        return False

class HitNowAndThen(HitOrFireNowAndThen):
    def ready(self, character):
        return HitOrFireNowAndThen.ready(self, character) and self.in_front_of_victim(character)

    def in_front_of_victim(self, character):
        return len(character.get_facing_characters(3)) > 0

    def get_command(self):
        return Hit()

class Hit(ComplexCommand):
    def get_commands(self):
        return (self.aim, self.hit, self.lower, self.done)

    def aim(self, character):
        if character.is_raising_fist():
            character.punch()
        else:
            character.raise_fist()
            return False

    def hit(self, character):
        facing_characters = character.get_facing_characters(2)
        if facing_characters:
            facing_characters[0].deck()

    def lower(self, character):
        if character.is_lowering_fist():
            character.complete_action()
        else:
            character.lower_fist()
            return False

    def is_interruptible(self):
        return False

class Floored(Command):
    def __init__(self, count):
        self.count = count

    def command(self, character):
        if character.is_standing():
            return self
        self.count -= 1
        if self.count < 0:
            character.stand_up()

    def is_interruptible(self):
        return False

class KnockedOut(Floored):
    def __init__(self):
        Floored.__init__(self, 10)

class Dethroned(Floored):
    def __init__(self):
        Floored.__init__(self, 8)

class KnockedOver(Floored):
    def __init__(self):
        Floored.__init__(self, 15)

    def command(self, character):
        if Floored.command(self, character) is self:
            return self
        if self.count == 14:
            character.reveal_secret(True)
        if self.count == 7:
            character.reprimand()

class WriteOnBoardUnless(Command):
    def __init__(self, signal):
        self.signal = signal
        self.wrote_on_board = False

    def command(self, character):
        if character.got_signal(self.signal) or character.room.blackboard_dirty():
            return self
        if self.wrote_on_board:
            return self
        self.wrote_on_board = True
        return WriteOnBoard(character.get_blackboard_message())

class SetRestartPoint(Command):
    def command(self, character):
        character.set_restart_point()
        return self

class JumpIfShut(Command):
    def __init__(self, door_id, offset):
        self.door_id = door_id
        self.offset = offset

    def command(self, character):
        character.jump_if_shut(self.door_id, self.offset)
        return self

class JumpIfOpen(Command):
    def __init__(self, door_id, offset):
        self.door_id = door_id
        self.offset = offset

    def command(self, character):
        character.jump_if_open(self.door_id, self.offset)
        return self

class WalkFast(Command):
    def command(self, character):
        character.run(True)
        return self

class StalkAndHit(HitNowAndThen):
    def __init__(self, character_id):
        self.character_id = character_id

    def command(self, character):
        character.stalk(self.character_id)
        return HitNowAndThen.command(self, character)

class WaitAtDoor(Command):
    def __init__(self, door_id):
        self.door_id = door_id

    def command(self, character):
        if character.wait_at_door(self.door_id):
            return self

class Jump(ComplexCommand):
    def get_commands(self):
        return (self.up, self.down, self.done)

    def up(self, character):
        character.jump()

    def down(self, character):
        character.descend()

class Write:
    def __init__(self):
        self.finished = False

    def command(self, character):
        if self.finished:
            character.lower_arm()
            return self
        self.finished = character.write()

class FindEric(Command):
    def command(self, character):
        character.stop_clock()
        if character.is_beside_eric():
            if character.freeze_eric():
                return self
            return
        character.run(True)
        return GoTowardsXY(*character.get_location_of_eric())

class Freeze:
    def command(self, eric):
        eric.check_understanding()

class TellEric(Command):
    def __init__(self, message):
        self.told_eric = False
        self.message = message

    def command(self, character):
        if self.told_eric:
            character.unfreeze_eric()
            return self
        self.told_eric = True
        return Say(character.expand_names(self.message))

class TellEricAndWait(Command):
    def __init__(self, message):
        self.told_eric = 0
        self.delay = 20
        self.message = message

    def command(self, character):
        if self.told_eric == 0:
            self.told_eric = 1
            return Say(character.expand_names(self.message))
        elif self.told_eric == 1:
            if character.eric_understood():
                character.unfreeze_eric()
                return self
            self.told_eric = 2
            return Say(character.get_understand_message())
        elif self.told_eric == 2:
            if character.eric_understood():
                character.unfreeze_eric()
                return self
            self.delay -= 1
            if self.delay < 0:
                self.delay = 20
                self.told_eric = 0

class SetClock(Command):
    def __init__(self, ticks):
        self.ticks = ticks

    def command(self, character):
        character.start_clock(self.ticks)
        return self

class CheckIfTouchingEric(Command):
    def __init__(self, eric_knows_signal, eric_has_mumps_signal):
        self.eric_knows_signal = eric_knows_signal
        self.eric_has_mumps_signal = eric_has_mumps_signal

    def command(self, character):
        if character.got_signal(self.eric_knows_signal) and character.is_touching_eric():
            character.signal(self.eric_has_mumps_signal)
        return self

class EndGame(Command):
    def command(self, character):
        character.end_game()

class AddLines(Command):
    def __init__(self, lines):
        self.lines = lines

    def command(self, character):
        character.add_lines(self.lines)
        return self

    def is_interruptible(self):
        return False

class FetchEric(Command):
    def command(self, character):
        if not character.is_eric_absent() or character.is_eric_expelled():
            character.run(False)
            return self
        if not (character.is_beside_eric() and character.is_facing_eric()):
            character.run(True)
            return GoTowardsXY(*character.get_location_of_eric())

class FindEricIfMissing(Command):
    def command(self, character):
        if character.is_eric_absent():
            return FetchEric()
        return self

class TripPeopleUp(Command):
    def command(self, character):
        character.run(True)
        character.trip_people_up()
        return self

class Follow(Command):
    def __init__(self, character_id):
        self.target_id = character_id
        self.done = False

    def command(self, character):
        if self.done:
            return self
        self.done = True
        return GoTo(None, character.get_destination(self.target_id))

class MoveMouse(Command):
    def __init__(self):
        self.reset_sprints()
        self.hide_x = None
        self.hide_delay = None

    def reset_sprints(self):
        self.sprints = random.randint(2, 5)
        self.reset_sprint_distance()

    def reset_sprint_distance(self):
        self.sprint_distance = random.randint(2, 5)

    def command(self, mouse):
        mouse.run(True)
        if self.hide_x:
            if self.hide_delay > 0:
                self.hide_delay -= 1
                return
            mouse.x = self.hide_x
            self.hide_x = None
            self.reset_sprints()
        elif self.sprint_distance > 0:
            if mouse.is_blocked():
                mouse.turn()
            mouse.x += mouse.direction
            self.sprint_distance -= 1
        elif self.sprints > 0:
            self.sprints -= 1
            self.reset_sprint_distance()
            mouse.direction = random.choice((-1, 1))
            if mouse.is_blocked():
                mouse.turn()
            mouse.x += mouse.direction
        else:
            self.hide_x = mouse.x
            self.hide_delay = random.randint(5, 20)
            mouse.hide()

    def is_interruptible(self):
        return False

class MoveFrog(Command):
    def command(self, frog):
        if frog.x < 0:
            return
        if not frog.is_eric_nearby() and random.randint(0, 3):
            return
        frog.run(True)
        turn_round = ((HOP1, 0, 1), (SIT, 0, -1))
        short_hop = ((HOP1, 1, 1), (SIT, 0, 1))
        long_hop = ((HOP1, 1, 1), (HOP2, 1, 1), (HOP1, 1, 1), (SIT, 0, 1))
        choice = random.randint(0, 7)
        if choice < 2:
            phases = turn_round
        elif choice < 5:
            phases = turn_round if frog.is_blocked(1) else short_hop
        else:
            phases = turn_round if frog.is_blocked(3) else long_hop
        return Hop(phases)

    def is_interruptible(self):
        return False

class Hop(Command):
    def __init__(self, phases):
        self.phases = phases
        self.index = 0

    def command(self, frog):
        if self.index >= len(self.phases):
            return self
        state, x_inc, dir_change = self.phases[self.index]
        frog.animatory_state = state
        frog.x += x_inc * frog.direction
        frog.direction *= dir_change
        self.index += 1

    def is_interruptible(self):
        return False

class Catch(ComplexCommand):
    def get_commands(self):
        return (self.bend_over, self.catch_animal, self.stand_up, self.done)

    def bend_over(self, character):
        character.bend_over()

    def catch_animal(self, character):
        character.catch_animal()

    def stand_up(self, character):
        character.stand_up()

#
# Unimplemented commands
#
class WatchForEric(Command):
    def command(self, character):
        return
