# -*- coding: utf-8 -*-
# Copyright 2008, 2010 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import random
import pygame
from character import Character, Animal
from eric import Eric
from pellet import Pellet
from ai import CommandListTemplate

class Cast:
    def __init__(self, sprites):
        self.sprites = sprites
        self.sprite_groups = {}
        self.eric = None
        self.everything = []     # Everything that must be drawn
        self.characters = {}     # Humans
        self.character_list = [] # Ordered list of humans
        self.animals = []        # Animals
        self.movables = []       # All computer-controlled things
        self.taps = {}

    def add_sprite(self, group_id, sprite_id, sprite_index):
        if group_id not in self.sprite_groups:
            self.sprite_groups[group_id] = [{}, {}]
        sprite = self.get_animatory_state(sprite_index)
        self.sprite_groups[group_id][0][sprite_id] = sprite
        self.sprite_groups[group_id][1][sprite_id] = pygame.transform.flip(sprite, True, False)

    def add_eric(self, character_id, name, sprite_group_id, initial_as, flags):
        self.eric = Eric(character_id, name, flags)
        self.eric.set_animatory_states(*self.sprite_groups[sprite_group_id])
        self.eric.initialise_animatory_state(initial_as)
        self.everything.append(self.eric)
        self.characters[character_id] = self.eric
        self.character_list.append(self.eric)

    def add_character(self, character_id, name_and_title, sprite_group_id, initial_as, flags):
        names = name_and_title.partition('/')
        name, title = names[0], names[2]
        character = Character(character_id, name, flags)
        character.set_title(title)
        character.set_animatory_states(*self.sprite_groups[sprite_group_id])
        character.initialise_animatory_state(initial_as)
        self.everything.append(character)
        self.characters[character_id] = character
        self.character_list.append(character)
        self.movables.append(character)

    def add_pellet(self, character_id, sprite_group_id, tap_id, pellet_range, hit_zone):
        pellet_id = character_id + '-PELLET'
        pellet = Pellet(pellet_id, tap_id, pellet_range, hit_zone)
        pellet.set_animatory_states(*self.sprite_groups[sprite_group_id])
        pellet.x = -3
        self.everything.append(pellet)
        self.movables.append(pellet)
        self.get(character_id).set_pellet(pellet)

    def add_animal(self, animal_id, sprite_group_id, initial_as, location, tap_id):
        animal = Animal(animal_id, tap_id)
        animal.set_animatory_states(*self.sprite_groups[sprite_group_id])
        animal.animatory_state = initial_as
        animal.x, animal.y = location
        animal.direction = 1
        self.everything.append(animal)
        self.animals.append(animal)
        self.movables.append(animal)

    def get_animatory_state(self, state):
        width = self.sprites.get_width() // 16
        height = self.sprites.get_height() // 8
        surface = self.sprites.subsurface((width * (state % 16), height * (state // 16)), (width, height))
        surface.set_colorkey((0, 254, 0))
        return surface

    def initialise(self, skool, screen, beeper):
        for character in self.everything:
            character.set_components(self, skool, screen, beeper)

    def reinitialise(self):
        for c in self.everything:
            c.reinitialise()

    def initialise_secrets(self):
        letters = ''
        for character in self.character_list:
            character.initialise_special_answer()
            if character.has_secret():
                letters += character.initialise_secret()
        return letters

    def add_command(self, tap_id, command_class, *params):
        if tap_id not in self.taps:
            self.taps[tap_id] = CommandListTemplate(tap_id)
        self.taps[tap_id].add_command(command_class, *params)

    def set_random_locations(self, character_id, locations):
        self.characters[character_id].set_random_locations(locations)

    def add_command_list(self, character_id, lesson_id, tap_id):
        self.characters[character_id].add_command_list(lesson_id, tap_id)

    def set_location(self, character_id, x, y):
        self.get(character_id).set_initial_location(x, y)

    def set_sit_down_message(self, character_id, message):
        self.characters[character_id].set_sit_down_message(message)

    def get_eric(self):
        return self.eric

    def get(self, character_id):
        return self.characters.get(character_id, None)

    def add_blackboard_message(self, character_id, message):
        self.characters[character_id].add_blackboard_message(message)

    def add_lines_message(self, character_id, message_id, message):
        message_lines = message.split('^')
        if character_id == '*':
            for c in self.get_lines_givers():
                c.add_lines_message(message_id, message_lines)
        else:
            self.get(character_id).add_lines_message(message_id, message_lines)

    def add_lesson_message(self, character_id, message, condition):
        if character_id == '*':
            for c in self.character_list:
                if c.is_a_teacher():
                    c.add_lesson_message(message, condition)
        else:
            self.get(character_id).add_lesson_message(message, condition)

    def move(self, keyboard):
        for movable in self.movables:
            movable.move()
        return self.eric.move(keyboard)

    def get_images(self):
        return [thing.get_image() for thing in self.everything]

    def get_speech_bubbles(self):
        return [character.get_speech_bubble() for character in self.character_list]

    def set_lesson(self, lesson_id):
        for movable in self.movables:
            tap_id = movable.get_tap_id(lesson_id)
            movable.set_command_list_template(self.taps[tap_id])
            movable.remove_bubble()
        self.eric.unfreeze()

    def get_facing_characters(self, character, offset):
        """Returns a list of any characters facing the given character from a
        distance equal to offset."""
        target_x = character.x + offset * character.direction
        target_y = character.y
        target_direction = -1 * character.direction
        facing_characters = []
        for c in self.get_punchables():
            if (c.x, c.y, c.direction) == (target_x, target_y, target_direction) and c.is_deckable():
                facing_characters.append(c)
        return facing_characters

    def get_punchables(self):
        """Returns a list of the punchable characters."""
        return [c for c in self.character_list if c.is_punchable()]

    def get_pelletables(self):
        """Returns a list of the pelletable characters."""
        return [c for c in self.character_list if c.is_pelletable()]

    def get_victim_at(self, x, y):
        """Returns the most suitable character to hit (if any) with a catapult
        pellet at the given coordinates."""
        pelletables = self.get_pelletables()
        for adult in [c for c in pelletables if c.is_adult()]:
            if (adult.x, adult.y) == (x, y):
                return adult
        for child in [c for c in pelletables if not c.is_adult()]:
            if (child.x, child.y) == (x, y):
                return child

    def get_adults(self):
        return [c for c in self.character_list if c.is_adult()]

    def get_lines_givers(self):
        return [c for c in self.character_list if c.can_give_lines()]

    def get_nearby_characters(self, character, candidates, witness):
        """Returns a list of characters from the given list of candidates who
        are close enough to the given character to be visible to him
        (regardless of the direction he's facing)."""
        x0 = character.x - 10
        x1 = character.x + 10
        y0 = character.y - 3
        y1 = character.y + 3
        nearby_characters = []
        for c in candidates:
            if x0 <= c.x <= x1 and y0 <= c.y <= y1 and c.has_line_of_sight_to(character):
                if not witness or c.direction * (character.x - c.x) >= 0:
                    nearby_characters.append(c)
        return nearby_characters

    def get_witnesses(self, character, candidates):
        """Returns a list of characters from the given list of candidates who
        are close enough to the given character (and are facing the right way)
        to be able to see him."""
        return self.get_nearby_characters(character, candidates, True)

    def get_nearby_adults(self, character):
        """Returns a list of adults who are close enough to the given character
        (and are facing the right way) to be able to see him."""
        return self.get_witnesses(character, self.get_adults())

    def get_nearby_lines_givers(self, character):
        """Returns a list of lines-givers who are close enough to the given
        character (and are facing the right way) to be able to see him."""
        return self.get_witnesses(character, self.get_lines_givers())

    def get_potential_lines_recipients(self, character):
        lines_recipients = [c for c in self.character_list if c.can_receive_lines()]
        return self.get_nearby_characters(character, lines_recipients, False)

    def get_nearest_lines_recipient(self, character):
        """Return the potential lines recipient nearest to the given
        character."""
        candidates = self.get_potential_lines_recipients(character)
        if len(candidates) > 0:
            nearest = candidates[0]
            for c in candidates[1:]:
                if abs(c.x - character.x) < abs(nearest.x - character.x):
                    nearest = c
            return nearest

    def get_animal(self, character):
        x = character.x + character.direction
        for animal in self.animals:
            if animal.y == character.y and animal.x == x and animal.is_sitting():
                return animal

    def expand_names(self, message):
        """Replace occurrences of $BLAH with the name of the character whose
        unique ID is 'BLAH'."""
        index = 0
        marker = '$'
        while message.find(marker, index) >=0:
            start = message.index(marker, index)
            end = start + 1
            while end < len(message) and message[end].isalnum():
                end += 1
            character = self.get(message[start + 1:end])
            if character:
                message = message.replace(message[start:end], character.name, 1)
            index = end
        return message

    def is_home(self, x):
        """Returns whether every character is on the 'home' side of the given
        x-coordinate."""
        return all(c.is_home(x) for c in self.character_list)

    def is_standing_on_kid(self, character):
        """Returns whether the character is standing on a kid who's been knocked
        out."""
        for c in self.character_list:
            if c.is_knocked_out() and c.x - 1 <= character.x <= c.x + 1 and character.y == c.y - 1:
                return True

    def somebody_near_door(self, door):
        """Returns True if somebody is standing near the door, False
        otherwise."""
        for c in self.character_list:
            if door.x - 2 <= c.x <= door.x + 1 and door.top_y <= c.y <= door.bottom_y:
                return True

    def is_beside_eric(self, character):
        """Returns whether the character is beside Eric (and so need go no
        further to find him)."""
        eric_x, eric_y = self.get_location_of_eric()
        return abs(character.x - eric_x) <= 4 and character.y == eric_y

    def get_location_of_eric(self):
        """Returns the non-staircase location closest to Eric."""
        # TODO Return the top or bottom of the staircase if Eric is on one
        return (self.eric.x, 3 + 7 * (self.eric.y // 7))

    def expel_eric(self):
        for c in self.character_list:
            if c.can_expel_eric():
                c.find_and_expel_eric()
                return

    def freeze_eric(self):
        return self.eric.freeze()

    def unfreeze_eric(self):
        self.eric.unfreeze()

    def eric_understood(self):
        return self.eric.understood_message()

    def is_touching_eric(self, character):
        return (character.x, character.y) == (self.eric.x, self.eric.y)

    def is_eric_absent(self):
        return self.eric.is_absent()

    def trip_people_up_at(self, x, y):
        # Assume trippability == pelletability
        for trippable in self.get_pelletables():
            if (trippable.x, trippable.y) == (x, y) and trippable.is_deckable():
                trippable.deck()

    ############################################################################
    # Grass config
    ############################################################################
    def set_hitters(self, hitters):
        self.hitters = hitters

    def set_writers(self, writers):
        self.writers = writers

    def set_hit_tale(self, hit_tale):
        self.hit_tale = hit_tale

    def set_write_tale(self, write_tale):
        self.write_tale = write_tale

    def set_absent_tale(self, absent_tale):
        self.absent_tale = absent_tale

    def expand_title(self, message, character):
        return message.replace('$TITLE', character.get_title())

    def get_hit_tale(self, teacher):
        hitter_id = random.choice(self.hitters)
        message = self.expand_names(self.hit_tale.replace('$1', '$%s' % hitter_id))
        return hitter_id, self.expand_title(message, teacher)

    def get_write_tale(self, writer_id, teacher):
        if writer_id in self.writers:
            culprit = random.choice(self.writers)
            message = self.expand_names(self.write_tale.replace('$1', '$%s' % writer_id))
            return writer_id, self.expand_title(message, teacher)
        return None, None

    def get_absent_tale(self, teacher):
        return self.expand_title(self.expand_names(self.absent_tale), teacher)
