# -*- coding: utf-8 -*-
# Copyright 2008, 2010 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

from character import Character
from animatorystates import *
from ai import Write, Jump, Hit, FireCatapult, Freeze, Catch
import lines
import items
import keys

class Eric(Character):
    def __init__(self, character_id, name, flags):
        Character.__init__(self, character_id, name, flags)
        self._reinitialise()

    def reinitialise(self):
        Character.reinitialise(self)
        self._reinitialise()

    def _reinitialise(self):
        self.writing = False
        self.frozen = False
        self.understood = False
        self.controller = None
        self.lines_delay = 40
        self.inventory = set()
        self.mice = 0

    def move(self, keyboard):
        self.keyboard = keyboard
        self.get_lines()
        self.walk_delay -= 1
        if self.walk_delay > 0:
            return 0
        self.keyboard.pump()
        self.walk_delay = 2
        if self.controller:
            if self.controller.command(self) is self.controller:
                self.controller = None
            return 0
        if self.midstride():
            self.make_walking_sound(self.get_walk_state_index())
            return self.walk()
        if self.is_knocked_out() or self.is_sitting():
            if self.keyboard.was_pressed(keys.SIT_STAND):
                if self.is_sitting_on_chair():
                    self.chair().vacate()
                self.animatory_state = WALK0
                self.beeper.make_sitting_sound()
            return 0
        if self.keyboard.was_pressed(keys.SIT_STAND):
            self.sit()
            self.beeper.make_sitting_sound()
            return 0
        if self.keyboard.was_pressed(keys.FIRE_CATAPULT) and not self.pellet.is_airborne():
            self.controller = FireCatapult()
            return 0
        if self.keyboard.was_pressed(keys.HIT):
            self.controller = Hit()
            return 0
        if self.keyboard.was_pressed(keys.JUMP):
            self.controller = Jump()
            return 0
        if self.keyboard.was_pressed(keys.WRITE) and self.skool.beside_blackboard(self):
            self.raise_arm()
            self.controller = Write()
            self.keyboard.start_writing()
            self.writing = True
            return 0
        if self.keyboard.was_pressed(keys.CATCH) and self.can_bend_over():
            self.controller = Catch()
            return 0
        self.staircase = self.skool.staircase(self)
        if not (self.staircase or self.cast.is_standing_on_kid(self) or self.skool.on_floor(self)):
            self.y += 1
            return 0
        self.barrier = self.skool.barrier(self)
        old_ws, old_as, old_direction = self.get_walk_state_index(), self.animatory_state, self.direction
        if self.keyboard.pressed(keys.LEFT):
            self.left()
        elif self.keyboard.pressed(keys.RIGHT):
            self.right()
        elif self.keyboard.pressed(keys.UP):
            self.up()
        elif self.keyboard.pressed(keys.DOWN):
            self.down()
        if (self.animatory_state, self.direction) != (old_as, old_direction):
            self.make_walking_sound(old_ws)
        return 0

    def get_lines(self):
        self.lines_delay = max(0, self.lines_delay - 1)
        if self.lines_delay > 0:
            return
        gave_lines = False
        home_room = self.skool.get_home_room()
        if self.is_sitting_on_stairs():
            gave_lines = self.alert_lines_givers(lines.NO_SITTING_ON_STAIRS)
        elif self.skool.in_no_go_zone(self):
            gave_lines = self.alert_lines_givers(lines.GET_OUT)
        elif self.should_get_along():
            gave_lines = self.alert_lines_givers(lines.GET_ALONG, True)
        elif home_room and home_room.contains(self) and home_room.chairs and not self.is_sitting_on_chair():
            gave_lines = self.alert_lines_givers(lines.SIT_DOWN)
        elif self.is_sitting_on_floor() or self.is_knocked_out():
            gave_lines = self.alert_lines_givers(lines.GET_UP)
        if gave_lines:
            self.lines_delay = 40

    def is_absent(self):
        """Returns whether Eric is playing truant."""
        return self.skool.get_home_room() not in (None, self.skool.room(self))

    def should_get_along(self):
        """Returns whether Eric is somewhere other than he should be."""
        return self.skool.should_get_along(self)

    def make_walking_sound(self, index):
        self.beeper.make_walking_sound(index)

    def walk(self, on_stairs=False):
        Character.walk(self, on_stairs)
        if not self.midstride():
            return self.screen.get_scroll_increment(self.x)
        return 0

    def dethrone(self):
        # Allow time for Eric to be drawn on the floor before lines are given
        self.lines_delay = max(self.lines_delay, 2)
        self.sit_on_floor()
        self.beeper.make_knocked_out_sound()

    def deck(self):
        # Allow time for Eric to be drawn on the floor before lines are given
        self.lines_delay = max(self.lines_delay, 2)
        if self.is_sitting_on_chair():
            self.chair().vacate()
        self.knock_over()
        self.beeper.make_knocked_out_sound()

    def alert_lines_givers(self, message_id, truant=False):
        nearby_lines_givers = self.cast.get_nearby_lines_givers(self)
        if nearby_lines_givers:
            teacher = self.skool.get_teacher()
            if truant and teacher in nearby_lines_givers:
                teacher.give_lines(self.character_id, teacher.get_come_along_message_id())
            else:
                nearby_lines_givers[0].give_lines(self.character_id, message_id)
        return len(nearby_lines_givers) > 0

    def fire_catapult(self):
        Character.fire_catapult(self)
        self.alert_lines_givers(lines.NO_CATAPULTS)
        self.beeper.make_catapult_sound()

    def punch(self):
        Character.punch(self)
        self.alert_lines_givers(lines.NO_HITTING)

    def jump(self):
        self.previous_as = self.animatory_state
        self.y -= 1
        self.walk_delay = 4
        self.animatory_state = ARM_UP

    def can_bend_over(self):
        return BENDING_OVER in self.as_dict_L

    def bend_over(self):
        self.previous_as = self.animatory_state
        self.walk_delay = 1
        self.animatory_state = BENDING_OVER

    def catch_animal(self):
        animal = self.cast.get_animal(self)
        if animal:
            animal.hide()
            if animal.is_mouse():
                self.beeper.make_mouse_sound()
                self.mice += 1
                self.screen.print_mice(self.mice, self.skool.get_mouse_image())
            elif animal.is_frog():
                self.beeper.make_bingo_sound()
                self.inventory.add(items.FROG)
                self.screen.print_inventory(self.skool.get_inventory_images(self.inventory))
        self.walk_delay = 4

    def stand_up(self):
        self.animatory_state = self.previous_as
        self.walk_delay = 1

    def get_location_above_hand(self):
        return (self.x + self.direction + 1, self.y)

    def descend(self):
        """Makes Eric finish his jump."""
        self.alert_lines_givers(lines.NO_JUMPING)
        self.skool.check_safe(*self.get_location_above_hand())
        self.check_shields()
        if not self.cast.is_standing_on_kid(self):
            self.y += 1
        self.animatory_state = self.previous_as

    def check_shields(self):
        self.check_shields_at(*self.get_location_above_hand())

    def write(self):
        """Controls Eric when he's writing on a board. Returns True if Eric has
        finished writing."""
        if self.has_arm_raised():
            self.lower_arm()
            self.writing = self.keyboard.writing
            return not self.writing
        blackboard = self.skool.room(self).get_blackboard()
        key_down_events = self.keyboard.key_down_events
        if key_down_events:
            first_event = key_down_events[0]
            if first_event.key == keys.ENTER:
                self.raise_arm()
                self.keyboard.finish_writing()
                return
            char = first_event.unicode
            if self.skool.font.has_char(char):
                self.raise_arm()
                self.skool.write_on_board(self, blackboard, char)
        else:
            self.walk_delay = 1 # Maximise responsiveness (hack)

    def is_eric(self):
        return True

    def freeze(self):
        if not self.writing:
            self.controller = Freeze()
            self.frozen = True
        return self.frozen

    def unfreeze(self):
        if self.frozen:
            self.controller = None
            self.frozen = False
            self.understood = False

    def check_understanding(self):
        if self.keyboard.was_pressed(keys.UNDERSTOOD):
            self.understood = True

    def understood_message(self):
        return self.understood
