# -*- coding: utf-8 -*-
# Copyright 2008, 2010 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import os
import pygame
import random
from cast import Cast
from skool import Skool, Timetable
from graphics import Screen, Gallery
from sound import Beeper
from input import Keyboard
from skoolbuilder import SkoolBuilder
import keys
import debug

class Game:
    def __init__(self, pyskool_dir, scale, ini_file, quick_start, version, icon_fname):
        pygame.init()
        builder = SkoolBuilder(ini_file)
        config = builder.get_game_config()

        images_dir = os.path.join(pyskool_dir, config['ImagesDir'])
        image_set = config['ImageSet']
        scale = scale or config['Scale']
        gallery_config = builder.get_gallery_config()
        gallery = Gallery(images_dir, image_set, scale, gallery_config)

        title = 'Pyskool %s: %s' % (version, config['Name'])
        self.create_screen(config, scale, gallery, title, icon_fname)
        self.create_skool(pyskool_dir, config, gallery, builder)
        self.screen.initialise_column(config['InitialColumn'], self.skool)

        self.max_lines = config['MaxLines'] // 10
        self.quick_start = quick_start or config['QuickStart']
        self.speed = 1
        self.fps = config['GameFps']

    def create_screen(self, config, scale, gallery, title, icon_fname):
        self.screen = Screen(config['GraphicsMode'], scale, config['Background'], gallery, config['ScrollFps'], title, icon_fname)
        self.screen.set_lesson_box_properties(config['LessonBoxAttribute'], config['LessonBoxPos'])
        self.screen.add_logo(config['LogoPos'])
        self.screen.add_score_box(config['ScoreBoxPos'], config['ScoreBoxXOffset'], config['ScoreBoxAttribute'])
        self.screen.add_mouse_inventory(config.get('MouseInventoryPos'))
        self.screen.add_inventory(config.get('InventoryPos'))

    def create_skool(self, pyskool_dir, config, gallery, builder):
        beeper = Beeper(os.path.join(pyskool_dir, config['SoundsDir']))
        cast = Cast(gallery.get_sprites())
        timetable = Timetable(config['LessonLength'], config['LessonStartTime'])
        self.skool = Skool(self, self.screen, beeper, cast, timetable, gallery)
        builder.build_skool(self.skool)
        self.skool.initialise_cast()

    def check_cheat_keys(self, keyboard):
        if keyboard.was_pressed(keys.NEXT_LESSON):
            self.skool.next_lesson(False)
        if keyboard.is_pressed(keys.SLOW):
            self.speed = 0.5
        elif keyboard.is_pressed(keys.FAST):
            self.speed = 2
        else:
            self.speed = 1
        if self.skool.shields:
            if keyboard.was_pressed(keys.SHIELD_MODE2):
                for shield in self.skool.shields:
                    shield.flash()
                self.skool.safe.unflash()
                self.skool.shield_mode = 2
            if keyboard.was_pressed(keys.SHIELD_MODE3):
                for shield in self.skool.shields[:-1]:
                    shield.unflash()
                self.skool.shields[-1].flash()
                self.skool.safe.flash()
                self.skool.shield_mode = 3
            if keyboard.was_pressed(keys.FLASH_MOST):
                for shield in self.skool.shields[:-1]:
                    shield.flash()
            if keyboard.was_pressed(keys.UNFLASH_MOST):
                for shield in self.skool.shields[:-1]:
                    shield.unflash()
        if keyboard.was_pressed(keys.ADD_LINES):
            self.skool.add_lines(10 * random.randrange(1, 9))
        if self.skool.safe_combination and keyboard.was_pressed(keys.SAFE_COMBO):
            debug.log('Safe: %s' % self.skool.safe_combination)
        if keyboard.was_pressed(keys.SPECIAL_ANSWER):
            for c in self.skool.cast.character_list:
                if c.special_answer:
                    debug.log('%s: %s' % (c.name, c.special_answer))

    def play(self, cheat=False):
        clock = pygame.time.Clock()

        self.ring_bell = False

        self.screenshot = 0
        self.paused = False
        keyboard = Keyboard()
        while True:
            if not self.quick_start:
                self.ring_bell = True
                self.skool.scroll_on(clock)

            self.quick_start = False
            self.game_over = False

            while not self.game_over:
                if self.main_loop(keyboard, clock, cheat):
                    return

            self.skool.reinitialise()

    def main_loop(self, keyboard, clock, cheat):
        if keyboard.got_quit() or keyboard.was_pressed(keys.QUIT):
            return True

        self.paused ^= keyboard.was_pressed(keys.PAUSE)
        if self.paused:
            clock.tick(20)
            keyboard.pump()
            return False

        if self.skool.tick():
            self.skool.next_lesson(self.ring_bell)
            self.ring_bell = True

        if cheat:
            self.check_cheat_keys(keyboard)

        if keyboard.was_pressed(keys.SCREENSHOT):
            img_format = 'bmp' if pygame.version.vernum < (1, 8) else 'png'
            img_fname = 'pyskool%03i.%s' % (self.screenshot, img_format)
            self.screen.take_screenshot(img_fname)
            debug.log('Took screenshot: %s' % img_fname)
            self.screenshot += 1

        scroll = self.skool.move_characters(keyboard)
        self.skool.auto_shut_doors()

        clock.tick(self.fps * self.speed)

        self.skool.draw()
        self.screen.update()
        self.skool.scroll(scroll, clock)

    def get_max_lines(self):
        return self.max_lines

    def end(self):
        self.game_over = True
