# -*- coding: utf-8 -*-
# Copyright 2008, 2010 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

"""
Classes concerned with the screen and drawing things on it.
"""

import os
import pygame

#: ID of the speech bubble image.
SPEECH_BUBBLE = 'SPEECH_BUBBLE'
#: ID of the font image.
FONT = 'FONT'
#: ID of the sprites image.
SPRITES = 'SPRITES'
#: ID of the inventory items image.
INVENTORY = 'INVENTORY'
#: ID of the logo image.
LOGO = 'LOGO'
#: ID of the mutables image.
MUTABLES = 'MUTABLES'
#: ID of the ink-only mutables image.
MUTABLES_INK = 'MUTABLES_INK'
#: ID of the paper-only mutables image.
MUTABLES_PAPER = 'MUTABLES_PAPER'
#: ID of the score box image.
SCOREBOX = 'SCOREBOX'
#: ID of the play area image.
SKOOL = 'SKOOL'
#: ID of the ink-only play area image.
SKOOL_INK = 'SKOOL_INK'
#: ID of the paper-only play area image.
SKOOL_PAPER = 'SKOOL_PAPER'

class Screen:
    """Represents the screen upon which everything is drawn.

    :type config: dict
    :param config: Configuration parameters from the ini file.
    :param scale: The scale at which the screen will be drawn.
    :type gallery: :class:`Gallery`
    :param gallery: The gallery of images to use for drawing.
    :param title: The window title.
    :param icon_fname: The filename of the Pyskool icon.
    """
    def __init__(self, config, scale, gallery, title, icon_fname):
        self.width = config.get('Width', 32)
        self.scale = scale
        self.screen = pygame.display.set_mode(self.scale_coords((self.width, config.get('Height', 24))))
        self.mode = config.get('GraphicsMode', 1)
        self.background = config.get('Background', 2)
        self.scroll_fps = config.get('ScrollFps', 20)
        self.lesson_box_ink = config.get('LessonBoxInk', (0, 198, 197))
        self.lesson_box_paper = config.get('LessonBoxPaper', (0, 0, 0))
        self.lesson_box_pos = self.scale_coords(config.get('LessonBoxPos', (12, 21)))
        self.lesson_box_size = self.scale_coords(config.get('LessonBoxSize', (8, 3)))
        self.score_box_coords = config.get('ScoreBoxPos', (24, 21))
        self.score_box_x_offset = config.get('ScoreBoxXOffset', 56)
        self.score_box_ink = config.get('ScoreBoxInk', (197, 198, 0))
        self.score_box_paper = config.get('ScoreBoxPaper', (0, 0, 0))
        self.score_offset = config.get('ScoreOffset', 1)
        self.lines_offset = config.get('LinesOffset', 9)
        self.hi_score_offset = config.get('HiScoreOffset', 17)
        mouse_box_pos = config.get('MouseInventoryPos')
        self.mouse_box_coords = self.scale_coords(mouse_box_pos) if mouse_box_pos else None
        inventory_pos = config.get('InventoryPos')
        self.inventory_coords = self.scale_coords(inventory_pos) if inventory_pos else None
        self.inventory_size = self.scale_coords(config.get('InventorySize', (0, 0)))
        self.scroll_right_offset = config.get('ScrollRightOffset', 9)
        self.scroll_left_offset = config.get('ScrollLeftOffset', 10)
        self.scroll_columns = config.get('ScrollColumns', 8)
        self.bubble = gallery.get_image(SPEECH_BUBBLE)
        self.speech_bubble_size = config.get('SpeechBubbleSize', (8, 3))
        self.speech_bubble_ink = config.get('SpeechBubbleInk', (0, 0, 0))
        self.speech_bubble_paper = config.get('SpeechBubblePaper', (205, 198, 205))
        # Create a colorkey for the speech bubble that is different from the
        # ink and paper colours
        red = (self.speech_bubble_ink[0] + 1) % 256
        if red == self.speech_bubble_paper[0]:
            red = (red + 1) % 256
        self.speech_bubble_colorkey = (red, 0, 0)
        self.open_lip_byte = config.get('OpenLipByte', 66)
        self.speech_bubble_inset = config.get('SpeechBubbleInset', (4, 4))
        self.lines_bubble_size = self.scale_coords(config.get('LinesBubbleSize', (8, 3)))
        self.font = Font(gallery.get_image(FONT), config.get('FontInk', (0, 1, 2)), config.get('FontPaper', (255, 254, 253)))
        self.skool_colorkey = config.get('SkoolInkKey', (255, 255, 255))
        self.flash_cycle = config.get('FlashCycle', 10)

        # Set the window title and icon, and draw the logo and score box
        if os.path.isfile(icon_fname):
            pygame.display.set_icon(pygame.image.load(icon_fname).convert())
        pygame.display.set_caption(title)
        self.screen.fill(self.background)
        self.screen.blit(gallery.get_image(LOGO), self.scale_coords(config.get('LogoPos', (0, 21))))
        self.screen.blit(gallery.get_image(SCOREBOX), self.scale_coords(self.score_box_coords))

    def scale_coords(self, coords):
        """Scale up a pair of coordinates and return them.

        :param coords: The coordinates.
        """
        return (8 * self.scale * coords[0], 8 * self.scale * coords[1])

    def initialise_column(self, initial_column, skool_width, eric_x):
        """Set the leftmost column of the play area that will appear on the
        screen when the game starts.

        :param initial_column: The initial leftmost column.
        :param skool_width: The width of the entire play area (in pixels).
        :param eric_x: Eric's initial x-coordinate.
        """
        self.max_column = skool_width // (8 * self.scale) - self.width
        default_initial_column = min(self.max_column, max((eric_x - self.width // 2) // 8 * 8, 0))
        column = min(self.max_column, default_initial_column if initial_column is None else initial_column)
        self.column = self.initial_column = max(0, 8 * (column // 8))

    def reinitialise(self):
        """Reinitialise the screen for a new game."""
        self.column = self.initial_column
        self.print_lesson('', '')
        self.print_inventory()
        self.print_mice()

    def add_font_character(self, char, offset, width):
        """Define the location and width of a font character bitmap in the font
        image.

        :param char: The font character.
        :param offset: The offset (in pixels) of the font character bitmap from
                       the left edge of the font image.
        :param width: The width of the font character bitmap.
        """
        self.font.add_character(char, offset, width)

    def get_scroll_increment(self, x):
        """Return the direction in which the screen should be scrolled when
        Eric is at a given x-coordinate.

        :param x: Eric's x-coordinate.
        :return: -1 if the screen should scroll right, 1 if it should scroll
                 left, or 0 if it should not scroll.
        """
        offset = x - self.column
        if self.width - self.scroll_left_offset <= offset < self.width:
            return 1
        if -3 < offset <= self.scroll_right_offset:
            return -1
        return 0

    def scroll_skool(self, skool, clock):
        """Scroll the skool across the entire width of the screen from right to
        left.

        :type skool: :class:`~skool.Skool`
        :param skool: The skool.
        :type clock: `pygame.time.Clock`
        :param clock: The clock to use to time the scrolling.
        """
        self.column -= self.width
        background = pygame.Surface((self.screen.get_width(), skool.get_height()))
        for n in range(self.width):
            self.column += 1
            skool.draw(False)
            self.screen.blit(background, (-8 * (n + 1) * self.scale, 0))
            self._update()
            clock.tick(self.scroll_fps)

    def scroll(self, inc, skool, clock):
        """Scroll the skool a number of columns across the screen.

        :param inc: The scroll increment (-1 to scroll one column at a time
                    rightwards, 1 to scroll one column at a time leftwards).
        :type skool: :class:`~skool.Skool`
        :param skool: The skool.
        :type clock: `pygame.time.Clock`
        :param clock: The clock to use to time the scrolling.
        """
        scroll_inc = 0
        num_cols = self.scroll_columns
        if inc > 0:
            scroll_inc = 1
            num_cols = min(num_cols, self.max_column - self.column)
        elif inc < 0:
            scroll_inc = -1
            num_cols = min(num_cols, self.column)
        for i in range(num_cols):
            self.column += scroll_inc
            skool.draw()
            self._update()
            clock.tick(self.scroll_fps)

    def get_text(self, words, ink, paper):
        """Return a `pygame.Surface` displaying some text in the skool font.

        :param words: The text.
        :param ink: The ink colour to use.
        :param paper: The paper colour to use.
        """
        return self.font.render(words, ink, paper)

    def print_lesson(self, *text_lines):
        """Print some text in the lesson box.

        :param text_lines: The lines of text to print.
        """
        while len(text_lines) < 2:
            text_lines.append('')
        line1, line2 = text_lines[:2]
        lesson_box = pygame.Surface(self.lesson_box_size)
        lesson_box.fill(self.lesson_box_paper)
        line1_text = self.get_text(line1, self.lesson_box_ink, self.lesson_box_paper)
        line2_text = self.get_text(line2, self.lesson_box_ink, self.lesson_box_paper)
        font_height = self.scale * 8
        line1_x = (lesson_box.get_width() - line1_text.get_width()) // 2
        line1_y = (lesson_box.get_height() - 2 * font_height) // 2
        lesson_box.blit(line1_text, (line1_x, line1_y))
        line2_x = (lesson_box.get_width() - line2_text.get_width()) // 2
        line2_y = line1_y + font_height
        lesson_box.blit(line2_text, (line2_x, line2_y))
        self.screen.blit(lesson_box, self.lesson_box_pos)
        self._update(pygame.Rect(self.lesson_box_pos, lesson_box.get_size()))

    def _print_number(self, number, y_offset):
        """Print a number (the score, lines total, or hi-score) in the score
        box.

        :param number: The number.
        :param y_offset: The vertical offset from the top of the score box at
                         which to print the number.
        """
        if number < 0:
            return
        number_text = self.get_text(str(number) if number > 0 else "", self.score_box_ink, self.score_box_paper)
        width = 24 * self.scale
        background = pygame.Surface((width, 7 * self.scale))
        background.fill(self.score_box_paper)
        background.blit(number_text, (width - number_text.get_width(), 0))
        coords = self.scale_coords(self.score_box_coords)
        top_left_x = coords[0] + self.score_box_x_offset * self.scale - width
        top_left_y = coords[1] + y_offset * self.scale
        self.screen.blit(background, (top_left_x, top_left_y))
        self._update(pygame.Rect((top_left_x, top_left_y), background.get_size()))

    def print_score(self, score):
        """Print the score in the score box.

        :param score: The score.
        """
        self._print_number(score, self.score_offset)

    def print_lines(self, lines):
        """Print the lines total in the score box.

        :param lines: The lines total.
        """
        self._print_number(lines, self.lines_offset)

    def print_hi_score(self, hiscore):
        """Print the hi-score in the score box.

        :param hiscore: The hi-score.
        """
        self._print_number(hiscore, self.hi_score_offset)

    def print_inventory(self, item_images=()):
        """Print the inventory. If no inventory is defined, nothing happens.

        :param item_images: A sequence of item images to draw in the inventory
                            box.
        """
        if not self.inventory_coords:
            return
        inventory_box = pygame.Surface(self.inventory_size)
        inventory_box.fill(self.background)
        x = 0
        for image in item_images:
            if image:
                inventory_box.blit(image, (x, 0))
                x += image.get_width()
        self.screen.blit(inventory_box, self.inventory_coords)
        self._update(pygame.Rect(self.inventory_coords, inventory_box.get_size()))

    def print_mice(self, count=0, mouse_image=None):
        """Print the mouse inventory. If no mouse inventory is defined, nothing
        happens.

        :param count: The number of mice to draw.
        :param mouse_image: An image of a captured mouse.
        """
        if not self.mouse_box_coords:
            return
        max_mice = 8
        mouse_box = pygame.Surface(self.scale_coords((max_mice, 1)))
        mouse_box.fill(self.background)
        for x in range(min((count, max_mice))):
            mouse_box.blit(mouse_image, (x * mouse_image.get_width(), 0))
        self.screen.blit(mouse_box, self.mouse_box_coords)
        self._update(pygame.Rect(self.mouse_box_coords, mouse_box.get_size()))

    def contains(self, character, full=True):
        """Return whether a character is on-screen.

        :type character: :class:`~character.Character`
        :param character: The character to check.
        :param full: If `True`, return whether the character's entire sprite
                     is on-screen; if `False`, return whether any part of the
                     character's sprite is on-screen.
        """
        return self.column <= character.x <= self.column + self.width - (3 if full else 1)

    def print_lines_bubble(self, x, y, message, ink, paper):
        """Print a lines message bubble.

        :param x: The x-coordinate of the lines-giver's head.
        :param y: The y-coordinate of the top row of the bubble.
        :type message: tuple
        :param message: The lines of text to write in the bubble.
        :type ink: RGB triplet
        :param ink: The ink colour of the bubble.
        :type paper: RGB triplet
        :param paper: The paper colour of the bubble.
        """
        bubble_x = 8 * (x // 8) - self.column
        bubble_coords = self.scale_coords((bubble_x, y))

        lines_bubble = pygame.Surface(self.lines_bubble_size)
        lines_bubble.fill(paper)
        bubble_width = lines_bubble.get_width()
        y = self.scale * 4
        for line in message:
            text_image = self.get_text(line, ink, paper)
            x = (bubble_width - text_image.get_width()) // 2
            lines_bubble.blit(text_image, (x, y))
            y += self.scale * 8
        self.screen.blit(lines_bubble, bubble_coords)
        self._update(pygame.Rect(bubble_coords, lines_bubble.get_size()))

    def get_bubble(self, words, lip_pos, shift):
        """Create a speech bubble displaying a portion of a message.

        :param words: The text of the message.
        :param lip_pos: The offset from the left edge of the speech bubble at
                        which to place the lip.
        :param shift: The offset (in tiles) by which to shift the text image
                      before displaying it in the bubble; if negative, leading
                      spaces will be displayed.
        :return: A 2-tuple, `(bubble, done}`, where `bubble` is the speech
                 bubble image (a `pygame.Surface`), and `done` is `True` if
                 the entire message has been spoken, `False` otherwise.
        """
        bubble = pygame.Surface(self.scale_coords(self.speech_bubble_size))
        bubble.fill(self.speech_bubble_colorkey)
        bubble.set_colorkey(self.speech_bubble_colorkey)
        bubble.blit(self.bubble, (0, 0))
        lip = self.bubble.subsurface(self.scale_coords((8, 0)), self.scale_coords((1, 1)))
        lip_coords = self.scale_coords((lip_pos, self.speech_bubble_size[1] - 1))
        bubble.blit(lip, lip_coords)

        # Open the lip of the speech bubble
        open_lip_byte = self.open_lip_byte
        for bit in range(8):
            bit_x = bit * self.scale
            colour = self.speech_bubble_ink if open_lip_byte & 128 else self.speech_bubble_paper
            for r in range(self.scale):
                for c in range(self.scale):
                    bubble.set_at((lip_coords[0] + bit_x + c, lip_coords[1] - 1 - r), colour)
            open_lip_byte *= 2

        text = self.get_text(words, self.speech_bubble_ink, self.speech_bubble_paper)
        tile_width = 8 * self.scale
        min_inset_x = self.speech_bubble_inset[0] * self.scale
        inset_x = min_inset_x - tile_width * min(shift, 0)
        inset_y = self.speech_bubble_inset[1] * self.scale
        text_x = max(shift, 0) * tile_width
        max_width = tile_width * self.speech_bubble_size[0] - 2 * min_inset_x
        width = min(min_inset_x + max_width - inset_x, text.get_width() - text_x)
        text_window = text.subsurface((text_x, 0), (width, tile_width))
        bubble.blit(text_window, (inset_x, inset_y))
        return (bubble, width < 0)

    def _update(self, *args):
        """Update the display.

        :param args: Arguments passed to `pygame.display.update`.
        """
        pygame.display.update(*args)

    def draw(self, skool_images, blackboards, cast, others, update):
        """Draw everything on the screen.

        :param skool_images: The play area images.
        :param blackboards: The blackboards (3-tuples, `(x, y, image)`, where
                            `x` and `y` are the coordinates, and `image` is an
                            image of the blackboard).
        :param cast: The cast (3-tuples, `(x, y, image)`, where `x` and `y`
                     are the coordinates, and `image` is an image of a cast
                     member).
        :param others: Anything else to be drawn (3-tuples, `(x, y, image)`).
        :param update: Whether to update the screen after drawing.
        """
        if self.mode == 0:
            self._draw_skool(self.screen, skool_images[0])
            for x, y, image in blackboards:
                self._draw_image(self.screen, x, y, image)
            for x, y, image in cast:
                self._draw_image(self.screen, x, y, image)
        else:
            height = skool_images[1].get_height()
            scratch = pygame.Surface((self.width * 8 * self.scale, height))
            self._draw_skool(scratch, skool_images[1])
            for x, y, image in blackboards:
                self._draw_image(scratch, x, y, image)
            for x, y, image in cast:
                self._draw_image(scratch, x, y, image)
            self._draw_skool(self.screen, skool_images[2])
            scratch.set_colorkey(self.skool_colorkey)
            self.screen.blit(scratch, (0, 0))
        for x, y, image in others:
            self._draw_image(self.screen, x, y, image)
        if update:
            self._update()

    def _draw_skool(self, surface, skool):
        """Draw the skool.

        :type surface: `pygame.Surface`
        :param surface: The surface on which to draw the skool.
        :type skool: `pygame.Surface`
        :param skool: An image of the play area.
        """
        surface.blit(skool, self.scale_coords((-self.column, 0)))

    def _draw_image(self, surface, x, y, image):
        """Draw an image on a surface.

        :type surface: `pygame.Surface`
        :param surface: The surface on which to draw the image.
        :param x: The x-coordinate of the image.
        :param y: The y-coordinate of the image.
        :type image: `pygame.Surface`
        :param image: The image.
        """
        if image:
            surface.blit(image, self.scale_coords((x - self.column, y)))

    def take_screenshot(self, filename):
        """Take a screenshot and save it to a file.

        :param filename: The name of the file.
        """
        pygame.image.save(self.screen, filename)

    def has_font_char(self, char):
        """Return whether the skool font contains a bitmap for a given
        character.

        :param char: The character to look for.
        """
        return self.font.has_char(char)

class Gallery:
    """A container for all the images used in a game.

    :param images_dir: The base directory for the images.
    :param image_set: The name of the image set.
    :param scale: The desired scale of the images.
    :type images: dict
    :param images: Key-value pairs (image ID, path) from the `Images` section.
    """
    def __init__(self, images_dir, image_set, scale, images):
        self.images_dir = images_dir
        self.image_set = image_set
        self.scale = scale
        self.images = images

    def get_image(self, image_id):
        """Return an image (a `pygame.Surface`) from the gallery, or `None`
        if there is no image in the gallery with the given ID. The image will
        be scaled up as necessary.

        :param image_id: The ID of the image.
        """
        if image_id not in self.images:
            return None
        scale_up = True
        image_set_dir = os.path.join(self.images_dir, '%sx%i' % (self.image_set, self.scale))
        if os.path.isdir(image_set_dir):
            scale_up = False
        else:
            image_set_dir = os.path.join(self.images_dir, '%sx1' % self.image_set)
        fname = os.path.join(*self.images[image_id].split('/'))
        image_file = os.path.join(image_set_dir, fname)
        if not os.path.exists(image_file):
            return None
        img = pygame.image.load(image_file).convert()
        if scale_up:
            img = pygame.transform.scale(img, (self.scale * img.get_width(), self.scale * img.get_height()))
        return img

class Font:
    """The skool font.

    :type image: `pygame.Surface`
    :param image: The font image.
    :type ink_key: RGB triplet
    :param ink_key: The ink colour in `font.png` (used to create transparency).
    :type paper_key: RGB triplet
    :param paper_key: The paper colour in `font.png` (used to create
                      transparency).
    """
    def __init__(self, image, ink_key, paper_key):
        self.image = image
        self.ink_key = ink_key
        self.paper_key = paper_key
        self.characters = {}

    def add_character(self, char, offset, width):
        """Define the location and width of a font character bitmap in the font
        image.

        :param char: The font character.
        :param offset: The offset (in pixels) of the font character bitmap from
                       the left edge of the font image.
        :param width: The width of the font character bitmap.
        """
        self.characters[char] = (offset, width)

    def render(self, words, ink, paper):
        """Return an image (a `pygame.Surface`) of a text message written in
        the skool font.

        :param words: The message.
        :param ink: RGB triplet
        :param ink: The desired ink colour.
        :param paper: RGB triplet
        :param paper: The desired paper colour.
        """
        character_images = []
        total_width = 0
        height = self.image.get_height()
        scale = height // 8
        for c in words:
            offset, width = self.characters[c]
            image = self.image.subsurface((scale * offset, 0), (scale * width, height))
            character_images.append(image)
            total_width += width
        text = pygame.Surface((scale * total_width, height))
        offset = 0
        for image in character_images:
            text.blit(image, (offset, 0))
            offset += image.get_width()
        paper_surface = pygame.Surface((text.get_width(), text.get_height()))
        paper_surface.fill(paper)
        ink_surface = pygame.Surface((text.get_width(), text.get_height()))
        ink_surface.fill(ink)
        text.set_colorkey(self.ink_key)
        ink_surface.blit(text, (0, 0))
        ink_surface.set_colorkey(self.paper_key)
        paper_surface.blit(ink_surface, (0, 0))
        return paper_surface

    def has_char(self, char):
        """Return whether the skool font contains a bitmap for a given
        character.

        :param char: The character to look for.
        """
        return char in self.characters
