#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright 2010 Richard Dymond (rjdymond@gmail.com)
#
# This file is part of Pyskool.
#
# Pyskool is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# Pyskool is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# Pyskool. If not, see <http://www.gnu.org/licenses/>.

import sys
import os
sys.path.insert(0, os.path.join(os.path.dirname(__file__), 'lib'))
import skoolgraphics

SKOOL_DAZE = 'sd'
BACK_TO_SKOOL = 'bts'

def write_images(udgs, prefix, flip=False, switch=False):
    # ink and paper
    write_image(udgs, os.path.join(odir, '%s.png' % prefix), flip=flip, switch=switch)

    # ink
    for row in udgs:
        for udg in row:
            udg.paper = 15
    write_image(udgs, os.path.join(odir, '%s_ink.png' % prefix), flip=flip, switch=switch)

    # paper
    blank = [0] * 8
    for row in udgs:
        for udg in row:
            udg.paper = None
            udg.data = blank
    write_image(udgs, os.path.join(odir, '%s_paper.png' % prefix), flip=flip, switch=switch)

def write_image(udgs, fname, mask=False, width=None, flip=False, switch=False, post_process=None):
    sys.stdout.write('Writing %s\n' % fname)
    writer.write_image(udgs, fname, scale, mask, width, flip, switch, post_process)

def get_animatory_states(skool):
    montage = []
    for y in range(8):
        montage.extend([[], [], [], []])
        for x in range(16):
            state = x + 16 * y
            sprite = skool.build_sprite(state, 120)
            for i, row in enumerate(sprite):
                montage[4 * y + i].extend(row)
    return montage

def parse_options(options):
    scale = 1
    custom = 0
    i = 0
    while i < len(options):
        option = options[i]
        if option == '-s':
            scale = int(options[i + 1])
            i += 1
        elif option == '-c':
            custom = int(options[i + 1])
            i += 1
        i += 1
    return scale, custom

def print_usage():
    sys.stderr.write("Usage:\n")
    sys.stderr.write("  %s [options] %s input.[sna|z80] output-dir\n" % (sys.argv[0], SKOOL_DAZE))
    sys.stderr.write("  %s [options] %s input.[sna|z80] output-dir\n" % (sys.argv[0], BACK_TO_SKOOL))
    sys.stderr.write("\nOptions:\n")
    sys.stderr.write("  -s N  Scale images by a factor of N (default=1)\n")
    sys.stderr.write("  -c N  Adjust images appropriately for customisation N (default=0):\n")
    sys.stderr.write("          Skool Daze:\n")
    sys.stderr.write("            N=0: default\n")
    sys.stderr.write("            N=1: Skool Daze Take Too\n")
    sys.stderr.write("            N=2: Ezad Looks\n")
    sys.stderr.write("          Back to Skool:\n")
    sys.stderr.write("            N=0: default\n")
    sys.stderr.write("            N=1: Back to Skool Daze\n")
    sys.exit(1)

###############################################################################
# Begin
###############################################################################
if len(sys.argv) < 4:
    print_usage()

scale, custom = parse_options(sys.argv[1:-3])
game, snafile, odir = sys.argv[-3:]

flip = False
switch = False
if game == SKOOL_DAZE:
    skool = skoolgraphics.SDMemory(snafile, custom)
    flip = custom == 2
    switch = flip
elif game == BACK_TO_SKOOL:
    skool = skoolgraphics.BTSMemory(snafile, custom)
else:
    print_usage()
writer = skoolgraphics.ImageWriter()

if not os.path.isdir(odir):
    os.makedirs(odir)

# Sprites
write_image(get_animatory_states(skool), os.path.join(odir, 'sprites.png'), mask=True, post_process=skool.fix_sprites)

# Logo
write_image(skool.get_logo_udgs(), os.path.join(odir, 'logo.png'), flip=flip)

# Message box
write_image(skool.get_message_box_udgs(), os.path.join(odir, 'message_box.png'))

# Lesson box
write_image(skool.get_lesson_box_udgs(), os.path.join(odir, 'lesson_box.png'))

# Score box
write_image(skool.get_score_box_udgs(), os.path.join(odir, 'scorebox.png'))

# Speech bubble
write_image(skool.get_bubble_udgs(), os.path.join(odir, 'bubble.png'), post_process=skool.fix_bubble)

# Skool
write_images(skool.get_play_area_udgs(), 'skool', flip=flip)

# Mutables
write_images(skool.get_mutable_udgs(), 'mutables', flip=flip, switch=switch)

# BTS inventory
if game == BACK_TO_SKOOL:
    write_image(skool.get_inventory_udgs(), os.path.join(odir, 'inventory.png'))

# Font
# Do this last because we need to change the values of WHITE and BLACK
skoolgraphics.WHITE = (255, 254, 253)
skoolgraphics.BLACK = (0, 1, 2)
udgs = skool.get_text(''.join([chr(c) for c in range(32, 128)]))
font_udg_array = [[skoolgraphics.Udg(56, udg) for udg in udgs]]
write_image(font_udg_array, os.path.join(odir, 'font.png'), width=459)
